/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.connection;

import com.mongodb.MongoSocketException;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ChangeEvent;
import com.mongodb.connection.ChangeListener;
import com.mongodb.connection.CommandHelper;
import com.mongodb.connection.ConnectionPool;
import com.mongodb.connection.DescriptionHelper;
import com.mongodb.connection.ExponentiallyWeightedMovingAverage;
import com.mongodb.connection.InternalConnection;
import com.mongodb.connection.InternalConnectionFactory;
import com.mongodb.connection.ServerConnectionState;
import com.mongodb.connection.ServerDescription;
import com.mongodb.connection.ServerId;
import com.mongodb.connection.ServerMonitor;
import com.mongodb.connection.ServerSettings;
import com.mongodb.connection.ServerType;
import com.mongodb.diagnostics.logging.Logger;
import com.mongodb.diagnostics.logging.Loggers;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.bson.BsonDocument;
import org.bson.BsonInt32;
import org.bson.BsonValue;

@ThreadSafe
class DefaultServerMonitor
implements ServerMonitor {
    private static final Logger LOGGER = Loggers.getLogger("cluster");
    private final ServerId serverId;
    private final ChangeListener<ServerDescription> serverStateListener;
    private final InternalConnectionFactory internalConnectionFactory;
    private final ConnectionPool connectionPool;
    private final ServerSettings settings;
    private volatile ServerMonitorRunnable monitor;
    private volatile Thread monitorThread;
    private final Lock lock = new ReentrantLock();
    private final Condition condition = this.lock.newCondition();
    private volatile boolean isClosed;

    DefaultServerMonitor(ServerId serverId, ServerSettings settings, ChangeListener<ServerDescription> serverStateListener, InternalConnectionFactory internalConnectionFactory, ConnectionPool connectionPool) {
        this.settings = settings;
        this.serverId = serverId;
        this.serverStateListener = serverStateListener;
        this.internalConnectionFactory = internalConnectionFactory;
        this.connectionPool = connectionPool;
        this.monitorThread = this.createMonitorThread();
        this.isClosed = false;
    }

    @Override
    public void start() {
        this.monitorThread.start();
    }

    @Override
    public void connect() {
        this.lock.lock();
        try {
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void invalidate() {
        Assertions.isTrue("open", !this.isClosed);
        this.monitor.close();
        this.monitorThread.interrupt();
        this.monitorThread = this.createMonitorThread();
        this.monitorThread.start();
    }

    @Override
    public void close() {
        this.monitor.close();
        this.monitorThread.interrupt();
        this.isClosed = true;
    }

    Thread createMonitorThread() {
        this.monitor = new ServerMonitorRunnable();
        Thread monitorThread = new Thread((Runnable)this.monitor, "cluster-" + this.serverId.getClusterId() + "-" + this.serverId.getAddress());
        monitorThread.setDaemon(true);
        return monitorThread;
    }

    static boolean descriptionHasChanged(ServerDescription previousServerDescription, ServerDescription currentServerDescription) {
        return !previousServerDescription.equals(currentServerDescription);
    }

    static boolean stateHasChanged(ServerDescription previousServerDescription, ServerDescription currentServerDescription) {
        return DefaultServerMonitor.descriptionHasChanged(previousServerDescription, currentServerDescription) || previousServerDescription.getRoundTripTimeNanos() != currentServerDescription.getRoundTripTimeNanos();
    }

    static boolean exceptionHasChanged(Throwable previousException, Throwable currentException) {
        if (currentException == null) {
            return previousException != null;
        }
        if (previousException == null) {
            return true;
        }
        if (!currentException.getClass().equals(previousException.getClass())) {
            return true;
        }
        if (currentException.getMessage() == null) {
            return previousException.getMessage() != null;
        }
        return !currentException.getMessage().equals(previousException.getMessage());
    }

    class ServerMonitorRunnable
    implements Runnable {
        private volatile boolean monitorIsClosed;
        private final ExponentiallyWeightedMovingAverage averageRoundTripTime = new ExponentiallyWeightedMovingAverage(0.2);

        ServerMonitorRunnable() {
        }

        public void close() {
            this.monitorIsClosed = true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public synchronized void run() {
            InternalConnection connection = null;
            try {
                ServerDescription currentServerDescription = this.getConnectingServerDescription(null);
                Throwable currentException = null;
                while (!this.monitorIsClosed) {
                    ServerDescription previousServerDescription = currentServerDescription;
                    Throwable previousException = currentException;
                    currentException = null;
                    try {
                        if (connection == null) {
                            connection = DefaultServerMonitor.this.internalConnectionFactory.create(DefaultServerMonitor.this.serverId);
                            try {
                                connection.open();
                            }
                            catch (Throwable t) {
                                connection = null;
                                throw t;
                            }
                        }
                        try {
                            currentServerDescription = this.lookupServerDescription(connection);
                        }
                        catch (MongoSocketException e) {
                            DefaultServerMonitor.this.connectionPool.invalidate();
                            connection.close();
                            connection = null;
                            connection = DefaultServerMonitor.this.internalConnectionFactory.create(DefaultServerMonitor.this.serverId);
                            try {
                                connection.open();
                            }
                            catch (Throwable t) {
                                connection = null;
                                throw t;
                            }
                            try {
                                currentServerDescription = this.lookupServerDescription(connection);
                            }
                            catch (MongoSocketException e1) {
                                connection.close();
                                connection = null;
                                throw e1;
                            }
                        }
                    }
                    catch (Throwable t) {
                        this.averageRoundTripTime.reset();
                        currentException = t;
                        currentServerDescription = this.getConnectingServerDescription(t);
                    }
                    if (this.monitorIsClosed) continue;
                    try {
                        this.logStateChange(previousServerDescription, previousException, currentServerDescription, currentException);
                        this.sendStateChangedEvent(previousServerDescription, currentServerDescription);
                    }
                    catch (Throwable t) {
                        LOGGER.warn("Exception in monitor thread during notification of server description state change", t);
                    }
                    this.waitForNext();
                }
            }
            finally {
                if (connection != null) {
                    connection.close();
                }
            }
        }

        private ServerDescription getConnectingServerDescription(Throwable exception) {
            return ServerDescription.builder().type(ServerType.UNKNOWN).state(ServerConnectionState.CONNECTING).address(DefaultServerMonitor.this.serverId.getAddress()).exception(exception).build();
        }

        private ServerDescription lookupServerDescription(InternalConnection connection) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(String.format("Checking status of %s", DefaultServerMonitor.this.serverId.getAddress()));
            }
            long start = System.nanoTime();
            BsonDocument isMasterResult = CommandHelper.executeCommand("admin", new BsonDocument("ismaster", (BsonValue)new BsonInt32(1)), connection);
            this.averageRoundTripTime.addSample(System.nanoTime() - start);
            return DescriptionHelper.createServerDescription(DefaultServerMonitor.this.serverId.getAddress(), isMasterResult, connection.getDescription().getServerVersion(), this.averageRoundTripTime.getAverage());
        }

        private void sendStateChangedEvent(ServerDescription previousServerDescription, ServerDescription currentServerDescription) {
            if (DefaultServerMonitor.stateHasChanged(previousServerDescription, currentServerDescription)) {
                DefaultServerMonitor.this.serverStateListener.stateChanged(new ChangeEvent<ServerDescription>(previousServerDescription, currentServerDescription));
            }
        }

        private void logStateChange(ServerDescription previousServerDescription, Throwable previousException, ServerDescription currentServerDescription, Throwable currentException) {
            if (DefaultServerMonitor.descriptionHasChanged(previousServerDescription, currentServerDescription) || DefaultServerMonitor.exceptionHasChanged(previousException, currentException)) {
                if (currentException != null) {
                    LOGGER.info(String.format("Exception in monitor thread while connecting to server %s", DefaultServerMonitor.this.serverId.getAddress()), currentException);
                } else {
                    LOGGER.info(String.format("Monitor thread successfully connected to server with description %s", currentServerDescription));
                }
            }
        }

        private void waitForNext() {
            try {
                long millisToSleep;
                long minimumNanosToWait;
                long timeWaiting;
                long timeRemaining = this.waitForSignalOrTimeout();
                if (timeRemaining > 0L && (timeWaiting = DefaultServerMonitor.this.settings.getHeartbeatFrequency(TimeUnit.NANOSECONDS) - timeRemaining) < (minimumNanosToWait = DefaultServerMonitor.this.settings.getMinHeartbeatFrequency(TimeUnit.NANOSECONDS)) && (millisToSleep = TimeUnit.MILLISECONDS.convert(minimumNanosToWait - timeWaiting, TimeUnit.NANOSECONDS)) > 0L) {
                    Thread.sleep(millisToSleep);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }

        private long waitForSignalOrTimeout() throws InterruptedException {
            DefaultServerMonitor.this.lock.lock();
            try {
                long l = DefaultServerMonitor.this.condition.awaitNanos(DefaultServerMonitor.this.settings.getHeartbeatFrequency(TimeUnit.NANOSECONDS));
                return l;
            }
            finally {
                DefaultServerMonitor.this.lock.unlock();
            }
        }
    }
}

