/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.services;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.protobuf.ByteString;
import com.google.protobuf.Duration;
import com.google.protobuf.MessageLite;
import com.google.protobuf.util.Durations;
import com.google.re2j.Matcher;
import com.google.re2j.Pattern;
import io.grpc.Attributes;
import io.grpc.BinaryLog;
import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.Context;
import io.grpc.Deadline;
import io.grpc.ForwardingClientCall;
import io.grpc.ForwardingClientCallListener;
import io.grpc.ForwardingServerCall;
import io.grpc.ForwardingServerCallListener;
import io.grpc.Grpc;
import io.grpc.InternalMetadata;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import io.grpc.binarylog.GrpcLogEntry;
import io.grpc.binarylog.Message;
import io.grpc.binarylog.Metadata;
import io.grpc.binarylog.Peer;
import io.grpc.binarylog.Uint128;
import io.grpc.internal.GrpcUtil;
import io.grpc.services.BinaryLogProvider;
import io.grpc.services.BinaryLogSink;
import io.grpc.services.InetAddressUtil;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
final class BinlogHelper {
    private static final Logger logger = Logger.getLogger(BinlogHelper.class.getName());
    private static final boolean SERVER = true;
    private static final boolean CLIENT = false;
    static final Metadata.Key<byte[]> STATUS_DETAILS_KEY = Metadata.Key.of((String)"grpc-status-details-bin", (Metadata.BinaryMarshaller)Metadata.BINARY_BYTE_MARSHALLER);
    @VisibleForTesting
    static final SocketAddress DUMMY_SOCKET = new DummySocketAddress();
    @VisibleForTesting
    static final boolean DUMMY_IS_COMPRESSED = false;
    @VisibleForTesting
    final SinkWriter writer;

    @VisibleForTesting
    BinlogHelper(SinkWriter writer) {
        this.writer = writer;
    }

    static SocketAddress getPeerSocket(Attributes streamAttributes) {
        SocketAddress peer = (SocketAddress)streamAttributes.get(Grpc.TRANSPORT_ATTR_REMOTE_ADDR);
        if (peer == null) {
            return DUMMY_SOCKET;
        }
        return peer;
    }

    private static Deadline min(@Nullable Deadline deadline0, @Nullable Deadline deadline1) {
        if (deadline0 == null) {
            return deadline1;
        }
        if (deadline1 == null) {
            return deadline0;
        }
        return deadline0.minimum(deadline1);
    }

    public ClientInterceptor getClientInterceptor(final BinaryLog.CallId callId) {
        return new ClientInterceptor(){

            public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(final MethodDescriptor<ReqT, RespT> method, CallOptions callOptions, Channel next) {
                final AtomicInteger seq = new AtomicInteger(1);
                final String methodName = method.getFullMethodName();
                final Deadline deadline = BinlogHelper.min(callOptions.getDeadline(), Context.current().getDeadline());
                return new ForwardingClientCall.SimpleForwardingClientCall<ReqT, RespT>(next.newCall(method, callOptions)){

                    public void start(ClientCall.Listener<RespT> responseListener, Metadata headers) {
                        Duration timeout = deadline == null ? null : Durations.fromNanos((long)deadline.timeRemaining(TimeUnit.NANOSECONDS));
                        BinlogHelper.this.writer.logSendInitialMetadata(seq.getAndIncrement(), methodName, timeout, headers, false, callId);
                        ForwardingClientCallListener.SimpleForwardingClientCallListener wListener = new ForwardingClientCallListener.SimpleForwardingClientCallListener<RespT>(responseListener){

                            public void onMessage(RespT message) {
                                BinlogHelper.this.writer.logInboundMessage(seq.getAndIncrement(), method.getResponseMarshaller(), message, false, false, callId);
                                super.onMessage(message);
                            }

                            public void onHeaders(Metadata headers) {
                                SocketAddress peer = BinlogHelper.getPeerSocket(this.getAttributes());
                                BinlogHelper.this.writer.logRecvInitialMetadata(seq.getAndIncrement(), null, null, headers, false, callId, peer);
                                super.onHeaders(headers);
                            }

                            public void onClose(Status status, Metadata trailers) {
                                BinlogHelper.this.writer.logTrailingMetadata(seq.getAndIncrement(), status, trailers, false, callId);
                                super.onClose(status, trailers);
                            }
                        };
                        super.start((ClientCall.Listener)wListener, headers);
                    }

                    public void sendMessage(ReqT message) {
                        BinlogHelper.this.writer.logOutboundMessage(seq.getAndIncrement(), method.getRequestMarshaller(), message, false, false, callId);
                        super.sendMessage(message);
                    }
                };
            }
        };
    }

    public ServerInterceptor getServerInterceptor(final BinaryLog.CallId callId) {
        return new ServerInterceptor(){

            public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(final ServerCall<ReqT, RespT> call, Metadata headers, ServerCallHandler<ReqT, RespT> next) {
                final AtomicInteger seq = new AtomicInteger(1);
                SocketAddress peer = BinlogHelper.getPeerSocket(call.getAttributes());
                String methodName = call.getMethodDescriptor().getFullMethodName();
                Long timeoutNanos = (Long)headers.get(GrpcUtil.TIMEOUT_KEY);
                Duration timeout = timeoutNanos == null ? null : Durations.fromNanos((long)timeoutNanos);
                BinlogHelper.this.writer.logRecvInitialMetadata(seq.getAndIncrement(), methodName, timeout, headers, true, callId, peer);
                ForwardingServerCall.SimpleForwardingServerCall wCall = new ForwardingServerCall.SimpleForwardingServerCall<ReqT, RespT>(call){

                    public void sendMessage(RespT message) {
                        BinlogHelper.this.writer.logOutboundMessage(seq.getAndIncrement(), call.getMethodDescriptor().getResponseMarshaller(), message, false, true, callId);
                        super.sendMessage(message);
                    }

                    public void sendHeaders(Metadata headers) {
                        BinlogHelper.this.writer.logSendInitialMetadata(seq.getAndIncrement(), null, null, headers, true, callId);
                        super.sendHeaders(headers);
                    }

                    public void close(Status status, Metadata trailers) {
                        BinlogHelper.this.writer.logTrailingMetadata(seq.getAndIncrement(), status, trailers, true, callId);
                        super.close(status, trailers);
                    }
                };
                return new ForwardingServerCallListener.SimpleForwardingServerCallListener<ReqT>(next.startCall((ServerCall)wCall, headers)){

                    public void onMessage(ReqT message) {
                        BinlogHelper.this.writer.logInboundMessage(seq.getAndIncrement(), call.getMethodDescriptor().getRequestMarshaller(), message, false, true, callId);
                        super.onMessage(message);
                    }
                };
            }
        };
    }

    static Uint128 callIdToProto(BinaryLog.CallId callId) {
        Preconditions.checkNotNull((Object)callId, (Object)"callId");
        return Uint128.newBuilder().setHigh(callId.hi).setLow(callId.lo).build();
    }

    @VisibleForTesting
    static Peer socketToProto(SocketAddress address) {
        Preconditions.checkNotNull((Object)address, (Object)"address");
        Peer.Builder builder = Peer.newBuilder();
        if (address instanceof InetSocketAddress) {
            InetAddress inetAddress = ((InetSocketAddress)address).getAddress();
            if (inetAddress instanceof Inet4Address) {
                builder.setPeerType(Peer.PeerType.PEER_IPV4).setAddress(InetAddressUtil.toAddrString(inetAddress));
            } else if (inetAddress instanceof Inet6Address) {
                builder.setPeerType(Peer.PeerType.PEER_IPV6).setAddress(InetAddressUtil.toAddrString(inetAddress));
            } else {
                logger.log(Level.SEVERE, "unknown type of InetSocketAddress: {}", address);
                builder.setAddress(address.toString());
            }
            builder.setIpPort(((InetSocketAddress)address).getPort());
        } else if (address.getClass().getName().equals("io.netty.channel.unix.DomainSocketAddress")) {
            builder.setPeerType(Peer.PeerType.PEER_UNIX).setAddress(address.toString());
        } else {
            builder.setPeerType(Peer.PeerType.UNKNOWN_PEERTYPE).setAddress(address.toString());
        }
        return builder.build();
    }

    @VisibleForTesting
    static void addMetadataToProto(GrpcLogEntry.Builder entryBuilder, Metadata metadata, int maxHeaderBytes) {
        Preconditions.checkNotNull((Object)entryBuilder, (Object)"entryBuilder");
        Preconditions.checkNotNull((Object)metadata, (Object)"metadata");
        Preconditions.checkArgument((maxHeaderBytes >= 0 ? 1 : 0) != 0, (Object)"maxHeaderBytes must be non negative");
        Metadata.Builder metaBuilder = io.grpc.binarylog.Metadata.newBuilder();
        byte[][] serialized = null;
        if (maxHeaderBytes > 0 && (serialized = InternalMetadata.serialize((Metadata)metadata)) != null) {
            int written = 0;
            for (int i = 0; i < serialized.length && written < maxHeaderBytes; i += 2) {
                byte[] key = serialized[i];
                byte[] value = serialized[i + 1];
                if (written + key.length + value.length > maxHeaderBytes) continue;
                metaBuilder.addEntryBuilder().setKey(ByteString.copyFrom((byte[])key)).setValue(ByteString.copyFrom((byte[])value));
                written += key.length;
                written += value.length;
            }
        }
        entryBuilder.setTruncated(maxHeaderBytes == 0 || serialized != null && metaBuilder.getEntryCount() < serialized.length / 2);
        entryBuilder.setMetadata(metaBuilder);
    }

    @VisibleForTesting
    static void messageToProto(GrpcLogEntry.Builder entryBuilder, byte[] message, boolean compressed, int maxMessageBytes) {
        Preconditions.checkNotNull((Object)message, (Object)"message");
        Preconditions.checkArgument((maxMessageBytes >= 0 ? 1 : 0) != 0, (Object)"maxMessageBytes must be non negative");
        Message.Builder msgBuilder = Message.newBuilder().setFlags(BinlogHelper.flagsForMessage(compressed)).setLength(message.length);
        if (maxMessageBytes > 0) {
            int desiredBytes = Math.min(maxMessageBytes, message.length);
            msgBuilder.setData(ByteString.copyFrom((byte[])message, (int)0, (int)desiredBytes));
        }
        entryBuilder.setMessage(msgBuilder);
        entryBuilder.setTruncated(maxMessageBytes < message.length);
    }

    @VisibleForTesting
    static int flagsForMessage(boolean compressed) {
        return compressed ? 1 : 0;
    }

    private static class DummySocketAddress
    extends SocketAddress {
        private static final long serialVersionUID = 0L;

        private DummySocketAddress() {
        }
    }

    static final class FactoryImpl
    implements Factory {
        private static final Pattern logPatternRe = Pattern.compile((String)"[^{]+");
        private static final Pattern logOptionsRe = Pattern.compile((String)"\\{[^}]+}");
        private static final Pattern configRe = Pattern.compile((String)String.format("^(%s)(%s)?$", logPatternRe.pattern(), logOptionsRe.pattern()));
        private static final Pattern msgRe = Pattern.compile((String)"\\{m(?::(\\d+))?}");
        private static final Pattern headerRe = Pattern.compile((String)"\\{h(?::(\\d+))?}");
        private static final Pattern bothRe = Pattern.compile((String)"\\{h(?::(\\d+))?;m(?::(\\d+))?}");
        private final BinlogHelper globalLog;
        private final Map<String, BinlogHelper> perServiceLogs;
        private final Map<String, BinlogHelper> perMethodLogs;
        private final Set<String> blacklistedMethods;

        @VisibleForTesting
        FactoryImpl(BinaryLogSink sink, String configurationString) {
            Preconditions.checkNotNull((Object)sink, (Object)"sink");
            BinlogHelper globalLog = null;
            HashMap<String, BinlogHelper> perServiceLogs = new HashMap<String, BinlogHelper>();
            HashMap<String, BinlogHelper> perMethodLogs = new HashMap<String, BinlogHelper>();
            HashSet<String> blacklistedMethods = new HashSet<String>();
            if (configurationString != null && configurationString.length() > 0) {
                for (String configuration : Splitter.on((char)',').split((CharSequence)configurationString)) {
                    Matcher configMatcher = configRe.matcher((CharSequence)configuration);
                    if (!configMatcher.matches()) {
                        throw new IllegalArgumentException("Bad input: " + configuration);
                    }
                    String methodOrSvc = configMatcher.group(1);
                    String binlogOptionStr = configMatcher.group(2);
                    BinlogHelper binLog = FactoryImpl.createBinaryLog(sink, binlogOptionStr);
                    if (binLog == null) continue;
                    if (methodOrSvc.equals("*")) {
                        if (globalLog != null) {
                            logger.log(Level.SEVERE, "Ignoring duplicate entry: {0}", configuration);
                            continue;
                        }
                        globalLog = binLog;
                        logger.log(Level.INFO, "Global binlog: {0}", binlogOptionStr);
                        continue;
                    }
                    if (FactoryImpl.isServiceGlob(methodOrSvc)) {
                        String service = MethodDescriptor.extractFullServiceName((String)methodOrSvc);
                        if (perServiceLogs.containsKey(service)) {
                            logger.log(Level.SEVERE, "Ignoring duplicate entry: {0}", configuration);
                            continue;
                        }
                        perServiceLogs.put(service, binLog);
                        logger.log(Level.INFO, "Service binlog: service={0} config={1}", new Object[]{service, binlogOptionStr});
                        continue;
                    }
                    if (methodOrSvc.startsWith("-")) {
                        String blacklistedMethod = methodOrSvc.substring(1);
                        if (blacklistedMethod.length() == 0 || blacklistedMethods.add(blacklistedMethod)) continue;
                        logger.log(Level.SEVERE, "Ignoring duplicate entry: {0}", configuration);
                        continue;
                    }
                    if (perMethodLogs.containsKey(methodOrSvc)) {
                        logger.log(Level.SEVERE, "Ignoring duplicate entry: {0}", configuration);
                        continue;
                    }
                    perMethodLogs.put(methodOrSvc, binLog);
                    logger.log(Level.INFO, "Method binlog: method={0} config={1}", new Object[]{methodOrSvc, binlogOptionStr});
                }
            }
            this.globalLog = globalLog;
            this.perServiceLogs = Collections.unmodifiableMap(perServiceLogs);
            this.perMethodLogs = Collections.unmodifiableMap(perMethodLogs);
            this.blacklistedMethods = Collections.unmodifiableSet(blacklistedMethods);
        }

        @Override
        public BinlogHelper getLog(String fullMethodName) {
            if (this.blacklistedMethods.contains(fullMethodName)) {
                return null;
            }
            BinlogHelper methodLog = this.perMethodLogs.get(fullMethodName);
            if (methodLog != null) {
                return methodLog;
            }
            BinlogHelper serviceLog = this.perServiceLogs.get(MethodDescriptor.extractFullServiceName((String)fullMethodName));
            if (serviceLog != null) {
                return serviceLog;
            }
            return this.globalLog;
        }

        @Nullable
        @VisibleForTesting
        static BinlogHelper createBinaryLog(BinaryLogSink sink, @Nullable String logConfig) {
            if (logConfig == null) {
                return new BinlogHelper(new SinkWriterImpl(sink, Integer.MAX_VALUE, Integer.MAX_VALUE));
            }
            try {
                int maxMsgBytes;
                int maxHeaderBytes;
                Matcher headerMatcher = headerRe.matcher((CharSequence)logConfig);
                if (headerMatcher.matches()) {
                    String maxHeaderStr = headerMatcher.group(1);
                    maxHeaderBytes = maxHeaderStr != null ? Integer.parseInt(maxHeaderStr) : Integer.MAX_VALUE;
                    maxMsgBytes = 0;
                } else {
                    Matcher msgMatcher = msgRe.matcher((CharSequence)logConfig);
                    if (msgMatcher.matches()) {
                        maxHeaderBytes = 0;
                        String maxMsgStr = msgMatcher.group(1);
                        maxMsgBytes = maxMsgStr != null ? Integer.parseInt(maxMsgStr) : Integer.MAX_VALUE;
                    } else {
                        Matcher bothMatcher = bothRe.matcher((CharSequence)logConfig);
                        if (bothMatcher.matches()) {
                            String maxHeaderStr = bothMatcher.group(1);
                            String maxMsgStr = bothMatcher.group(2);
                            maxHeaderBytes = maxHeaderStr != null ? Integer.parseInt(maxHeaderStr) : Integer.MAX_VALUE;
                            maxMsgBytes = maxMsgStr != null ? Integer.parseInt(maxMsgStr) : Integer.MAX_VALUE;
                        } else {
                            logger.log(Level.SEVERE, "Illegal log config pattern: " + logConfig);
                            return null;
                        }
                    }
                }
                return new BinlogHelper(new SinkWriterImpl(sink, maxHeaderBytes, maxMsgBytes));
            }
            catch (NumberFormatException e) {
                logger.log(Level.SEVERE, "Illegal log config pattern: " + logConfig);
                return null;
            }
        }

        static boolean isServiceGlob(String input) {
            return input.endsWith("/*");
        }
    }

    static interface Factory {
        @Nullable
        public BinlogHelper getLog(String var1);
    }

    static abstract class SinkWriter {
        SinkWriter() {
        }

        abstract void logSendInitialMetadata(int var1, String var2, Duration var3, Metadata var4, boolean var5, BinaryLog.CallId var6);

        abstract void logRecvInitialMetadata(int var1, String var2, Duration var3, Metadata var4, boolean var5, BinaryLog.CallId var6, SocketAddress var7);

        abstract void logTrailingMetadata(int var1, Status var2, Metadata var3, boolean var4, BinaryLog.CallId var5);

        abstract <T> void logOutboundMessage(int var1, MethodDescriptor.Marshaller<T> var2, T var3, boolean var4, boolean var5, BinaryLog.CallId var6);

        abstract <T> void logInboundMessage(int var1, MethodDescriptor.Marshaller<T> var2, T var3, boolean var4, boolean var5, BinaryLog.CallId var6);

        abstract int getMaxHeaderBytes();

        abstract int getMaxMessageBytes();
    }

    static final class SinkWriterImpl
    extends SinkWriter {
        private final BinaryLogSink sink;
        private final int maxHeaderBytes;
        private final int maxMessageBytes;

        SinkWriterImpl(BinaryLogSink sink, int maxHeaderBytes, int maxMessageBytes) {
            this.sink = sink;
            this.maxHeaderBytes = maxHeaderBytes;
            this.maxMessageBytes = maxMessageBytes;
        }

        @Override
        void logSendInitialMetadata(int seq, @Nullable String methodName, @Nullable Duration timeout, Metadata metadata, boolean isServer, BinaryLog.CallId callId) {
            Preconditions.checkArgument((methodName == null || !isServer ? 1 : 0) != 0);
            Preconditions.checkArgument((timeout == null || !isServer ? 1 : 0) != 0);
            Preconditions.checkArgument((methodName == null || !methodName.startsWith("/") ? 1 : 0) != 0);
            GrpcLogEntry.Builder entryBuilder = GrpcLogEntry.newBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.Type.SEND_INITIAL_METADATA).setLogger(isServer ? GrpcLogEntry.Logger.SERVER : GrpcLogEntry.Logger.CLIENT).setCallId(BinlogHelper.callIdToProto(callId));
            BinlogHelper.addMetadataToProto(entryBuilder, metadata, this.maxHeaderBytes);
            if (methodName != null) {
                entryBuilder.setMethodName("/" + methodName);
            }
            if (timeout != null) {
                entryBuilder.setTimeout(timeout);
            }
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        void logRecvInitialMetadata(int seq, @Nullable String methodName, @Nullable Duration timeout, Metadata metadata, boolean isServer, BinaryLog.CallId callId, SocketAddress peerSocket) {
            Preconditions.checkArgument((methodName == null || isServer ? 1 : 0) != 0);
            Preconditions.checkArgument((timeout == null || isServer ? 1 : 0) != 0);
            Preconditions.checkArgument((methodName == null || !methodName.startsWith("/") ? 1 : 0) != 0);
            GrpcLogEntry.Builder entryBuilder = GrpcLogEntry.newBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.Type.RECV_INITIAL_METADATA).setLogger(isServer ? GrpcLogEntry.Logger.SERVER : GrpcLogEntry.Logger.CLIENT).setCallId(BinlogHelper.callIdToProto(callId)).setPeer(BinlogHelper.socketToProto(peerSocket));
            BinlogHelper.addMetadataToProto(entryBuilder, metadata, this.maxHeaderBytes);
            if (methodName != null) {
                entryBuilder.setMethodName("/" + methodName);
            }
            if (timeout != null) {
                entryBuilder.setTimeout(timeout);
            }
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        void logTrailingMetadata(int seq, Status status, Metadata metadata, boolean isServer, BinaryLog.CallId callId) {
            byte[] statusDetailBytes;
            GrpcLogEntry.Builder entryBuilder = GrpcLogEntry.newBuilder().setSequenceIdWithinCall(seq).setType(isServer ? GrpcLogEntry.Type.SEND_TRAILING_METADATA : GrpcLogEntry.Type.RECV_TRAILING_METADATA).setLogger(isServer ? GrpcLogEntry.Logger.SERVER : GrpcLogEntry.Logger.CLIENT).setCallId(BinlogHelper.callIdToProto(callId)).setStatusCode(status.getCode().value());
            String statusDescription = status.getDescription();
            if (statusDescription != null) {
                entryBuilder.setStatusMessage(statusDescription);
            }
            if ((statusDetailBytes = (byte[])metadata.get(STATUS_DETAILS_KEY)) != null) {
                entryBuilder.setStatusDetails(ByteString.copyFrom((byte[])statusDetailBytes));
            }
            BinlogHelper.addMetadataToProto(entryBuilder, metadata, this.maxHeaderBytes);
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        <T> void logOutboundMessage(int seq, MethodDescriptor.Marshaller<T> marshaller, T message, boolean compressed, boolean isServer, BinaryLog.CallId callId) {
            if (marshaller != BinaryLogProvider.BYTEARRAY_MARSHALLER) {
                throw new IllegalStateException("Expected the BinaryLog's ByteArrayMarshaller");
            }
            GrpcLogEntry.Builder entryBuilder = GrpcLogEntry.newBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.Type.SEND_MESSAGE).setLogger(isServer ? GrpcLogEntry.Logger.SERVER : GrpcLogEntry.Logger.CLIENT).setCallId(BinlogHelper.callIdToProto(callId));
            BinlogHelper.messageToProto(entryBuilder, (byte[])message, compressed, this.maxMessageBytes);
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        <T> void logInboundMessage(int seq, MethodDescriptor.Marshaller<T> marshaller, T message, boolean compressed, boolean isServer, BinaryLog.CallId callId) {
            if (marshaller != BinaryLogProvider.BYTEARRAY_MARSHALLER) {
                throw new IllegalStateException("Expected the BinaryLog's ByteArrayMarshaller");
            }
            GrpcLogEntry.Builder entryBuilder = GrpcLogEntry.newBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.Type.RECV_MESSAGE).setLogger(isServer ? GrpcLogEntry.Logger.SERVER : GrpcLogEntry.Logger.CLIENT).setCallId(BinlogHelper.callIdToProto(callId));
            BinlogHelper.messageToProto(entryBuilder, (byte[])message, compressed, this.maxMessageBytes);
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        int getMaxHeaderBytes() {
            return this.maxHeaderBytes;
        }

        @Override
        int getMaxMessageBytes() {
            return this.maxMessageBytes;
        }
    }
}

