/*
 * Decompiled with CFR 0.152.
 */
package com.github.felfert.sslutils;

import com.github.felfert.sslutils.SimpleDERReader;
import java.io.ByteArrayInputStream;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.spec.DSAPrivateKeySpec;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.Collection;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.xml.bind.DatatypeConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PEMDecoder {
    static final Logger LOGGER = LoggerFactory.getLogger(PEMDecoder.class);
    private static final Pattern PRIVKEY_DSA_BEGIN = Pattern.compile("^-----BEGIN DSA PRIVATE KEY-----\n");
    private static final Pattern PRIVKEY_DSA_END = Pattern.compile("\n-----END DSA PRIVATE KEY-----\\s*$");
    private static final Pattern PRIVKEY_RSA_BEGIN = Pattern.compile("^-----BEGIN RSA PRIVATE KEY-----\n");
    private static final Pattern PRIVKEY_RSA_END = Pattern.compile("\n-----END RSA PRIVATE KEY-----\\s*$");
    private static final Pattern PRIVKEY_EC_BEGIN = Pattern.compile("^-----BEGIN EC PRIVATE KEY-----\n");
    private static final Pattern PRIVKEY_EC_END = Pattern.compile("-----END EC PRIVATE KEY-----\\s*$");
    private static final Pattern PRIVKEY_PROCTYPE = Pattern.compile("^Proc-Type:\\s+4,ENCRYPTED\n");
    private static final Pattern PRIVKEY_DEKINFO = Pattern.compile("^DEK-Info:\\s+([^,]+),(\\S+)\n");

    public Collection<? extends Certificate> decodeCertificates(String pem) throws CertificateException {
        if (null == pem || pem.isEmpty()) {
            throw new IllegalArgumentException("PEM data is null or empty");
        }
        try {
            ByteArrayInputStream bis = new ByteArrayInputStream(pem.getBytes(Charset.defaultCharset()));
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            return cf.generateCertificates(bis);
        }
        catch (Throwable t) {
            LOGGER.warn("", t);
            throw t;
        }
    }

    public KeyPair decodePrivKey(String pem, String password) {
        boolean encrypted = false;
        KeyType keyType = KeyType.UNKNOWN;
        KeyAlgo algo = KeyAlgo.UNKNOWN;
        String tmp = null;
        byte[] salt = null;
        if (null == pem) {
            throw new IllegalArgumentException("Not a PEM-encoded private key");
        }
        if (PRIVKEY_RSA_BEGIN.matcher(pem).find() && PRIVKEY_RSA_END.matcher(pem).find()) {
            tmp = PRIVKEY_RSA_BEGIN.matcher(pem).replaceFirst("");
            tmp = PRIVKEY_RSA_END.matcher(tmp).replaceFirst("").trim();
            keyType = KeyType.RSA;
        }
        if (PRIVKEY_DSA_BEGIN.matcher(pem).find() && PRIVKEY_DSA_END.matcher(pem).find()) {
            tmp = PRIVKEY_DSA_BEGIN.matcher(pem).replaceFirst("");
            tmp = PRIVKEY_DSA_END.matcher(tmp).replaceFirst("").trim();
            keyType = KeyType.DSA;
        }
        if (PRIVKEY_EC_BEGIN.matcher(pem).find() && PRIVKEY_EC_END.matcher(pem).find()) {
            tmp = PRIVKEY_EC_BEGIN.matcher(pem).replaceFirst("");
            tmp = PRIVKEY_EC_END.matcher(tmp).replaceFirst("").trim();
            keyType = KeyType.EC;
        }
        if (keyType.equals((Object)KeyType.UNKNOWN)) {
            throw new IllegalArgumentException("Not a PEM-encoded private key");
        }
        if (PRIVKEY_PROCTYPE.matcher(tmp).find()) {
            encrypted = true;
            Matcher m = PRIVKEY_DEKINFO.matcher(tmp = PRIVKEY_PROCTYPE.matcher(tmp).replaceFirst("").trim());
            if (m.find()) {
                try {
                    algo = KeyAlgo.valueOf(m.group(1).replaceAll("-", "_"));
                }
                catch (Throwable t) {
                    throw new IllegalArgumentException("Invalid or unsupported algorithm in DEK-Info", t);
                }
                String saltstr = m.group(2);
                if (0 != saltstr.length() % 2 || saltstr.length() < 16) {
                    throw new IllegalArgumentException("Length of hex salt in DEK-Info is less than 16 or not a multiple of 2");
                }
                salt = DatatypeConverter.parseHexBinary((String)saltstr.toUpperCase(Locale.getDefault()));
                tmp = PRIVKEY_DEKINFO.matcher(tmp).replaceFirst("").trim();
            } else {
                throw new IllegalArgumentException("Missing or invalid DEK-Info header");
            }
        }
        byte[] data = DatatypeConverter.parseBase64Binary((String)tmp);
        if (encrypted) {
            if (null == password) {
                throw new IllegalArgumentException("PEM is encrypted, but no password was specified");
            }
            data = this.decryptPEM(data, algo, salt, password.getBytes(Charset.defaultCharset()));
        }
        return this.decodePrivKey(data, keyType);
    }

    private KeyPair decodePrivKey(byte[] data, KeyType keyType) {
        SimpleDERReader dr = new SimpleDERReader(data);
        byte[] seq = dr.readSequenceAsByteArray();
        if (dr.available() != 0) {
            throw new IllegalArgumentException("Padding in PRIVATE KEY DER stream.");
        }
        dr.resetInput(seq);
        BigInteger version = dr.readInt();
        switch (keyType) {
            case DSA: {
                if (version.compareTo(BigInteger.ZERO) != 0) {
                    throw new IllegalArgumentException("Wrong version (" + version + ") in DSA PRIVATE KEY DER stream.");
                }
                BigInteger p = dr.readInt();
                BigInteger q = dr.readInt();
                BigInteger g = dr.readInt();
                BigInteger y = dr.readInt();
                BigInteger x = dr.readInt();
                if (dr.available() != 0) {
                    throw new IllegalArgumentException("Padding in DSA PRIVATE KEY DER stream.");
                }
                return this.generateKeyPair("DSA", new DSAPrivateKeySpec(x, p, q, g), new DSAPublicKeySpec(y, p, q, g));
            }
            case RSA: {
                if (version.compareTo(BigInteger.ZERO) != 0 && version.compareTo(BigInteger.ONE) != 0) {
                    throw new IllegalArgumentException("Wrong version (" + version + ") in RSA PRIVATE KEY DER stream.");
                }
                BigInteger n = dr.readInt();
                BigInteger e = dr.readInt();
                BigInteger d = dr.readInt();
                BigInteger primeP = dr.readInt();
                BigInteger primeQ = dr.readInt();
                BigInteger expP = dr.readInt();
                BigInteger expQ = dr.readInt();
                BigInteger coeff = dr.readInt();
                return this.generateKeyPair("RSA", new RSAPrivateCrtKeySpec(n, e, d, primeP, primeQ, expP, expQ, coeff), new RSAPublicKeySpec(n, e));
            }
            case EC: {
                if (version.compareTo(BigInteger.ZERO) != 0) {
                    throw new IllegalArgumentException("Wrong version (" + version + ") in EC PRIVATE KEY DER stream.");
                }
                throw new IllegalArgumentException("Not yet");
            }
        }
        throw new IllegalArgumentException("Unknown key type");
    }

    private byte[] decryptPEM(byte[] data, KeyAlgo algo, byte[] salt, byte[] password) {
        try {
            SecretKeySpec keyspec;
            Cipher cipher;
            switch (algo) {
                case AES_128_CBC: {
                    cipher = Cipher.getInstance("AES/CBC/NoPadding");
                    int mks = Cipher.getMaxAllowedKeyLength("AES/CBC/NoPadding");
                    if (mks < 128) {
                        throw new IllegalArgumentException("Maximum key size for AES is " + mks + ". cryptograpy export restrictions?");
                    }
                    keyspec = new SecretKeySpec(this.generateKeyFromPasswordSaltWithMD5(password, salt, 16), "AES");
                    break;
                }
                case AES_192_CBC: {
                    cipher = Cipher.getInstance("AES/CBC/NoPadding");
                    int mks = Cipher.getMaxAllowedKeyLength("AES/CBC/NoPadding");
                    if (mks < 192) {
                        throw new IllegalArgumentException("Maximum key size for AES is " + mks + ". cryptography export restrictions?");
                    }
                    keyspec = new SecretKeySpec(this.generateKeyFromPasswordSaltWithMD5(password, salt, 24), "AES");
                    break;
                }
                case AES_256_CBC: {
                    cipher = Cipher.getInstance("AES/CBC/NoPadding");
                    int mks = Cipher.getMaxAllowedKeyLength("AES/CBC/NoPadding");
                    if (mks < 256) {
                        throw new IllegalArgumentException("Maximum key size for AES is " + mks + ". cryptography export restrictions?");
                    }
                    keyspec = new SecretKeySpec(this.generateKeyFromPasswordSaltWithMD5(password, salt, 32), "AES");
                    break;
                }
                case DES_EDE3_CBC: {
                    cipher = Cipher.getInstance("DESede/CBC/NoPadding");
                    int mks = Cipher.getMaxAllowedKeyLength("DESede/CBC/NoPadding");
                    if (mks < 192) {
                        throw new IllegalArgumentException("Maximum key size for TripleDES is " + mks + ". cryptography export restrictions?");
                    }
                    keyspec = new SecretKeySpec(this.generateKeyFromPasswordSaltWithMD5(password, salt, 24), "DESede");
                    break;
                }
                case DES_CBC: {
                    cipher = Cipher.getInstance("DES/CBC/NoPadding");
                    int mks = Cipher.getMaxAllowedKeyLength("DES/CBC/NoPadding");
                    if (mks < 64) {
                        throw new IllegalArgumentException("Maximum key size for DES is " + mks + ". cryptography export restrictions?");
                    }
                    keyspec = new SecretKeySpec(this.generateKeyFromPasswordSaltWithMD5(password, salt, 8), "DES");
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid key encryption algorithm");
                }
            }
            cipher.init(2, (Key)keyspec, new IvParameterSpec(salt));
            return this.removePadding(cipher.doFinal(data), cipher.getBlockSize());
        }
        catch (Throwable t) {
            LOGGER.debug("", t);
            if (t instanceof IllegalArgumentException) {
                throw (IllegalArgumentException)t;
            }
            throw new IllegalArgumentException("Unable to decrypt key data", t);
        }
    }

    private byte[] removePadding(byte[] data, int blockSize) {
        int padding = data[data.length - 1] & 0xFF;
        String padError = "Decrypted PEM has wrong padding, did you specify the correct password?";
        if (padding < 1 || padding > blockSize) {
            throw new IllegalArgumentException("Decrypted PEM has wrong padding, did you specify the correct password?");
        }
        for (int i = 2; i <= padding; ++i) {
            if (data[data.length - i] == padding) continue;
            throw new IllegalArgumentException("Decrypted PEM has wrong padding, did you specify the correct password?");
        }
        byte[] tmp = new byte[data.length - padding];
        System.arraycopy(data, 0, tmp, 0, data.length - padding);
        return tmp;
    }

    private byte[] generateKeyFromPasswordSaltWithMD5(byte[] password, byte[] salt, int keyLen) {
        MessageDigest md5;
        try {
            md5 = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("JVM does not support MD5", e);
        }
        byte[] key = new byte[keyLen];
        byte[] tmp = new byte[md5.getDigestLength()];
        int kl = keyLen;
        while (true) {
            md5.update(password, 0, password.length);
            md5.update(salt, 0, 8);
            int copy = kl < tmp.length ? kl : tmp.length;
            try {
                md5.digest(tmp, 0, tmp.length);
            }
            catch (Throwable t) {
                throw new IllegalArgumentException("Could not digest password", t);
            }
            System.arraycopy(tmp, 0, key, key.length - kl, copy);
            if ((kl -= copy) <= 0) {
                return key;
            }
            md5.update(tmp, 0, tmp.length);
        }
    }

    private KeyPair generateKeyPair(String algorithm, KeySpec privSpec, KeySpec pubSpec) {
        try {
            KeyFactory kf = KeyFactory.getInstance(algorithm);
            PublicKey pubKey = kf.generatePublic(pubSpec);
            PrivateKey privKey = kf.generatePrivate(privSpec);
            return new KeyPair(pubKey, privKey);
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException ex) {
            throw new IllegalArgumentException(ex);
        }
    }

    private static enum KeyAlgo {
        UNKNOWN,
        AES_128_CBC,
        AES_192_CBC,
        AES_256_CBC,
        DES_EDE3_CBC,
        DES_CBC;

    }

    private static enum KeyType {
        UNKNOWN,
        RSA,
        DSA,
        EC;

    }
}

