/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.yeepay.sdk.request.order;

import com.fshows.yeepay.sdk.request.YopBizRequest;
import com.fshows.yeepay.sdk.response.order.YopBankTransferResponse;
import lombok.Data;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;

/**
 * 银行转账支付接口
 * @author
 * @version YopBankTransferRequest.java, v 0.1 2025-11-07
 */
@Data
public class YopBankTransferRequest extends YopBizRequest<YopBankTransferResponse> {

    private static final long serialVersionUID = -2213838555285397468L;

    /**
     * 商户编号 收单主体商编，平台商或服务商下的子商户，普通特约商户
     * (必填)
     */
    @NotBlank(message = "merchantNo不能为空")
    @Length(max = 11, message = "merchantNo长度不能超过11")
    private String merchantNo;

    /**
     * 商户收款请求号<br>商户系统内部生成的订单号，需要保持在同一个商户下唯一
     * (必填)
     */
    @NotBlank(message = "orderId不能为空")
    @Length(max = 64, message = "orderId长度不能超过64")
    private String orderId;

    /**
     * 订单金额<br>业务上是必须参数，单位： 元， 两位小数， 最低 0.01
     * (必填)
     */
    @NotNull(message = "orderAmount不能为空")
    private BigDecimal orderAmount;

    /**
     * 商品名称<br>如：旺仔牛奶
     * (必填)
     */
    @NotBlank(message = "goodsName不能为空")
    @Length(max = 32, message = "goodsName长度不能超过32")
    private String goodsName;

    /**
     * 自定义参数信息<br>json格式
     * 自定义参数信息：json格式
     * userIp必传
     * 示例值：{"userIp":"127.0.0.1"}
     * (必填)
     */
    @NotBlank(message = "businessInfo不能为空")
    @Length(max = 128, message = "businessInfo长度不能超过128")
    private String businessInfo;

    /**
     * 付款方名称<br>【必填条件】当校验模式选择PAYERACCOUNT_SUNINGBANK时，必填
     * (条件必填)
     */
    @Length(max = 64, message = "payerAccountName长度不能超过64")
    private String payerAccountName;

    /**
     * 银行账户号<br>【必填条件】当校验模式选择易汇通时，必须上送银行账户号
     * (条件必填)
     */
    @Length(max = 64, message = "bankAccountNo长度不能超过64")
    private String bankAccountNo;

    /**
     * 客户id<br>客户自定义字段，长度需要在15位-32位。向同一个账号转账，就用同一个customerid
     * 【必填条件】当校验模式选择REMARKCODE_SUNINGBANK或PAYERACCOUNT_SUNINGBANK时，必须上送客户id
     * (条件必填)
     */
    @Length(min = 15, max = 32, message = "customerId长度必须在15-32位之间")
    private String customerId;

    /**
     * 付款方账户号<br>【必填条件】当校验模式选择PAYERACCOUNT_SUNINGBANK时，必填
     * (条件必填)
     */
    @Length(max = 64, message = "payerAccountNo长度不能超过64")
    private String payerAccountNo;

    /**
     * 入金通知地址<br>【必填条件】当核验模式选择PAYERACCOUNT_SUNINGBANK时必须上送入金通知地址
     * (条件必填)
     */
    @Length(max = 128, message = "accountNotifyUrl长度不能超过128")
    private String accountNotifyUrl;

    /**
     * 订单截止时间<br>格式为yyyy-MM-dd HH:mm:ss，为空时默认在请求下单120分钟后失效，最长支持30天
     * (选填)
     */
    private String expiredTime;

    /**
     * 支付结果通知地址<br>接收支付结果的通知地址
     * (选填)
     */
    @Length(max = 200, message = "notifyUrl长度不能超过200")
    private String notifyUrl;

    /**
     * 自定义对账备注<br>商户可以自定义自身业务需要使用的字段： 如对账时定义该订单应属的会计科目
     * (选填)
     */
    @Length(max = 256, message = "memo长度不能超过256")
    private String memo;

    /**
     * 校验模式<br>
     * REMARKCODE：固定账户号+动态附言模式<br>
     * ACCOUNTNO：动态账户号模式<br>
     * ACCOUNTNO_SUNINGBANK:易汇通-动态账户号模式-江苏苏商银行<br>
     * REMARKCODE_SUNINGBANK:易汇通-固定账户号+动态附言模式-江苏苏商银行<br>
     * PAYERACCOUNT_SUNINGBANK:易汇通-固定账户号+付款方账户模式-江苏苏商银行<br>
     * 默认为动态账户号模式
     * (选填)
     */
    private String checkType;

    /**
     * 分账订单标记<br>
     * DELAY_SETTLE:需要分账<br>
     * REAL_TIME:不需要分账
     * (选填)
     */
    private String fundProcessType;

    /**
     * 清算回调地址
     * (选填)
     */
    @Length(max = 200, message = "csUrl长度不能超过200")
    private String csUrl;

    /**
     * 易宝营销信息<br>json格式；需要参加易宝营销活动的可传入此值
     * (选填)
     */
    @Length(max = 1024, message = "ypPromotionInfo长度不能超过1024")
    private String ypPromotionInfo;

    @Override
    public Class<YopBankTransferResponse> getResponseClass() {
        return YopBankTransferResponse.class;
    }
}