/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.yeepay.sdk.client.impl;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.yeepay.sdk.client.YopBaseClient;
import com.fshows.yeepay.sdk.exception.YopPayException;
import com.fshows.yeepay.sdk.request.YopBizRequest;
import com.fshows.yeepay.sdk.request.sign.YopSignRequest;
import com.fshows.yeepay.sdk.response.YopBaseResponse;
import com.fshows.yeepay.sdk.response.sign.YopSignResponse;
import com.fshows.yeepay.sdk.support.CustomSdkConfigProvider;
import com.fshows.yeepay.sdk.util.SignUtil;
import com.fshows.yeepay.sdk.util.StringPool;
import com.fshows.yeepay.sdk.util.YopSignUtil;
import com.yeepay.g3.sdk.yop.client.YopRequest;
import com.yeepay.g3.sdk.yop.client.YopResponse;
import com.yeepay.g3.sdk.yop.client.YopRsaClient;
import com.yeepay.g3.sdk.yop.config.AppSdkConfigProviderRegistry;
import com.yeepay.g3.sdk.yop.error.YopError;
import com.yeepay.g3.sdk.yop.http.HttpMethodName;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.text.MessageFormat;
import java.util.Map;

/**
 * @author youmingming
 * @version YopBaseClientImpl.java, v 0.1 2022-02-09 下午11:06 youmingming
 */
@Slf4j
public class YopBaseClientImpl implements YopBaseClient {

    private static final String SUCCESS = "SUCCESS";

    /**
     * 应用appId;
     */
    private String appId;
    /**
     * 私钥
     */
    private String fbPrivateKey;
    /**
     * 公钥
     */
    private String yopPublicKey;
    /**
     * 域名前缀
     */
    private String prefixUrl;


    public YopBaseClientImpl(String appId, String fbPrivateKey, String yopPublicKey, String prefixUrl) {
        this.appId = appId;
        this.fbPrivateKey = fbPrivateKey;
        this.yopPublicKey = yopPublicKey;
        this.prefixUrl = prefixUrl;
        CustomSdkConfigProvider configProvider = new CustomSdkConfigProvider(this.prefixUrl, this.yopPublicKey, this.appId);
        AppSdkConfigProviderRegistry.registerCustomProvider(configProvider);
    }

    /**
     * 请求联动接口
     *
     * @param request    请求参数
     * @param definition 方法枚举
     * @param <R>        返参
     * @return UmBaseResponse
     */
    @Override
    public <R> YopBaseResponse<R> execute(YopBizRequest<R> request, YopApiDefinition definition) throws YopPayException {
        //获取开始时间
        final long startTime = System.currentTimeMillis();
        try {
            YopRequest yopRequest = new YopRequest(appId, fbPrivateKey);
            //填充业务参数
            this.populateParam(request, yopRequest);
            log.info("【yeepay-sdk】接口调用开始 >> url={},method={},request={}", prefixUrl, definition.getMethod(), yopRequest.getParams());
            YopResponse yopResponse;
            if (StringPool.POST.equals(definition.getMethodType())) {
                yopResponse = YopRsaClient.post(definition.getMethod(), yopRequest);
            } else {
                yopResponse = YopRsaClient.get(definition.getMethod(), yopRequest);
            }
            log.info("【yeepay-sdk】接口调用结束 >> url={},method={},request={},response={},cost={}", prefixUrl, definition.getMethod(), yopRequest.getParams(), yopResponse, System.currentTimeMillis() - startTime);
            if (!SUCCESS.equals(yopResponse.getState())) {
                YopError yopError = yopResponse.getError();
                throw new YopPayException(MessageFormat.format("[{0}]{1}", yopError.getCode(), yopError.getSubMessage()));
            }
            return parseResponse(request, yopResponse);
        } catch (Exception ex) {
            log.error("【yeepay-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", prefixUrl, definition.getMethod(), request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new YopPayException(ex.getMessage());
        }
    }

    /**
     * 填充业务参数
     *
     * @param request
     * @param yopRequest
     * @param <R>
     */
    private <R> void populateParam(YopBizRequest<R> request, YopRequest yopRequest) {
        String jsonString = JSON.toJSONString(request);
        Map<String, Object> map = JSON.parseObject(jsonString, Map.class);
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            yopRequest.addParam(entry.getKey(), entry.getValue());
        }
    }

    /**
     * 解析响应参数
     *
     * @param request
     * @param yopResponse
     * @param <R>
     * @return
     */
    private <R> YopBaseResponse<R> parseResponse(YopBizRequest<R> request, YopResponse yopResponse) throws YopPayException {
        if (ObjectUtil.isNull(yopResponse.getResult())) {
            throw new YopPayException("易宝SDK请求返回为空");
        }
        String jsonString = JSON.toJSONString(yopResponse.getResult());
        JSONObject jsonObject = JSON.parseObject(jsonString);
        YopBaseResponse<R> response = new YopBaseResponse<>();
        response.setErrorCode(StringUtils.isBlank(jsonObject.getString("code")) ? jsonObject.getString("returnCode") : jsonObject.getString("code"));
        response.setErrorMessage(StringUtils.isBlank(jsonObject.getString("message")) ? jsonObject.getString("returnMsg") : jsonObject.getString("message"));
        response.setReturnValue(jsonObject.toJavaObject(request.getResponseClass()));
        return response;
    }


    @Override
    public YopSignResponse getYopSignData(YopSignRequest request) throws YopPayException {
        try {
            YopRequest yopRequest = new YopRequest(appId, fbPrivateKey);
            log.error("【yeepay-sdk】getYopSignData调用开始 >> request={},yopRequest={}", request, yopRequest);
            //填充业务参数
            Map<String, Object> map = JSON.parseObject(request.getJsonData(), Map.class);
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                yopRequest.addParam(entry.getKey(), entry.getValue());
            }
            HttpMethodName httpMethodName;
            if (StringPool.POST.equals(request.getMethodType())) {
                httpMethodName = HttpMethodName.POST;
            } else {
                httpMethodName = HttpMethodName.GET;
            }
            YopSignUtil.getSignData(request.getApiUrl(), yopRequest, httpMethodName);
            YopSignResponse yopSignResponse = new YopSignResponse();
            yopSignResponse.setHeaderData(JSON.toJSONString(yopRequest.getHeaders()));
            return yopSignResponse;
        } catch (Exception ex) {
            log.error("【yeepay-sdk】getYopSignData调用失败 >> request={},ex={}", request, ExceptionUtils.getStackTrace(ex));
            throw new YopPayException(ex.getMessage());
        }
    }
}