/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.xft.sdk.client.impl;

import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.xft.sdk.client.XftBaseClient;
import com.fshows.xft.sdk.enums.XftApiEnum;
import com.fshows.xft.sdk.exception.XftPayException;
import com.fshows.xft.sdk.request.XftBizRequest;
import com.fshows.xft.sdk.response.XftBaseResponse;
import com.fshows.xft.sdk.util.SignUtil;
import com.fshows.xft.sdk.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.util.Map;
import java.util.TreeMap;

/**
 * @author zhaoxumin
 * @version XftBaseClientImpl.java, v 0.1 2023-08-25 下午11:06 zhaoxumin
 */
@Slf4j
public class XftBaseClientImpl implements XftBaseClient {

    /**
     * 应用appId
     */
    private final String appId;
    /**
     * 签名类型
     */
    private final String signType;
    /**
     * 密钥
     */
    private final String key;


    private static final SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }


    public XftBaseClientImpl(String appId, String signType, String key) {
        this.appId = appId;
        this.signType = signType;
        this.key = key;
    }

    /**
     * 请求星富通接口
     *
     * @param request          请求参数
     * @param xftApiDefinition 方法枚举
     * @param url
     * @return UmBaseResponse
     * <p>
     * XftApiDefinition api方法请参考下列API枚举
     */
    @Override
    public <R> XftBaseResponse<R> execute(XftBizRequest<R> request, XftApiDefinition xftApiDefinition, String url) throws XftPayException {
        return execute(request, xftApiDefinition, url, this.appId);
    }

    /**
     * 请求星富通接口
     *
     * @param request    请求参数
     * @param definition 方法枚举
     * @param <R>        返参
     * @return UmBaseResponse
     */
    @Override
    public <R> XftBaseResponse<R> execute(XftBizRequest<R> request, XftApiDefinition definition, String postUrl, String subAppId) throws XftPayException {
        String method = definition.getMethod();
        //验证参数
        validateParam(request, method);
        //获取开始时间
        final long startTime = System.currentTimeMillis();
        String url = postUrl + method;
        try {
            //获取请求参数
            final Map<String, String> requestData = getRequestData(request, StringUtils.defaultIfBlank(subAppId, this.appId));
            log.info("【xft-sdk】接口调用开始 >> url={},method={},request={}", url, method, requestData);
            String xftResponse = HttpUtil.post(url, JSONObject.toJSONString(requestData));
            log.info("【xft-sdk】接口调用结束 >> url={},method={},request={},response={},cost={}", url, method, requestData, xftResponse, System.currentTimeMillis() - startTime);
            return parseResponse(xftResponse, request);
        } catch (Exception ex) {
            log.error("【xft-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", url, method, request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new XftPayException(ex.getMessage());
        }
    }

    /**
     * 结果解析
     *
     * @param result  返回结果
     * @param request 请求参数
     * @param <R>     结果
     * @return UmBaseResponse
     */
    @SuppressWarnings("unchecked")
    private <R> XftBaseResponse<R> parseResponse(String result, XftBizRequest<R> request) {
        JSONObject jsonObject = JSON.parseObject(result);
        //7. 设置结果
        XftBaseResponse response = new XftBaseResponse<>();
        response.setCode(jsonObject.getString("code"));
        response.setMessage(jsonObject.getString("message"));
        response.setResult((Boolean) jsonObject.get("result"));
        response.setData(JSONObject.parseObject(jsonObject.getString("data"), request.getResponseClass()));
        return response;
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R> Map<String, String> getRequestData(XftBizRequest<R> request, String appId) throws XftPayException {
        TreeMap<String, String> parameterMap = SignUtil.getParameterMap(request);
        parameterMap.put("app_id", appId);
        parameterMap.put("sign_type", "MD5");
        String sign = SignUtil.generateSign(parameterMap, key);
        parameterMap.put("sign", sign);
        return parameterMap;
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R> void validateParam(XftBizRequest<R> request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(key)) {
            throw new IllegalArgumentException("签名密钥不能为空");
        }
        if (StringUtils.isBlank(signType)) {
            throw new IllegalArgumentException("签名类型不能为空");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        if (null == XftApiEnum.getByValue(method)) {
            throw new IllegalArgumentException("请求方法不存在");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }
}