/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package com.fshows.xft.sdk.client.impl;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.xft.sdk.client.XftBaseClient;
import com.fshows.xft.sdk.enums.XftApiEnum;
import com.fshows.xft.sdk.exception.XftPayException;
import com.fshows.xft.sdk.request.XftBizRequest;
import com.fshows.xft.sdk.response.XftBaseResponse;
import com.fshows.xft.sdk.util.FsHttpUtil;
import com.fshows.xft.sdk.util.SignUtil;
import com.fshows.xft.sdk.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import java.util.Map;
import java.util.TreeMap;

/**
 * @author zhaoxumin
 * @version XftBaseClientImpl.java, v 0.1 2023-08-25 下午11:06 zhaoxumin
 */
@Slf4j
public class XftBaseClientImpl implements XftBaseClient {

    private static final String SUCCESS = "SUCCESS";

    /**
     * 应用appId;
     */
    private String appId;
    /**
     * 私钥
     */
    private String fbPrivateKey;
    /**
     * 公钥
     */
    private String xftPublicKey;
    /**
     * 域名前缀
     */
    private String prefixUrl;

    private static SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }


    public XftBaseClientImpl(String appId, String fbPrivateKey, String xftPublicKey, String prefixUrl) {
        this.appId = appId;
        this.fbPrivateKey = fbPrivateKey;
        this.xftPublicKey = xftPublicKey;
        this.prefixUrl = prefixUrl;
    }

    /**
     * 请求星富通接口
     *
     * @param request          请求参数
     * @param xftApiDefinition 方法枚举
     * @param url
     * @return UmBaseResponse
     * <p>
     * XftApiDefinition api方法请参考下列API枚举
     */
    @Override
    public <R> XftBaseResponse<R> execute(XftBizRequest<R> request, XftApiDefinition xftApiDefinition, String url) throws XftPayException {
        return execute(request, xftApiDefinition, url, this.appId);
    }

    /**
     * 请求星富通接口
     *
     * @param request    请求参数
     * @param definition 方法枚举
     * @param <R>        返参
     * @return UmBaseResponse
     */
    @Override
    public <R> XftBaseResponse<R> execute(XftBizRequest<R> request, XftApiDefinition definition, String postUrl, String subAppId) throws XftPayException {
        String method = definition.getMethod();
        //验证参数
        validateParam(request, method);
        //获取开始时间
        final long startTime = System.currentTimeMillis();
        try {
            //获取请求参数
            final Map<String, String> requestData = getRequestData(request, StringUtils.defaultIfBlank(subAppId, this.appId));
            log.info("【xft-sdk】接口调用开始 >> url={},method={},request={}", prefixUrl, definition.getMethod(), requestData);
            String xftResponse = FsHttpUtil.post("" + method, requestData);
            log.info("【xft-sdk】接口调用结束 >> url={},method={},request={},response={},cost={}", prefixUrl, definition.getMethod(), requestData, xftResponse, System.currentTimeMillis() - startTime);
            return parseResponse(xftResponse, request);
        } catch (Exception ex) {
            log.error("【xft-sdk】接口调用失败 >> url={},method={},request={},ex={},cost={}", prefixUrl, definition.getMethod(), request, ExceptionUtils.getStackTrace(ex), System.currentTimeMillis() - startTime);
            throw new XftPayException(ex.getMessage());
        }
    }

    /**
     * 结果解析
     *
     * @param result  返回结果
     * @param request 请求参数
     * @param <R>     结果
     * @return UmBaseResponse
     */
    @SuppressWarnings("unchecked")
    private <R> XftBaseResponse<R> parseResponse(String result, XftBizRequest<R> request) {
        JSONObject jsonObject = JSON.parseObject(result);
        //7. 设置结果
        XftBaseResponse response = new XftBaseResponse<>();
        response.setCode(jsonObject.getString("code"));
        response.setMessage(jsonObject.getString("message"));
        response.setResult((Boolean)jsonObject.get("result"));
        response.setData(JSONObject.parseObject(jsonObject.getString("data"), request.getResponseClass()));
        return response;
    }

    /**
     * 获取签名参数
     *
     * @param request 请求参数
     * @param <R>     返参
     * @return Map<String, String>
     */
    private <R> Map<String, String> getRequestData(XftBizRequest<R> request, String appId) throws XftPayException {
        TreeMap<String, String> parameterMap = SignUtil.getParameterMap(request);
        parameterMap.put("app_id", appId);
        String sign = SignUtil.generateSign(parameterMap);
        parameterMap.put("sign", sign);
        parameterMap.put("sign_type", "MD5");
        return parameterMap;
    }

    /**
     * 参数校验
     *
     * @param request 请求参数
     * @param method  请求方法
     * @param <R>
     */
    private <R> void validateParam(XftBizRequest<R> request, String method) {
        if (request == null) {
            throw new IllegalArgumentException("接口请求参数不能为空");
        }
        if (StringUtils.isBlank(fbPrivateKey)) {
            throw new IllegalArgumentException("私钥不能为空");
        }
        if (StringUtils.isBlank(xftPublicKey)) {
            throw new IllegalArgumentException("公钥不能为空");
        }
        if (StringUtils.isBlank(method)) {
            throw new IllegalArgumentException("请求方法不能为空");
        }
        if (null == XftApiEnum.getByValue(method)) {
            throw new IllegalArgumentException("请求方法不存在");
        }
        //注解验证
        ValidateUtil.validateWithThrow(request);
    }
}