/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.swift.client.impl.signer;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import com.fshows.swift.client.base.ApiClientConfig;
import com.fshows.swift.client.base.ISigner;
import com.fshows.swift.client.base.SignParam;
import com.fshows.swift.client.base.VerifySignParam;
import com.fshows.swift.exception.SwiftApiException;
import com.fshows.swift.request.base.SwiftBizRequest;
import com.fshows.swift.util.LogUtil;
import com.fshows.swift.util.RequestParamUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;

/**
 * rsa数据加签（基于对于Map<String, Object> requestMap的加签验签实现）
 * （如果参与加签验签的属性只有一层，没有多层嵌套的结果，可直接使用该实现）
 *
 * @author liluqing
 * @version RsaSignerImpl.java, v 0.1 2023-12-12 18:53
 */
@Slf4j
public class RsaSignHandlerImpl implements ISigner {

    public static String getWaitSignStr(Map<String, String> dataMap, ApiClientConfig apiClientConfig) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry : dataMap.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (StringUtils.equals(key, "sign")
                    || StringUtils.startsWith(key, "reserved")) {
                continue;
            }
            if (StrUtil.isBlank(value)) {
                continue;
            }
            if (sb.length() > 0) {
                sb.append("&");
            }
            sb.append(key).append("=").append(value);
        }
        return sb.toString();
    }

    @Override
    public String sign(SignParam signParam, ApiClientConfig apiClientConfig) throws SwiftApiException {
        String waitSignStr = null;
        try {
            SwiftBizRequest request = signParam.getRequest();
            // 获取待加签字符串
            Map<String, String> data = RequestParamUtils.toMap(request, apiClientConfig.isHump());
            // 获取验签公钥
            String privateKey = apiClientConfig.getFubeiPrivateKey();
            // 获取待加签字符串
            waitSignStr = getWaitSignStr(data, apiClientConfig);
            // 创建加签对象
            Sign sign = new Sign(
                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(privateKey),
                    null);
            // 执行加签操作
            byte[] signed = sign.sign(waitSignStr.getBytes(apiClientConfig.getCharset()));
            return Base64.encode(signed);
        } catch (Exception e) {
            LogUtil.error(log, "【swiftpass-sdk】请求加签失败 >> apiClientConfig={}, waitSignStr={}", e, apiClientConfig, waitSignStr);
            throw new SwiftApiException("swiftpass请求加签失败", e);
        }
    }

    @Override
    public Boolean verifySign(Map<String, String> resMap, VerifySignParam verifySignParam, ApiClientConfig apiClientConfig) throws SwiftApiException {
        return true;
//        try {
//            // 响应签名
//            String resSign = resMap.get("sign");
//            // 获取待加签字符串
//            String waitSignStr = getWaitSignStr(resMap, apiClientConfig).replace(" ", "");
//            // 创建加签对象
//            Sign sign = new Sign(
//                    apiClientConfig.getSignTypeEnum().getAlgorithm(),
//                    null,
//                    SecureUtil.decode(apiClientConfig.getPayCompanyPublicKey()));
//            // 执行加签操作
//            return sign.verify(waitSignStr.getBytes(apiClientConfig.getCharset()), Base64.decode(resSign));
//        } catch (Exception e) {
//            LogUtil.error(log, "【swiftpass-sdk】响应结果验签失败 >> signParam={}", e, verifySignParam);
//            throw new SwiftApiException("swiftpass响应结果验签失败", e);
//        }
    }
}