/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.swift.client.impl;

import com.fshows.swift.client.base.ApiClientConfig;
import com.fshows.swift.client.base.IHttpRequest;
import com.fshows.swift.client.base.ISwiftApiClient;
import com.fshows.swift.client.base.ISwiftApiDefinition;
import com.fshows.swift.client.base.ISigner;
import com.fshows.swift.client.base.SwiftHttpResult;
import com.fshows.swift.client.base.SignParam;
import com.fshows.swift.client.impl.http.DefaultHttpRequestImpl;
import com.fshows.swift.client.impl.signer.DefaultSignerImpl;
import com.fshows.swift.exception.SwiftApiException;
import com.fshows.swift.request.base.SwiftBizRequest;
import com.fshows.swift.response.base.SwiftBizResponse;
import com.fshows.swift.util.LogUtil;
import com.fshows.swift.util.ValidateUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;

/**
 * 默认威富通api客户端实现
 *
 * @author liluqing
 * @version AbstractSwiftApiClientImpl.java, v 0.1 2022-03-02 11:59
 */
@Setter
@Slf4j
public abstract class AbstractSwiftApiClientImpl implements ISwiftApiClient {

    /**
     * 客户端配置
     */
    protected ApiClientConfig apiClientConfig;

    /**
     * http请求器
     */
    protected IHttpRequest httpRequest;

    /**
     * 加签
     */
    protected ISigner signer;

    /**
     * 日志名称
     */
    protected String logName = getLogName();

    public AbstractSwiftApiClientImpl(ApiClientConfig apiClientConfig) throws SwiftApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = new DefaultHttpRequestImpl();
        this.signer = new DefaultSignerImpl();
    }

    public AbstractSwiftApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws SwiftApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = httpRequest;
        this.signer = signer;
    }

    /**
     * 执行威富通请求
     *
     * @param bizRequest
     * @param tradeApiDefinitionEnum
     * @return
     * @throws SwiftApiException
     */
    protected SwiftBizResponse doExecute(SwiftBizRequest bizRequest, ISwiftApiDefinition tradeApiDefinitionEnum, String subAgentId) throws SwiftApiException {
        try {
            // 参数校验
            checkParam(bizRequest, tradeApiDefinitionEnum);
            // 补充全局参数
            buildSwiftBizRequest(bizRequest, tradeApiDefinitionEnum, subAgentId);
            // 构建签名参数
            SignParam signParam = buildSignParam(bizRequest);
            // 获取签名
            String sign = signer.sign(signParam, this.apiClientConfig);
            // 设置签名
            bizRequest.setSign(sign);
            // 序列化请求参数
            String body = serializableRequest(bizRequest, sign, tradeApiDefinitionEnum);
            // 执行加签操作获取签名
            String url = getServerURL(tradeApiDefinitionEnum);
            // 执行HTTP post请求
            SwiftHttpResult httpResult = requestPost(url, sign, body);
            // 反序列化响应结果
            SwiftBizResponse response = parseResponse(httpResult.getBody(), tradeApiDefinitionEnum);
            LogUtil.info(log, "【{}】响应结果映射结束 >> url={}, request={}, response={}", logName, url, bizRequest, response);
            // 根据是否验签配置判断是否需要验签
            // 只有响应成功才进行验签处理，因为威富通在响应失败时不会返回签名
            if (this.apiClientConfig.isVrifySignResponse() && response != null
                    && StringUtils.equals(response.getStatus(), "0")) {
                boolean bo = signer.verifySign(response.getResMap(), null, this.apiClientConfig);
                if (!bo) {
                    LogUtil.error(log, "【{}】响应结果验签失败 >> httpResult={}", logName, httpResult);
                    throw new SwiftApiException("响应结果验签失败");
                }
            }
            return response;
        } catch (SwiftApiException e) {
            LogUtil.error(log, "【{}】威富通请求异常 >> tradeApiDefinition={}, bizRequest={}", e, logName, tradeApiDefinitionEnum, bizRequest);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【{}】威富通请求异常 >> tradeApiDefinition={}, bizRequest={}", e, logName, tradeApiDefinitionEnum, bizRequest);
            throw new SwiftApiException(e.getMessage(), e);
        }
    }

    /**
     * 执行post请求
     *
     * @param url
     * @param authorization
     * @param requestBody
     * @return
     */
    protected SwiftHttpResult requestPost(String url, String authorization, String requestBody) throws IOException, SwiftApiException {
        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.info(log, "【{}】 请求开始 >> url={}, authorization={}, request={}", logName, url, authorization, requestBody);
            SwiftHttpResult httpResult = httpRequest.post(url, authorization, requestBody, apiClientConfig);
            if (httpResult == null) {
                throw new SwiftApiException("swiftpass响应结果为空");
            }
            LogUtil.info(log, "【{}】 请求结束 >> url={}, request={}, response={}, cost={}ms",  logName, url, requestBody,
                    httpResult.getBody(), System.currentTimeMillis() - beginTime);
            return httpResult;
        } catch (SwiftApiException e) {
            LogUtil.error(log, "【{}】 网络请求异常 >> url={}, request={}, cost={}ms",
                    e, logName, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【{}】 网络请求异常 >> url={}, request={}, cost={}ms",
                    e, logName, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw new SwiftApiException("请求swiftpass接口异常", e);
        }
    }

    /**
     * 请求参数校验
     *
     * @param bizRequest
     * @return
     */
    protected void checkParam(SwiftBizRequest bizRequest, ISwiftApiDefinition swiftApiDefinition) throws SwiftApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        System.out.println();
        boolean checkRequestClass = StringUtils.equals(bizRequest.getClass().getCanonicalName(), swiftApiDefinition.getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new SwiftApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.apiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    /**
     * 获取请求地址
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected String getServerURL(ISwiftApiDefinition tradeApiDefinitionEnum) {
        return apiClientConfig.getApiParentURL();
    }


    /**
     * 构建完整的请求参数
     *
     * @param request 业务请求参数
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract SwiftBizRequest buildSwiftBizRequest(
            SwiftBizRequest request,
            ISwiftApiDefinition tradeApiDefinitionEnum,
            String subAgentId);

    /**
     * 构建加签的请求参数
     *
     * @param bizRequest 实际请求的完整参数
     * @return
     */
    protected abstract SignParam buildSignParam(
            SwiftBizRequest bizRequest);

    /**
     * 序列化请求参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract String serializableRequest(
            SwiftBizRequest baseRequest,
            String sign,
            ISwiftApiDefinition tradeApiDefinitionEnum);

    /**
     * 解析响应参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract SwiftBizResponse parseResponse(
            String body,
            ISwiftApiDefinition tradeApiDefinitionEnum) throws SwiftApiException;

    /**
     * 设置日志前缀
     */
    protected abstract String getLogName();
}