/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.swift.client.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.RandomUtil;
import com.fshows.swift.client.base.ApiClientConfig;
import com.fshows.swift.client.base.IHttpRequest;
import com.fshows.swift.client.base.ISigner;
import com.fshows.swift.client.base.ISwiftApiDefinition;
import com.fshows.swift.client.base.SignParam;
import com.fshows.swift.client.impl.signer.RsaSignHandlerImpl;
import com.fshows.swift.constant.CommonConstant;
import com.fshows.swift.constant.RefundConstant;
import com.fshows.swift.enums.api.CibApiDefinitionEnum;
import com.fshows.swift.enums.api.SwiftApiDefinitionEnum;
import com.fshows.swift.exception.SwiftApiException;
import com.fshows.swift.request.base.SwiftBizRequest;
import com.fshows.swift.response.base.SwiftBizResponse;
import com.fshows.swift.util.LogUtil;
import com.fshows.swift.util.RequestParamUtils;
import lombok.extern.slf4j.Slf4j;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * 兴业银行交易接口客户端
 *
 * @author liluqing
 * @version CibApiClientImpl.java, v 0.1 2022-03-04 15:02
 */
@Slf4j
public class CibApiClientImpl extends AbstractSwiftApiClientImpl {

    private static final List<String> REFUND_QUERY_FIELD_LIST = Arrays.asList(RefundConstant.OUT_REFUND_NO,RefundConstant.REFUND_ID,
            RefundConstant.OUT_REFUND_ID,RefundConstant.REFUND_CHANNEL,RefundConstant.REFUND_FEE,RefundConstant.OUT_REFUND_ID,
            RefundConstant.COUPON_REFUND_FEE,RefundConstant.REFUND_TIME,RefundConstant.REFUND_STATUS,RefundConstant.REFUND_STATUS_INFO);

    public CibApiClientImpl(ApiClientConfig apiClientConfig) throws SwiftApiException {
        super(apiClientConfig);
        this.signer = new RsaSignHandlerImpl();
    }

    public CibApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws SwiftApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends SwiftBizResponse, R extends ISwiftApiDefinition> T execute(SwiftBizRequest<T, R> request, R cibApiDefinitionEnum) throws SwiftApiException {
        return (T) super.doExecute(request, cibApiDefinitionEnum);
    }

    /**
     * 添加公共参数构建完整的请求
     *
     * @param bizRequest 业务请求参数
     * @param cibApiDefinitionEnum
     * @return
     */
    @Override
    protected SwiftBizRequest buildSwiftBizRequest(SwiftBizRequest bizRequest, ISwiftApiDefinition cibApiDefinitionEnum) {
        // 补充参数
        bizRequest.setSignAgentno(apiClientConfig.getAppId());
        bizRequest.setService(cibApiDefinitionEnum.getApiSubURI());
        bizRequest.setCharset(CommonConstant.UTF_8_UP);
        bizRequest.setVersion(cibApiDefinitionEnum.getVersion());
        bizRequest.setNonceStr(IdUtil.simpleUUID());
        bizRequest.setSignType(this.apiClientConfig.getSignTypeEnum().getValue());
        return bizRequest;
    }

    @Override
    protected SignParam buildSignParam(SwiftBizRequest baseRequest) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setNonceStr(RandomUtil.randomString(12));
        signParam.setRequest(baseRequest);
        return signParam;
    }

    @Override
    protected String serializableRequest(SwiftBizRequest baseRequest, String sign, ISwiftApiDefinition tradeApiDefinitionEnum) {
        Map<String, String> requstMap = RequestParamUtils.toMap(baseRequest);
        return RequestParamUtils.toXml(requstMap);
    }

    @Override
    protected SwiftBizResponse parseResponse(String resBody, ISwiftApiDefinition tradeApiDefinitionEnum) throws SwiftApiException {
        try {
            String xml = "<?xml version=\"1.0\" encoding=\"GBK\" standalone=\"yes\"?>" + resBody;
            Map<String, String> resMap = RequestParamUtils.getMapFromXML(xml);
            // 退款查询特殊处理
            if (CibApiDefinitionEnum.REFUND_QUERY.equals(tradeApiDefinitionEnum)) {
                Map<String ,String> refundMap = new HashMap<>();
                refundMap.putAll(resMap);
                dealRefundQueryMap(refundMap);
                SwiftBizResponse swiftBizResponse = RequestParamUtils.getObjectFromMap(refundMap, tradeApiDefinitionEnum.getResponseClass());
                swiftBizResponse.setResMap(resMap);
                return swiftBizResponse;
            }
            SwiftBizResponse swiftBizResponse = RequestParamUtils.getObjectFromMap(resMap, tradeApiDefinitionEnum.getResponseClass());
            swiftBizResponse.setResMap(resMap);
            return swiftBizResponse;
        } catch (Exception e) {
            LogUtil.error(log, "【{}】响应结果反序列化异常 >> resBody={}", e, logName, resBody);
            throw new SwiftApiException("响应结果反序列化异常");
        }
    }

    /**
     * 设置日志前缀
     */
    @Override
    protected String getLogName() {
        return "swiftpass-cib-sdk";
    }

    /**
     * 退款参数特殊处理
     * @param map
     * @return {@link Map}<{@link String} ,{@link String}>
     */
    private Map<String ,String> dealRefundQueryMap(Map<String, String> map) {

        if (MapUtil.isEmpty(map)) {
            return map;
        }
        Map<String ,String> addMap = new HashMap<>();
        Iterator<Map.Entry<String, String>> iterator = map.entrySet().iterator();

        while (iterator.hasNext()) {
            Map.Entry<String, String> entry = iterator.next();
            String value = entry.getValue();
            String key = entry.getKey();
            // 循环匹配字段
            for (String filed : REFUND_QUERY_FIELD_LIST) {
                // 过滤字段 例: refund_time_0, refund_time_10 、 为 refund_time、
                if (key.startsWith(filed) && ((key.length() == (filed.length() + 2)) || (key.length() == (filed.length() + 3)))) {
                    // 添加过滤后字段
                    addMap.put(filed,value);
                }
            }
        }
        if (MapUtil.isNotEmpty(addMap)) {
            map.putAll(addMap);
        }
        return map;
    }
}