/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.remit.agent;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.Sign;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.com.shande.openapi.sdk.util.CryptoUtil;
import com.fshows.remit.agent.common.RemitClientConfig;
import com.fshows.remit.agent.common.ShandeAgentRemitApiEnum;
import com.fshows.remit.agent.request.ShandeBaseRequest;
import com.fshows.remit.agent.request.ShandeBizRequest;
import com.fshows.remit.agent.response.ShandeBaseResponse;
import com.fshows.remit.agent.response.ShandeBizResponse;
import com.fshows.remit.agent.util.ShandeAgentRemitUtil;
import com.fshows.shande.sdk.common.ShandeApiEnum;
import com.fshows.shande.sdk.common.ShandeException;
import com.fshows.shande.sdk.request.ShandeRequest;
import com.fshows.shande.sdk.response.ShandeResponseBody;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.net.URLDecoder;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.HashMap;
import java.util.Map;

/**
 *
 * @author liluqing
 * @version ShandeAgentRemitClient.java, v 0.1 2023-09-20 14:51
 */
@Slf4j
public class ShandeAgentRemitClient {

    private static final String DEFAULT_ENCODING = "UTF-8";

    /**
     * 客户端配置
     */
    private RemitClientConfig remitClientConfig;

    public ShandeAgentRemitClient(RemitClientConfig remitClientConfig) {
        this.remitClientConfig = remitClientConfig;
        remitClientConfig.init();
    }


    /**
     * 请求衫德接口
     *
     * @param apiEnum ShandeApiEnum
     * @param bizRequest ShandeRequest
     * @param <T>     <T>
     * @return ShandeResponseBody
     */
    public <T extends ShandeBizResponse> ShandeBaseResponse<T> request(ShandeAgentRemitApiEnum apiEnum,
                                                                       ShandeBizRequest<T> bizRequest) throws ShandeException {
        Map<String, Object> fromParam = null;
        String url = null;
        String result = null;
        try {
            // 请求对象初始化
            ShandeBaseRequest httpReq = bulidRealRequest(bizRequest, apiEnum);
            // 请求地址
            url = remitClientConfig.getBaseUrl() + apiEnum.getValue();
            // 请求参数
            fromParam = object2Map(httpReq);

            // 发送请求
            result = HttpRequest.post(url)
                    .header("Content-Type", "application/x-www-form-urlencoded;charset=UTF-8")
                    .form(fromParam)
                    .timeout(remitClientConfig.getTimeout())
                    .execute().body();
            log.info("[shande-sdk] >> 请求杉德接口结束密文 >> url={}, request={},       response={}", url, JSONObject.toJSONString(fromParam), result);
            if (StrUtil.isBlank(result)) {
                throw ShandeException.SERVER_EXCEPTION.newInstance("杉德返回空结果");
            }
            // 解析响应
            result = URLDecoder.decode(result, DEFAULT_ENCODING);
            Map<String, String> responseMap = ShandeAgentRemitUtil.convertResultStringToMap(result);
            ShandeBaseResponse<T> baseResponse = map2Object(responseMap, ShandeBaseResponse.class);
            byte[] respDataBytes = handleResult(baseResponse.getEncryptData(), baseResponse.getEncryptKey());
            // 验签
            verifySign(baseResponse.getSign(), respDataBytes);
            // 反序列化业务响应结果
            T bizResult = JSON.parseObject(new String(respDataBytes, DEFAULT_ENCODING), bizRequest.getResponseClass());
            baseResponse.setData(bizResult);

            fromParam.put("data", bizRequest);
            fromParam.remove("encryptKey");
            fromParam.remove("encryptData");
            log.info("[shande-sdk] >> 请求杉德接口结束明文 >> url={},  param={},       result={}", url, JSONObject.toJSONString(fromParam), JSONObject.toJSONString(baseResponse));
            return baseResponse;
        } catch (ShandeException e) {
            log.error(StrUtil.format("[shande-sdk] >> 请求衫德接口异常明文 >> url={}, bizParam={},  request={},       result={}", url, JSONObject.toJSONString(bizRequest), JSONObject.toJSONString(fromParam), result), e);
            throw e;
        } catch (Exception e) {
            log.error(StrUtil.format("[shande-sdk] >> 请求处理异常明文 >> url={}, bizParam={},  request={},       result={}", url, JSONObject.toJSONString(bizRequest), JSONObject.toJSONString(fromParam), result), e);
            throw ShandeException.SERVER_EXCEPTION.newInstance(e.getMessage(), e);
        }
    }

    /**
     * 验证签名
     *
     * @param retSign
     * @param respDataBytes
     */
    private void verifySign(String retSign, byte[] respDataBytes) {
        try {
            byte[] signBytes = Base64.decodeBase64(retSign
                    .getBytes(DEFAULT_ENCODING));
            boolean isValid = CryptoUtil.verifyDigitalSign(respDataBytes, signBytes,
                    remitClientConfig.getPublicKey(), "SHA1WithRSA");
            if (!isValid) {
                throw ShandeException.SERVER_EXCEPTION.newInstance("响应验签不通过");
            }
        } catch (ShandeException e) {
            throw e;
        } catch (Exception e) {
            throw ShandeException.SERVER_EXCEPTION.newInstance("响应验签异常", e);
        }
    }

    /**
     * 处理请求结果
     *
     * @param encryptData
     * @param encryptKey
     * @return
     * @param <T>
     */
    private byte[] handleResult(String encryptData, String encryptKey) {
        try {
            // 解密加密KEY
            byte[] decodeBase64KeyBytes = Base64.decodeBase64(encryptKey
                    .getBytes(DEFAULT_ENCODING));
            byte[] merchantAESKeyBytes = CryptoUtil.RSADecrypt(
                    decodeBase64KeyBytes, remitClientConfig.getPrivateKey(), 2048, 11,
                    "RSA/ECB/PKCS1Padding");

            // 使用key和aes算法解密响应报文
            byte[] decodeBase64DataBytes = Base64.decodeBase64(encryptData
                    .getBytes(DEFAULT_ENCODING));

            byte[] respDataBytes = CryptoUtil.AESDecrypt(decodeBase64DataBytes,
                    merchantAESKeyBytes, "AES", "AES/ECB/PKCS5Padding", null);



            return respDataBytes;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * map转对象
     *
     */
    private <T> T map2Object(Map<String, String> map, Class<T> clazz) {
        if (map == null || clazz == null) {
            return null;
        }
        try {
            Field[] fieldList = clazz.getDeclaredFields();
            Constructor<T> constructor = clazz.getDeclaredConstructor();
            constructor.setAccessible(true);
            T obj = constructor.newInstance();
            for (Field field: fieldList) {
                field.setAccessible(true);
                String value = map.get(field.getName());
                if (value != null) {
                    field.set(obj, convert(field.getType(), value));
                }
            }
            return obj;
        } catch (Exception e) {
            log.error("参数转换异常", e);
            throw ShandeException.SERVER_EXCEPTION.newInstance("返回参数参数转换失败", e);
        }
    }

    /**
     *
     * @param type
     * @param value
     * @return
     */
    private Object convert(Class<?> type, String value) {
        if (type == String.class) {
            return value;
        }
        if (value == null) {
            return null;
        }
        if (type == int.class || type == Integer.class) {
            return Integer.parseInt(value);
        } else if (type == long.class || type == Long.class) {
            return Long.parseLong(value);
        } else if (type == double.class || type == Double.class) {
            return Double.parseDouble(value);
        } else if (type == float.class || type == Float.class) {
            return Float.parseFloat(value);
        } else if (type == boolean.class || type == Boolean.class) {
            return Boolean.parseBoolean(value);
        } else if (type == BigDecimal.class) {
            return new BigDecimal(value);
        } else {
            return null;
        }
    }

    /**
     * 对象转map
     *
     * @param obj
     * @return
     */
    private Map<String, Object> object2Map(Object obj) {
        String[] ignoreFields = new String[]{"serialVersionUID"};
        Map<String, Object> map = new HashMap<>();
        if (obj == null) {
            return map;
        }
        Class<?> clazz = obj.getClass();
        for (Field field : clazz.getDeclaredFields()) {
            if (ArrayUtil.contains(ignoreFields, field.getName())) {
                continue;
            }
            field.setAccessible(true);
            try {
                Object value = field.get(obj);
                map.put(field.getName(), ObjectUtil.defaultIfNull(value, "").toString());
            } catch (IllegalAccessException e) {
                throw ShandeException.SERVER_EXCEPTION.newInstance("请求入参参数转换失败", e);
            }
        }
        return map;
    }


    /**
     *
     * @param request
     * @param apiEnum
     * @return
     * @throws Exception
     */
    private ShandeBaseRequest bulidRealRequest(ShandeBizRequest request, ShandeAgentRemitApiEnum apiEnum) throws Exception {
        ShandeBaseRequest baseRequest = new ShandeBaseRequest();
        baseRequest.setTransCode(apiEnum.getTransCode());
        baseRequest.setAccessType(remitClientConfig.getAccessType());
        baseRequest.setMerId(remitClientConfig.getMid());
        baseRequest.setPlId(remitClientConfig.getPlId());
        baseRequest.setExtend("");

        // 请求数据加密
        String aesKey = RandomUtil.randomString(16);
        // 业务参数json序列化
        String reqData = JSON.toJSONString(request);
        byte[] aesKeyBytes = aesKey.getBytes(remitClientConfig.getCharset());
        byte[] plainBytes = reqData.getBytes(DEFAULT_ENCODING);
        String encryptData = new String(Base64.encodeBase64(
                CryptoUtil.AESEncrypt(plainBytes, aesKeyBytes, "AES",
                        "AES/ECB/PKCS5Padding", null)),
                DEFAULT_ENCODING);
        baseRequest.setEncryptData(encryptData);

        // 加密AES加密KEY
        String encryptKey = new String(Base64.encodeBase64(
                CryptoUtil.RSAEncrypt(aesKeyBytes, remitClientConfig.getPublicKey(), 2048, 11,
                        "RSA/ECB/PKCS1Padding")), DEFAULT_ENCODING);
        baseRequest.setEncryptKey(encryptKey);

        // 加签
        String sign = doSign(reqData);
        baseRequest.setSign(sign);
        return baseRequest;
    }

    /**
     * 执行方法加签
     */
    public String doSign(String waitSignStr) {
        try {
            log.info("【fuiou-sdk】待加签字符串 >> waitSignStr={}", waitSignStr);
            // 创建加签对象
            Sign sign = new Sign(
                    remitClientConfig.getSignTypeEnum().getAlgorithm(),
                    SecureUtil.decode(remitClientConfig.getFubeiPrivateKey()),
                    null);
            // 执行加签操作
            byte[] signed = sign.sign(waitSignStr.getBytes(remitClientConfig.getCharset()));
            return cn.hutool.core.codec.Base64.encode(signed);
        } catch (Exception e) {
            log.error("【fuiou-sdk】fuiou请求加签失败 >> signParam=" + waitSignStr, e );
            throw ShandeException.SERVER_EXCEPTION.newInstance("请求加签失败", e);
        }
    }


}