/**
 * fshows.com
 * Copyright (C) 2013-2020 All Rights Reserved.
 */
package com.fshows.shande.sdk;

import cn.hutool.core.util.StrUtil;
import cn.hutool.system.SystemUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.parser.ParserConfig;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.com.shande.openapi.sdk.client.OpenParameters;
import com.fshows.com.shande.openapi.sdk.client.ShandeOpenApiClient;
import com.fshows.com.shande.openapi.sdk.config.Configuration;
import com.fshows.com.shande.openapi.sdk.config.FileConfiguration;
import com.fshows.com.shande.openapi.sdk.constant.RequestConstants;
import com.fshows.shande.sdk.common.ShandeApiEnum;
import com.fshows.shande.sdk.common.ShandeException;
import com.fshows.shande.sdk.common.ValidateUtil;
import com.fshows.shande.sdk.request.FileDownloadRequest;
import com.fshows.shande.sdk.request.ShandeRequest;
import com.fshows.shande.sdk.request.item.file.BizData;
import com.fshows.shande.sdk.response.ShandeResponseBody;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;

/**
 * 衫德请求客户端
 *
 * @author linzy
 * @version ShandeRequest.java, v 0.1 2022-12-02 13:42 linzy
 * @date 2022/12/02
 */
@Slf4j
public class ShandeRequestClient extends ShandeOpenApiClient {

    /**
     * 请求成功 code
     */
    public static final String REQUEST_SUCCESS_CODE = "0000";
    private static final String FILE_SEPARATOR = SystemUtil.getOsInfo().getFileSeparator();

    private static SerializeConfig SNAKE_CASE_CONFIG = new SerializeConfig();
    private static ParserConfig CAMEL_CASE_CONFIG = new ParserConfig();

    public ShandeRequestClient(Configuration configuration) {
        super(configuration);
    }

    static {
        SNAKE_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }

    static {
        CAMEL_CASE_CONFIG.propertyNamingStrategy = PropertyNamingStrategy.CamelCase;
    }

    /**
     * 请求衫德接口
     *
     * @param apiEnum ShandeApiEnum
     * @param request ShandeRequest
     * @param <T>     <T>
     * @return ShandeResponseBody
     */
    public <T> ShandeResponseBody<T> request(ShandeApiEnum apiEnum,
                                             ShandeRequest request) throws ShandeException {
        long startTime = System.currentTimeMillis();
        // 2. 参数校验
        ValidateUtil.validateWithThrow(request);
        //3. 拼接请求参数
        final OpenParameters context = new OpenParameters.Builder()
                .apiName(apiEnum.getValue())
                .param(JSON.toJSONString(request, SNAKE_CASE_CONFIG))
                .build();
        try {
            // 4. 发布请求
            final String result = send(context);
            JSONObject jsonResult = JSON.parseObject(result);
            log.info("request >> 衫德请求服务出入参 >> url = {}, request = {}, result = {}, 耗时={}ms",
                    this.getConfiguration().remoteAddress() + apiEnum.getValue(), context.getParams(), result
                    , System.currentTimeMillis() - startTime);
            if (!StrUtil.equalsIgnoreCase(jsonResult.getString(RequestConstants.ResponseBody.KEY_RESPONSE_CODE), REQUEST_SUCCESS_CODE)) {
                throw new ShandeException(jsonResult.getString(RequestConstants.ResponseBody.KEY_RESPONSE_CODE), jsonResult.getString(RequestConstants.ResponseBody.KEY_RESPONSE_DESC));
            }
            // 5. 转化返回值
            ShandeResponseBody<T> response = new ShandeResponseBody<>();
            response.setData(JSON.parseObject(jsonResult.getString(RequestConstants.RequestBody.KEY_DATA), apiEnum.getResponseClass(), CAMEL_CASE_CONFIG));
            response.setResponseCode(jsonResult.getString(RequestConstants.ResponseBody.KEY_RESPONSE_CODE));
            response.setResponseDesc(jsonResult.getString(RequestConstants.ResponseBody.KEY_RESPONSE_DESC));
            response.setResponseTime(jsonResult.getString(RequestConstants.ResponseBody.KEY_TIMESTAMP));
            response.setCustomerOrderNo(jsonResult.getString(RequestConstants.ResponseBody.KEY_CUSTOMER_ORDER_NO));
            return response;
        } catch (ShandeException e) {
            log.error("request >> 衫德请求服务业务异常 >> apiEnum = {}, request = {}, exception = {}",
                    apiEnum.getValue(), request, ExceptionUtils.getStackTrace(e));
            throw e;
        } catch (Exception e) {
            log.error("request >> 衫德请求服务异常 >> apiEnum = {}, request = {}, exception = {}",
                    apiEnum.getValue(), request, ExceptionUtils.getStackTrace(e));
            throw ShandeException.SERVER_EXCEPTION;
        }
    }


    /**
     * 下载文件
     *
     * @param request 请求
     * @return {@link String}
     */
    public String downloadFile(FileDownloadRequest request) {
        ShandeApiEnum apiEnum = ShandeApiEnum.CUS_OPEN;

        log.info("request >> 衫德请求开始 >> apiEnum = {}, request = {}",
                apiEnum, request);

        // 1. 生成公共参数
        final BizData bizData = request.getBizData();
        bizData.setMerchantNo(this.getConfiguration().mid());

        // 2. 参数校验
        ValidateUtil.validateWithThrow(request);

        //3. 拼接请求参数
        final String fileName = request.getFileName();
        final OpenParameters context = new OpenParameters.Builder()
                .apiName(apiEnum.getValue())
                .fileName(fileName)
                .param(JSON.toJSONString(request))
                .build();

        try {
            // 4. 发布请求
            downloadFile(context);
            // 拼接本地地址
            return ((FileConfiguration) this.getConfiguration()).storagePath()
                    + FILE_SEPARATOR
                    + fileName + "." + request.getFileType();
        } catch (Exception e) {
            log.error("request >> 衫德下载文件异常 >> apiEnum = {},  exception = {}, request = {}",
                    apiEnum, ExceptionUtils.getStackTrace(e), request);
            throw ShandeException.SERVER_EXCEPTION;
        }
    }
}