package com.fshows.com.shande.openapi.sdk.util;

import com.fshows.com.shande.openapi.sdk.config.Configuration;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.pkcs.ContentInfo;
import sun.security.pkcs.PKCS7;
import sun.security.pkcs.SignerInfo;
import sun.security.x509.AlgorithmId;
import sun.security.x509.X500Name;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.X509Certificate;

/**
 * RSA加解密、签名验签单元类
 *
 * @author miwenming
 * @date 2020/5/21 11:52
 */
public class RSAUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(RSAUtils.class);

    private static final String SHA1_WITH_RSA = "SHA1WithRSA";

    private static final String SHA1 = "SHA1";

    /**
     * 使用私钥签名
     *
     * @param data       待签名报文
     * @param privateKey 私钥
     */
    public static String signature(String data, String privateKey) throws Exception {
        Signature signature = Signature.getInstance(SHA1_WITH_RSA);
        signature.initSign(CertFileUtils.toPrivateKey(privateKey));
        signature.update(data.getBytes());
        return Base64.encodeBase64String(signature.sign());
    }

    /**
     * 使用公钥验签
     *
     * @param data 待验签报文
     * @param signature 签名结果
     * @param publicKey 公钥
     */
    public static boolean verifySignature(String data, String signature, String publicKey) throws Exception {
        // 取公钥匙对象
        PublicKey pubKey = CertFileUtils.toPublicKey(publicKey);
        Signature sign = Signature.getInstance(SHA1_WITH_RSA);
        sign.initVerify(pubKey);
        sign.update(data.getBytes(StandardCharsets.UTF_8));
        // 验证签名是否正常
        return sign.verify(Base64.decodeBase64(signature));
    }

    /**
     * pkcs7签名,包括了证书信息，如序列号、证书颁发者和持有者
     *
     * @param dataBytes 源数据字节数组
     */
    public static String detachSign(byte[] dataBytes, Configuration configuration) throws Exception {
        Signature signature = Signature.getInstance(SHA1_WITH_RSA);
        signature.initSign(CertFileUtils.toPrivateKey(configuration.fubeiSignPrivateKey()));
        signature.update(dataBytes, 0, dataBytes.length);
        return base64Encode(createPkcs7(configuration, signature.sign()));
    }

    /**
     * 构造PKCS7数据
     *
     * @param configuration 调用配置
     * @param signedDataBytes 签名数据字节
     */
    private static byte[] createPkcs7(Configuration configuration, byte[] signedDataBytes) throws
            NoSuchAlgorithmException, IOException {
        AlgorithmId[] digestAlgorithmIds = new AlgorithmId[]{AlgorithmId.get(SHA1)};
        ContentInfo contentInfo = new ContentInfo(ContentInfo.DATA_OID, null);
        X509Certificate certificate = getSignCert("configuration.pfxFileName()", "configuration.pfxPassword()");
//        X509Certificate certificate = getSignCert(configuration.pfxFileName(), configuration.pfxPassword());

        if (certificate != null) {
            X509Certificate[] certificates = new X509Certificate[]{certificate};
            SignerInfo si = new SignerInfo((X500Name) certificate.getIssuerDN(), certificate.getSerialNumber(),
                    AlgorithmId.get(SHA1), null,
                    new AlgorithmId(AlgorithmId.RSAEncryption_oid), signedDataBytes, null);
            SignerInfo[] signerInfo = new SignerInfo[]{si};
            PKCS7 p7 = new PKCS7(digestAlgorithmIds, contentInfo, certificates, signerInfo);
            try (ByteArrayOutputStream bout = new ByteArrayOutputStream()) {
                p7.encodeSignedData(bout);
                return bout.toByteArray();
            }
        } else {
            throw new IllegalArgumentException("something wrong with cert file, pls check it!");
        }
    }

    /**
     * 获取签名证书
     *
     * @param pfxFileName 证书文件名称
     * @param pfxPassword 证书文件密码
     */
    private static X509Certificate getSignCert(String pfxFileName, String pfxPassword) {
        try {
            KeyStore keyStore = CertFileUtils.getKeyStore(pfxFileName, pfxPassword);
            String keyAlias = CertFileUtils.getKeyAlias(keyStore);
            return (X509Certificate) keyStore.getCertificate(keyAlias);
        } catch (Exception e) {
            LOGGER.error("get sign cert error ", e);
        }

        return null;
    }

    /**
     * base64编码，并进行格式处理
     *
     * @param data 待编码字节数组
     */
    private static String base64Encode(byte[] data) {
        String strRet = Base64.encodeBase64String(data);
        strRet = strRet.replaceAll("\r\n", "");
        strRet = strRet.replaceAll("\n", "");
        strRet = strRet.replaceAll(" ", "");
        strRet = strRet.replaceAll("\t", "");
        return strRet;
    }

    /**
     * 验签 无须使用验签公钥字符串，直接根据签名原文和签名结果进行验签
     *
     * @param originData 签名原文
     * @param signedData 签名结果
     * @return 验签是否通过
     */
    public static boolean detachVerify(byte[] originData, String signedData)
            throws NoSuchAlgorithmException, SignatureException, IOException {
        if (signedData == null) {
            throw new IllegalArgumentException("signedData can't be null");
        } else {
            PKCS7 exp = new PKCS7(Base64.decodeBase64(signedData));
            SignerInfo[] signers;
            if (originData == null) {
                signers = exp.verify();
            } else {
                signers = exp.verify(originData);
            }

            if (signers != null) {
                X509Certificate[] certs = exp.getCertificates();
                return certs[0].getSerialNumber().equals(signers[0].getCertificateSerialNumber());
            } else {
                return false;
            }
        }
    }

    public static void main(String[] args) throws Exception {
        String rsaKey = CertFileUtils.getPrivateKeyFromPfxFile("Merchant_Private_CFCA_sandbox.pfx", "11111111");
        System.out.println(rsaKey);
        /*rsaKey = "MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQCGqhwXTEZH46S9d8jKtg23+G4A4iiIAmNTY+dQXUHW4NeTw49Vga6WpyEdmm0vB608h/oumWc1bFMVl3/83kplP0PEUtEZXwlLzdfxxcPXuHy1rEQGkLKJA1EAIVtWna8KYoOvj8QfrQ2Dm3+02NKOfkLHxn6+JV+nEEU6lWaOsNawjoKHJm1bCjor+UtzOzwBYaBcIJ1Bno3oDP9jcfmAf8VjZ3TXCIcbuntLi2zyReXNj+Uhw3cLJ3tvUSTvDclAArggjyoPdmUS12hBkIQxxWdw4qdLmNEWqK1JeKxyHfLc8P73onM7qWJUQVcFac7J1vLA1UKMktAZe22wt9XNAgMBAAECggEAS3Okqaki98Sk2UP0TSAUL7AScoxFCW/NFzwyruSRr0Prhe9csBwpJmAePef4pRE4D2Jg2xs4qMY5BzALljM2m+QGpEmCvyRwoPiEtgKI6KWFuoT62HLalco16EzIUI+pM7he8E8tJFJh8AuAUmPqFczIvgaJnSBzp7eGEn0DKY87qCLJhiaggrt/zZAgx15dEYI9d9nv/P2YxTdrh6yPRI1sTZnx9EPJ4rg0UP3KZfr0JZOus1kSrAbs6uYSqdHfu1aQLoSI3zuGeGe9A3eJM2uxdWPA1xclDh3NLNKxyJlM2l4LAfM06esor9K/qQ42ofatL8k/mEQrSTfq6ZsMAQKBgQC7e/E49w1fZOqh1b1VkmBRVyX1HhgkVwbfl3QtLklt+AEergIU+pLyYQdSO+ixqntt4F7lUtks5jBliybPbPsq1ppJf2NY/oQCmbl2yE5P0Lw/rIppk3yotyza74M+/b+qTo/DjxVSWVwO0JelTtpy0bGQ4DM7l7lwQWAicLmNXQKBgQC34KFnLAoPkcJTAMoxugr2YNcPkGoh+RxFG0a3qnMCi8CuVBVmAjvQBt8sHsqV/1wT+hOFZHIACq9MbMoWCee33SZQ7DWUzfzxHTcA5or01R81JSGKKiqvh/SMVmPeX26IADFz2eUz3fODSpUsVRU3tyIKtt6Z9vE4fIEbf09zMQKBgAVCyhcOH6rwmwDljYWMMPW3W1JQgN7sLj4x8smdxeO3FNj2iwppqZvxixIMqk/p98KyjBclRwDSiMMgr3sclMOz9+peD7ycj7m+9IxXvJLSyaWDwvKJd/MICNMv61/mIO23fKNBLZj89u9XfRnfZ5PXPINTNzzR2htDSl5NCXflAoGANomLlbw8J64FwvSjA+i4Q6cpYlZlBDqMpfqXGPaXMhQdTBdoWV8kOmkvXDitwZuBxfMQQyotndGNeM9ziOHbNm+DY9ELe101PpZ5r+yxHjjwDqU6s24aRwG+CLm+2T/Y2sGj6UahcqbMCQ7f+8wohOPuVtsyXtwdY7GZ4Xv7JqECgYB0G4B5KGMisM61k78QLq5W6pircyp+Y+zqzjk5COWJyFEomBGSRa4WchwQP8XCvcEt5xnE8Pb9NHsPXUhJ4thqFsI2vLkXi+1upBFFoKyWdjeOdsCc4/ZmKEkwvA9p4mykXd9k23lUt0ibZmrt+Liywy28Fw6+9B8jO2vmWEqkGg==";*/
        String data = "testInfo";
        String signResult = signature(data, rsaKey);
        String publicKey = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAnORAIOgdgIfkV/FUxyzNUF/nPz03YqThw3qdOERgAf7Hpzqc4newR3MXt/FNSefPBG+I/L8LQxJTlk3Rn3EcL/Vq/7Xw2SMMr3Xz/CkfwRsZ9CSN6pI+zKE86xvHkGvLvTYSDmX40XpaUhIyJ7+KTc/Gnzu44ctWHzSAeQaFnu2nPMhDThL2Dcbfcj+qBSoberdcCgreW/ul9HQUykkIva8oHQagI+vK4vlqbLDHaJfvmR+Y3ny+3N4QATULwjBIPn4zBDjgrbQiegskxsUTwtCGVpXivT4ZJdOXpIAm2aELRnqr1OOjMO3s00TmMmdh3NFrXink5cRJaz/GkYM4lQIDAQAB";
        System.out.println(verifySignature(data, signResult, publicKey));
    }

}
