package com.fshows.msfpay.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.msfpay.apienum.MsfpayAutoSignApiDefinitionEnum;
import com.fshows.msfpay.client.model.MsfpayMerchantClientConfigModel;
import com.fshows.msfpay.constant.MsfpayConstant;
import com.fshows.msfpay.request.MsfpayBizRequest;
import com.fshows.msfpay.response.MsfpayBaseResponse;
import com.fshows.msfpay.utils.MsfpayUtil;
import com.fshows.msfpay.utils.SM2Utils;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import msfpay.cfca.sadk.util.Base64;
import msfpay.cfca.sadk.util.CertUtil;
import msfpay.cfca.sadk.x509.certificate.X509Cert;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.CookieStore;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
public class MsfpayAutoSignClient extends AbstractApiClient<MsfpayBizRequest, MsfpayBaseResponse, MsfpayAutoSignApiDefinitionEnum> {
    private PrivateKey merchantSignPrivateKey;
    private PrivateKey merchantEncPrivateKey;
    private X509Cert platformEncCert;
    private X509Cert platformSignCert;
    private X509Cert merchantEncCert;
    private X509Cert merchantSignCert;

    public MsfpayAutoSignClient(MsfpayMerchantClientConfigModel apiClientConfig) throws Exception {
        super(apiClientConfig);
        initCertificatesAndKeys(apiClientConfig);
    }

    private void initCertificatesAndKeys(MsfpayMerchantClientConfigModel config) throws Exception {
        try {
            // 1. 加载商户签名私钥
            merchantSignPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantSignPath(), config.getMerchantSignPwd());

            // 2. 加载商户加密私钥
            merchantEncPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantEncPath(), config.getMerchantEncPwd());

            // 3. 加载平台加密证书
            platformEncCert = MsfpayUtil.getX509Cert(config.getPlatformEncCertPath());

            // 4. 加载平台签名证书
            platformSignCert = MsfpayUtil.getX509Cert(config.getPlatformSignCertPath());

            merchantEncCert = CertUtil.getCertFromSM2(config.getMerchantEncPath());

            merchantSignCert = CertUtil.getCertFromSM2(config.getMerchantSignPath());

            log.info("证书和密钥初始化成功");
        } catch (Exception e) {
            log.error("初始化证书和密钥失败", e);
            throw new Exception("初始化证书和密钥失败", e);
        }
    }

    @Override
    protected DefaultRequestContext buildRequestContext(MsfpayAutoSignApiDefinitionEnum iApiDefinition, MsfpayBizRequest request, DefaultClientConfigModel customConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(iApiDefinition);

        MsfpayMerchantClientConfigModel config = new MsfpayMerchantClientConfigModel();
        MsfpayMerchantClientConfigModel clientConfig = (MsfpayMerchantClientConfigModel) this.apiClientConfig;

        config.setApiParentURL(clientConfig.getApiParentURL());
        config.setCharset(clientConfig.getCharset());
        config.setSignType(clientConfig.getSignType());
        config.setVersion(clientConfig.getVersion());
        config.setPlatmerid(clientConfig.getPlatmerid());
        config.setTimeout(clientConfig.getTimeout());

        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    @Override
    protected ApiRequestModel buildApiRequestModel(MsfpayBizRequest request, DefaultRequestContext context) {
        MsfpayMerchantClientConfigModel configModel = (MsfpayMerchantClientConfigModel) context.getApiClientConfig();
        MsfpayAutoSignApiDefinitionEnum iApiDefinition = (MsfpayAutoSignApiDefinitionEnum) context.getIApiDefinition();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        apiRequestModel.setApiURL(configModel.getApiParentURL());
        apiRequestModel.setContentType(MsfpayConstant.Http.CONTENT_TYPE_FORM);

        try {
            fillCommonHead(request, configModel, iApiDefinition);
            apiRequestModel.setRequest(request);

            Map<String, String> messageData = new HashMap<>();
            Map<String, Object> dataMap = new HashMap<>();
            Map<String, String> headMap = new HashMap<>();
            Map<String, Object> bodyMap = new HashMap<>();

            headMap.put(MsfpayConstant.HeadField.MERTRANDATE, request.getMertrandate());
            headMap.put(MsfpayConstant.HeadField.MERTRANTIME, request.getMertrantime());
            headMap.put(MsfpayConstant.HeadField.PLATMERID, request.getPlatmerid());
            headMap.put(MsfpayConstant.HeadField.TRANCODE, request.getTrancode());
            headMap.put(MsfpayConstant.HeadField.TRANFLOW, request.getTranflow());
            headMap.put(MsfpayConstant.HeadField.ZTPAGE, request.getZtpage());
            headMap.put(MsfpayConstant.HeadField.RESPCODE, "");
            headMap.put(MsfpayConstant.HeadField.RESPMSG, "");


            String requestJson = JSON.toJSONString(request);
            Map<String, Object> requestMap = JSON.parseObject(requestJson, Map.class);
            requestMap.remove(MsfpayConstant.HeadField.PLATMERID);
            requestMap.remove(MsfpayConstant.HeadField.TRANFLOW);
            requestMap.remove(MsfpayConstant.HeadField.TRANCODE);
            requestMap.remove(MsfpayConstant.HeadField.MERTRANDATE);
            requestMap.remove(MsfpayConstant.HeadField.MERTRANTIME);
            requestMap.remove(MsfpayConstant.HeadField.ZTPAGE);
            bodyMap.putAll(requestMap);

            dataMap.put(MsfpayConstant.MessageField.HEAD, headMap);
            dataMap.put(MsfpayConstant.MessageField.BODY, bodyMap);

            String dataJson = JSON.toJSONString(dataMap, SerializerFeature.WriteNullStringAsEmpty);
            String sign = sign(dataJson);

            messageData.put(MsfpayConstant.MessageField.DATA, dataJson);
            messageData.put(MsfpayConstant.MessageField.SIGN, sign);

            Map<String, Object> message = new HashMap<>();
            message.put(MsfpayConstant.MessageField.MESSAGE, messageData);

            apiRequestModel.setParamMap(message);

            String messageJson = JSON.toJSONString(message, SerializerFeature.WriteNullStringAsEmpty);
            byte[] envelopdata = SM2Utils.envelopeMessage(messageJson, MsfpayConstant.Encryption.SM4_ECB_PKCS7, platformEncCert);

            Map<String, String> paramMap = new HashMap<>();
            paramMap.put(MsfpayConstant.RequestField.FORMAT_TYPE, MsfpayConstant.RequestValue.FORMAT_JSON);
            paramMap.put(MsfpayConstant.RequestField.CRYPT_TYPE, MsfpayConstant.RequestValue.CRYPT_SM2);
            paramMap.put(MsfpayConstant.RequestField.SIGN_TYPE, MsfpayConstant.RequestValue.SIGN_SM2);
            paramMap.put(MsfpayConstant.RequestField.DATA, new String(envelopdata));

            apiRequestModel.setRequestForm(paramMap);
        } catch (Exception e) {
            throw new FsApiException("构建请求失败", e);
        }

        return apiRequestModel;
    }

    private void fillCommonHead(MsfpayBizRequest request, MsfpayMerchantClientConfigModel configModel, MsfpayAutoSignApiDefinitionEnum iApiDefinition) {
        if (StringUtils.isEmpty(request.getPlatmerid())) {
            request.setPlatmerid(configModel.getPlatmerid());
        }
        request.setTrancode(iApiDefinition.getTrancode());
        request.setMertrandate(MsfpayUtil.getCurrentDateTime(MsfpayConstant.DateTimeFormat.DATE_FORMAT));
        request.setMertrantime(MsfpayUtil.getCurrentDateTime(MsfpayConstant.DateTimeFormat.TIME_FORMAT));
        request.setZtpage(MsfpayConstant.BooleanString.TRUE);

        if (request.getTranflow() == null || request.getTranflow().isEmpty()) {
            if (StringUtils.isEmpty(request.getPlatmerid())) {
                request.setPlatmerid(MsfpayUtil.generateTranflow(configModel.getPlatmerid()));
            } else {
                request.setTranflow(MsfpayUtil.generateTranflow(request.getPlatmerid()));
            }
        }
    }

    private String sign(String content) {
        try {
            return SM2Utils.P7SignMessageDetach(MsfpayConstant.Encryption.SM3_WITH_SM2,
                    new String(Base64.encode(content.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    merchantSignPrivateKey,
                    merchantSignCert,
                    SM2Utils.session);
        } catch (Exception e) {
            log.error("签名失败", e);
            throw new FsApiException("签名失败", e);
        }
    }

    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName(MsfpayConstant.ClientInfo.CLIENT_NAME);
        clientInfoModel.setClientCode(MsfpayConstant.ClientInfo.CLIENT_CODE);
        return clientInfoModel;
    }

    @Override
    protected MsfpayBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        JSONObject body = JSONObject.parseObject(apiResponseModel.getResponseBody());
        MsfpayBaseResponse baseResponse = new MsfpayBaseResponse();
        baseResponse.setCode(body.getString(MsfpayConstant.ResponseField.RESP_CODE));
        baseResponse.setMsg(body.getString(MsfpayConstant.ResponseField.RESP_MSG));
        return baseResponse;
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext defaultRequestContext) throws IOException {
        // 创建Cookie存储
        CookieStore cookieStore = new BasicCookieStore();
        HttpClientContext context = HttpClientContext.create();
        context.setCookieStore(cookieStore);

        // 创建HttpClient实例
        HttpClient httpClient = HttpClientBuilder.create()
                .setDefaultCookieStore(cookieStore)
                .build();
        // 第一个请求
        HttpPost firstRequest = new HttpPost(apiRequestModel.getApiURL());
        // 设置请求头
        firstRequest.setHeader("Accept", "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7");
        firstRequest.setHeader("Accept-Language", "zh-CN,zh;q=0.9");
        firstRequest.setHeader("Content-Type", "application/x-www-form-urlencoded");
        firstRequest.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/139.0.0.0 Safari/537.36");

        // 设置请求体
        List<NameValuePair> firstParams = new ArrayList<>();
        firstParams.add(new BasicNameValuePair("data", apiRequestModel.getRequestForm().get(MsfpayConstant.RequestField.DATA)));
        firstParams.add(new BasicNameValuePair("formatType", "json"));
        firstParams.add(new BasicNameValuePair("cryptType", "sm2"));
        firstParams.add(new BasicNameValuePair("signType", "sm2"));

        firstRequest.setEntity(new UrlEncodedFormEntity(firstParams, "UTF-8"));
        HttpResponse firstResponse = httpClient.execute(firstRequest, context);

        HttpPost secondRequest = new HttpPost(apiRequestModel.getApiURL() + "?random=0.9149800696113427");
        // 设置请求头
        secondRequest.setHeader("Accept", "application/json, text/javascript, */*");
        secondRequest.setHeader("Accept-Language", "zh-CN,zh;q=0.9");
        secondRequest.setHeader("Content-Type", "application/x-www-form-urlencoded");
        secondRequest.setHeader("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/139.0.0.0 Safari/537.36");

        // 设置请求体
        List<NameValuePair> secondParams = new ArrayList<>();
        secondParams.add(new BasicNameValuePair("phoneVerCode", ""));
        secondParams.add(new BasicNameValuePair("phonetoken", ""));
        secondParams.add(new BasicNameValuePair("publicmoneyamt", ""));
        JSONObject jsonObject = JSONObject.parseObject(JSON.toJSONString(apiRequestModel.getRequest()));
        secondParams.add(new BasicNameValuePair("merid", jsonObject.getString("merid")));
        secondRequest.setEntity(new UrlEncodedFormEntity(secondParams, "UTF-8"));
        HttpResponse secondResponse = httpClient.execute(secondRequest, context);
        String secondResponseBody = EntityUtils.toString(secondResponse.getEntity());
        ApiResponseModel apiResponseModel = new ApiResponseModel();
        apiResponseModel.setResponseBody(secondResponseBody);
        return apiResponseModel;
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayAutoSignApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayAutoSignApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected MsfpayBaseResponse doExecute(MsfpayBizRequest request, MsfpayAutoSignApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();

        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;

        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, JSONObject.toJSONString(apiRequestModel.getParamMap()));
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();
            // 构建响应数据
            MsfpayBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
            LogUtil.info(log, "{} >> 执行请求结束 >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getParamMap()),
                    JSONObject.toJSONString(response), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);
            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw new FsApiException("请求未知异常", e);
        }
    }


}
