package com.fshows.msfpay.client;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import com.fshows.msfpay.client.model.MsfpayMerchantClientConfigModel;
import com.fshows.msfpay.constant.MsfpayConstant;
import com.fshows.msfpay.utils.SM2Utils;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import msfpay.cfca.sadk.util.Base64;
import msfpay.cfca.sadk.util.CertUtil;
import msfpay.cfca.sadk.x509.certificate.X509Cert;

import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.PrivateKey;

@Slf4j
public class MsfpayDecryptClient {
    private PrivateKey merchantSignPrivateKey;
    private PrivateKey merchantEncPrivateKey;
    private X509Cert platformEncCert;
    private X509Cert platformSignCert;
    private X509Cert merchantEncCert;
    private X509Cert merchantSignCert;

    public MsfpayDecryptClient(MsfpayMerchantClientConfigModel apiClientConfig) throws Exception {
        initCertificatesAndKeys(apiClientConfig);
    }

    private void initCertificatesAndKeys(MsfpayMerchantClientConfigModel config) throws Exception {
        try {
            // 1. 加载商户签名私钥
            merchantSignPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantSignPath(), config.getMerchantSignPwd());

            // 2. 加载商户加密私钥
            merchantEncPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantEncPath(), config.getMerchantEncPwd());

            // 3. 加载平台加密证书
            String platformEncCertContent = new String(Files.readAllBytes(Paths.get(config.getPlatformEncCertPath())))
                    .replaceAll(MsfpayConstant.StringPattern.LINE_FEED, MsfpayConstant.StringPattern.EMPTY).replace(MsfpayConstant.StringPattern.CARRIAGE_RETURN, MsfpayConstant.StringPattern.EMPTY);
            platformEncCert = new X509Cert(platformEncCertContent.getBytes(MsfpayConstant.Http.CHARSET_UTF8));

            // 4. 加载平台签名证书
            String platformSignCertContent = new String(Files.readAllBytes(Paths.get(config.getPlatformSignCertPath())))
                    .replaceAll(MsfpayConstant.StringPattern.LINE_FEED, MsfpayConstant.StringPattern.EMPTY).replace(MsfpayConstant.StringPattern.CARRIAGE_RETURN, MsfpayConstant.StringPattern.EMPTY);
            platformSignCert = new X509Cert(platformSignCertContent.getBytes(MsfpayConstant.Http.CHARSET_UTF8));

            merchantEncCert = CertUtil.getCertFromSM2(config.getMerchantEncPath());

            merchantSignCert = CertUtil.getCertFromSM2(config.getMerchantSignPath());

            log.info("证书和密钥初始化成功");
        } catch (Exception e) {
            log.error("初始化证书和密钥失败", e);
            throw new Exception("初始化证书和密钥失败", e);
        }
    }

    public String decrypt(String encryptedData) {
        try {
            LogUtil.info(log, "【msfpay-sdk】中投支付 >> 返参解密开始 >> encryptedData={}", encryptedData);
            byte[] decryptedData = SM2Utils.openEvelopedMessage(encryptedData,
                    merchantEncPrivateKey,
                    merchantEncCert,
                    SM2Utils.session);
            String msg = new String(decryptedData, MsfpayConstant.Http.CHARSET_UTF8);

            JSONObject Obj = JSON.parseObject(msg, Feature.OrderedField);
            String messageStr = Obj.getString(MsfpayConstant.ResponseField.MESSAGE);
            JSONObject messageObj = JSON.parseObject(messageStr, Feature.OrderedField);
            String dataStr = messageObj.getString(MsfpayConstant.MessageField.DATA);
            JSONObject body = JSON.parseObject(dataStr, Feature.OrderedField);
            String bodyStr = body.getString(MsfpayConstant.ResponseField.BODY);
            String headStr = body.getString(MsfpayConstant.MessageField.HEAD);
            JSONObject headObj = JSON.parseObject(headStr, Feature.OrderedField);
            String signStr = messageObj.getString(MsfpayConstant.MessageField.SIGN);

            boolean verify = SM2Utils.P7VerifyMessageDetach(
                    new String(Base64.encode(dataStr.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    signStr,
                    platformSignCert,
                    SM2Utils.session);

            if (!verify) {
                throw new FsApiException("验签失败");
            }
            JSONObject jsonObject = JSON.parseObject(bodyStr);
            jsonObject.put(MsfpayConstant.HeadField.RESPCODE, headObj.get(MsfpayConstant.HeadField.RESPCODE));
            jsonObject.put(MsfpayConstant.HeadField.RESPMSG, headObj.get(MsfpayConstant.HeadField.RESPMSG));
            LogUtil.info(log, "【msfpay-sdk】中投支付 >> 返参解密结束 >> encryptedData={}，参数明={}", encryptedData, jsonObject.toJSONString());
            return jsonObject.toJSONString();
        } catch (Exception e) {
            log.error("解密失败", e);
            throw new FsApiException("解密失败");
        }
    }
}
