package com.fshows.msfpay.client;

import com.fshows.msfpay.utils.MsfpayUtil;
import msfpay.cfca.sadk.util.Base64;
import msfpay.cfca.sadk.util.CertUtil;
import msfpay.cfca.sadk.x509.certificate.X509Cert;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.msfpay.apienum.MsfpayMerchantApiDefinitionEnum;
import com.fshows.msfpay.client.model.MsfpayImgClientConfigModel;
import com.fshows.msfpay.constant.MsfpayConstant;
import com.fshows.msfpay.request.MsfpayBizRequest;
import com.fshows.msfpay.response.MsfpayBaseResponse;
import com.fshows.msfpay.utils.SM2Utils;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import msfpay.cfca.sadk.util.Base64;
import msfpay.cfca.sadk.util.CertUtil;
import msfpay.cfca.sadk.x509.certificate.X509Cert;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.Map;

/**
 * 中投科信支付API客户端
 */
@Slf4j
public class MsfpayImgApiClient extends AbstractApiClient<MsfpayBizRequest, MsfpayBaseResponse, MsfpayMerchantApiDefinitionEnum> {

    private PrivateKey merchantSignPrivateKey;
    private PrivateKey merchantEncPrivateKey;
    private X509Cert platformEncCert;
    private X509Cert platformSignCert;
    private X509Cert merchantEncCert;
    private X509Cert merchantSignCert;



    /**
     * 请求执行器
     */
    protected IHttpRequestHandler httpRequestHandler = new com.fshows.sdk.core.client.component.http.FromHttpRequestHandler();

    public MsfpayImgApiClient(MsfpayImgClientConfigModel apiClientConfig) throws Exception {
        super(apiClientConfig);
        initCertificatesAndKeys(apiClientConfig);
    }

    private void initCertificatesAndKeys(MsfpayImgClientConfigModel config) throws Exception {
        try {
            // 1. 加载商户签名私钥
            merchantSignPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantSignPath(), config.getMerchantSignPwd());

            // 2. 加载商户加密私钥
            merchantEncPrivateKey = SM2Utils.getPrivateKeyFromSM2(config.getMerchantEncPath(), config.getMerchantEncPwd());

            // 3. 加载平台加密证书
            platformEncCert = MsfpayUtil.getX509Cert(config.getPlatformEncCertPath());

            // 4. 加载平台签名证书
            platformSignCert = MsfpayUtil.getX509Cert(config.getPlatformSignCertPath());

            merchantEncCert = CertUtil.getCertFromSM2(config.getMerchantEncPath());

            merchantSignCert = CertUtil.getCertFromSM2(config.getMerchantSignPath());

            log.info("证书和密钥初始化成功");
        } catch (Exception e) {
            log.error("初始化证书和密钥失败", e);
            throw new Exception("初始化证书和密钥失败", e);
        }
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayMerchantApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public MsfpayBaseResponse execute(MsfpayBizRequest request, MsfpayMerchantApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected MsfpayBaseResponse doExecute(MsfpayBizRequest request, MsfpayMerchantApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}", getClientInfo().getClientDesc(), iApiDefinition);

        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;

        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);

            LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, JSONObject.toJSONString(apiRequestModel.getParamMap()));
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();
            // 构建响应数据
            MsfpayBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 执行请求结束 >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getParamMap()),
                    JSONObject.toJSONString(response), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);

            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw new FsApiException("请求未知异常", e);
        }
    }

    @Override
    protected DefaultRequestContext buildRequestContext(MsfpayMerchantApiDefinitionEnum iApiDefinition, MsfpayBizRequest request, DefaultClientConfigModel customConfig) {
        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(iApiDefinition);

        MsfpayImgClientConfigModel config = new MsfpayImgClientConfigModel();
        MsfpayImgClientConfigModel clientConfig = (MsfpayImgClientConfigModel) this.apiClientConfig;

        config.setApiParentURL(clientConfig.getApiParentURL());
        config.setCharset(clientConfig.getCharset());
        config.setSignType(clientConfig.getSignType());
        config.setVersion(clientConfig.getVersion());
        config.setPlatmerid(clientConfig.getPlatmerid());
        config.setTimeout(clientConfig.getTimeout());

        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    @Override
    protected ApiRequestModel buildApiRequestModel(MsfpayBizRequest request, DefaultRequestContext context) {
        MsfpayImgClientConfigModel configModel = (MsfpayImgClientConfigModel) context.getApiClientConfig();
        MsfpayMerchantApiDefinitionEnum iApiDefinition = (MsfpayMerchantApiDefinitionEnum) context.getIApiDefinition();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        apiRequestModel.setApiURL(configModel.getApiParentURL());
        apiRequestModel.setContentType(MsfpayConstant.Http.CONTENT_TYPE_FORM);

        try {
            fillCommonHead(request, configModel, iApiDefinition);

            String requestJson = JSON.toJSONString(request);
            Map<String, Object> requestMap = JSON.parseObject(requestJson, Map.class);

            String imgs = String.valueOf(requestMap.get(MsfpayConstant.RequestField.IMGS));

            // 构建签名数据
            Map<String, String> signDataMap = new HashMap<>();
            signDataMap.put(MsfpayConstant.RequestField.IMG, imgs.substring(Math.max(0, imgs.length()-50)));
            signDataMap.put(MsfpayConstant.HeadField.PLATMERID, request.getPlatmerid());
            signDataMap.put(MsfpayConstant.HeadField.TRANCODE, request.getTrancode());
            signDataMap.put(MsfpayConstant.HeadField.TRANTIME, request.getTrantime());

            String dataJson = JSON.toJSONString(signDataMap, SerializerFeature.WriteNullStringAsEmpty);
            String sign = sign(dataJson);

            // 构建message结构
            JSONObject message = new JSONObject();
            message.put(MsfpayConstant.MessageField.SIGN, sign);
            message.put(MsfpayConstant.MessageField.DATA, dataJson);

            apiRequestModel.setParamMap(message);

            String messageJson = JSON.toJSONString(message, SerializerFeature.WriteNullStringAsEmpty);
            byte[] envelopdata = SM2Utils.envelopeMessage(messageJson, MsfpayConstant.Encryption.SM4_ECB_PKCS7, platformEncCert);

            Map<String,String> paramMap = new HashMap<String,String>();
            paramMap.put(MsfpayConstant.RequestField.FORMAT_TYPE, MsfpayConstant.RequestValue.FORMAT_JSON);
            paramMap.put(MsfpayConstant.RequestField.CRYPT_TYPE, MsfpayConstant.RequestValue.CRYPT_SM2);
            paramMap.put(MsfpayConstant.RequestField.SIGN_TYPE, MsfpayConstant.RequestValue.SIGN_SM2);
            paramMap.put(MsfpayConstant.RequestField.DATA, new String(envelopdata, MsfpayConstant.Http.CHARSET_UTF8));
            paramMap.put(MsfpayConstant.RequestField.IMGS, imgs);

            apiRequestModel.setRequestForm(paramMap);
        } catch (Exception e) {
            throw new FsApiException("构建请求失败", e);
        }

        return apiRequestModel;
    }

    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName(MsfpayConstant.ClientInfo.CLIENT_NAME);
        clientInfoModel.setClientCode(MsfpayConstant.ClientInfo.CLIENT_CODE);
        return clientInfoModel;
    }

    @Override
    protected MsfpayBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        try {
            String responseBody = apiResponseModel.getResponseBody();

            byte[] decryptedData = SM2Utils.openEvelopedMessage(responseBody,
                    merchantEncPrivateKey,
                    merchantEncCert,
                    SM2Utils.session);

            String msg = new String(decryptedData, MsfpayConstant.Http.CHARSET_UTF8);

            JSONObject Obj = JSON.parseObject(msg, Feature.OrderedField);
            String dataStr = Obj.getString(MsfpayConstant.MessageField.DATA);

            String signStr = Obj.getString(MsfpayConstant.MessageField.SIGN);

            boolean verify = SM2Utils.P7VerifyMessageDetach(
                    new String(Base64.encode(dataStr.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    signStr,
                    platformSignCert,
                    SM2Utils.session);

            if(!verify){
                throw new FsApiException("验签失败");
            }

            Map<String, String> parse =(Map<String, String>) JSONObject.parse(dataStr);
            String pictureid = parse.get(MsfpayConstant.ResponseField.PICTURE_ID);
            String respCode = parse.get(MsfpayConstant.ResponseField.RESP_CODE);
            String respMsg = parse.get(MsfpayConstant.ResponseField.RESP_MSG);

            MsfpayBaseResponse baseResponse = new MsfpayBaseResponse();
            baseResponse.setCode(respCode);
            baseResponse.setMsg(respMsg);

            Map<String,String> bodyNew = new HashMap<>();
            bodyNew.put(MsfpayConstant.ResponseField.PICTURE_ID, pictureid);

            IResponseDefinition iResponseDefinition = JSON.parseObject(JSON.toJSONString(bodyNew), requestContext.getIApiDefinition().getResponseClass());
            baseResponse.setiResponseDefinition(iResponseDefinition);

            return baseResponse;
        } catch (Exception e) {
            LogUtil.error(log, "解析响应失败", e);
            throw new FsApiException("解析响应失败", e);
        }
    }

    private String sign(String content) {
        try {
            return SM2Utils.P7SignMessageDetach(MsfpayConstant.Encryption.SM3_WITH_SM2,
                    new String(Base64.encode(content.getBytes(MsfpayConstant.Http.CHARSET_UTF8))),
                    merchantSignPrivateKey,
                    merchantSignCert,
                    SM2Utils.session);
        } catch (Exception e) {
            log.error("签名失败", e);
            throw new FsApiException("签名失败", e);
        }
    }

    private void fillCommonHead(MsfpayBizRequest request, MsfpayImgClientConfigModel configModel, MsfpayMerchantApiDefinitionEnum iApiDefinition) {
        if(StringUtils.isEmpty(request.getPlatmerid())){
            request.setPlatmerid(configModel.getPlatmerid());
        }
        request.setTrancode(iApiDefinition.getTrancode());
        request.setTrantime(MsfpayUtil.getCurrentDateTime(MsfpayConstant.DateTimeFormat.DATETIME_FORMAT));
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }
}