package com.fshows.leshuapay.sdk.client.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.leshuapay.sdk.client.LeshuaRiskClient;
import com.fshows.leshuapay.sdk.enums.RiskApiEnum;
import com.fshows.leshuapay.sdk.exception.LeshuaException;
import com.fshows.leshuapay.sdk.request.LeshuaBizRequest;
import com.fshows.leshuapay.sdk.response.LeshuaBaseResponse;
import com.fshows.leshuapay.sdk.util.SignUtil;
import com.fshows.leshuapay.sdk.util.ValidateUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import java.io.IOException;

/**
 * @author cll
 * @version LeshuaRiskClientImpl.java, v 0.1 2021-08-12 11:20 cll
 */
@Slf4j
@Data
public class LeshuaRiskClientImpl implements LeshuaRiskClient {

  /**
   * 从connectManager获取Connection 超时时间
   */
  private static final int CONNECTION_REQUEST_TIMEOUT = 10000;

  /**
   * 读取数据的超时时间
   */
  private static final int SOCKET_TIMEOUT = 10000;

  /**
   * 连接超时时间
   */
  private static final int CONNECT_TIMEOUT = 10000;

  private static CloseableHttpClient httpclient = HttpClients.createDefault();

  /**
   * 商户服务地址
   */
  private String serverUrl;

  /**
   * 服务商编号
   */
  private String agentId;

  /**
   * 服务商私钥
   */
  private String privateKey;

  /**
   * 请求超时时间，单位毫秒
   */
  private Integer timeout;

  /**
   * @param serverUrl  商户地址
   * @param agentId    服务商编号
   * @param privateKey 服务商私钥
   * @param timeout    毫秒
   */
  public LeshuaRiskClientImpl(
          String serverUrl, String agentId, String privateKey, Integer timeout) {
    this.serverUrl = serverUrl;
    this.agentId = agentId;
    this.privateKey = privateKey;
    this.timeout = timeout;
  }

  @Override
  public <T> LeshuaBaseResponse<T> execute(
          LeshuaBizRequest<T> request, RiskApiEnum riskApiEnum, String url, Integer timeout)
          throws LeshuaException {
    ValidateUtil.validateWithThrow(request);
    // 要调用的接口方法
    HttpPost httpPost = new HttpPost(serverUrl + url);
    // 根据不同的url获取不同的签名值
    request = riskApiEnum.getRiskApiSign().doSign(request, privateKey, agentId);
    StringEntity stringEntity =
            new StringEntity(JSON.toJSONString(request), ContentType.APPLICATION_JSON);
    httpPost.setEntity(stringEntity);
    CloseableHttpResponse httpResponse = null;
    HttpEntity resEntity = null;
    try {
      log.info("请求乐刷接口入参:{}", JSON.toJSONString(request));
      httpResponse = httpclient.execute(httpPost);
      resEntity = httpResponse.getEntity();
      String responseStr = EntityUtils.toString(resEntity, SignUtil.DEFAULT_CHARSET);
      LeshuaBaseResponse<T> response = new LeshuaBaseResponse<>();
      JSONObject jsonObject = JSON.parseObject(responseStr);
      response.setRespCode(jsonObject.getString("error_code"));
      response.setRespMsg(jsonObject.getString("error_msg"));
      response.setData(JSON.parseObject(jsonObject.getString("data"), request.getResponseClass()));
      log.info("请求乐刷接口返回参数:{}", response);
      return response;
    } catch (Exception e) {
      throw new LeshuaException("调用失败", e);
    } finally {
      try {
        EntityUtils.consume(resEntity);
      } catch (IOException e) {
        log.error("释放HttpEntity出错，错误信息：" + e.getMessage(), e);
      }
      if (httpResponse != null) {
        try {
          httpResponse.close();
        } catch (IOException e) {
          log.error("关闭HttpResponse出错，错误信息：" + e.getMessage(), e);
        }
      }
    }
  }
}
