/**
 * fshows.com
 * Copyright (C) 2013-2020 All Rights Reserved.
 */
package com.fshows.leshuapay.sdk.client.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fshows.leshuapay.sdk.client.LeshuaPaymentClient;
import com.fshows.leshuapay.sdk.enums.LeshuaSettlementApiEnum;
import com.fshows.leshuapay.sdk.exception.LeshuaException;
import com.fshows.leshuapay.sdk.request.merchant.settlement.MerchantSettlementOrderRequest;
import com.fshows.leshuapay.sdk.response.merchant.settlement.MerchantSettlementOrderBaseResponse;
import com.fshows.leshuapay.sdk.response.merchant.settlement.MerchantSettlementOrderResponse;
import com.fshows.leshuapay.sdk.util.FsHttpUtil;
import com.fshows.leshuapay.sdk.util.LeshuaSignature;
import com.fshows.leshuapay.sdk.util.ReqSerialNoUtil;
import com.fshows.leshuapay.sdk.util.SignUtil;
import com.fshows.leshuapay.sdk.util.ValidateUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * 商户打款clientImpl
 *
 * @author zhaoyi
 * @version LeshuaPaymentClientImpl.java, v 0.1 2020-07-30 16:57 zhaoyi
 */
@Slf4j
@Data
public class LeshuaPaymentClientImpl implements LeshuaPaymentClient {

    private String agentId;

    private String serverUrl;

    private String privateKey;

    /**
     * 从connectManager获取Connection 超时时间
     */
    private static final int CONNECTION_REQUEST_TIMEOUT = 10000;
    /**
     * 读取数据的超时时间
     */
    private static final int SOCKET_TIMEOUT = 10000;
    /**
     * 连接超时时间
     */
    private static final int CONNECT_TIMEOUT = 10000;
    private static CloseableHttpClient httpclient = HttpClients.createDefault();
    private Integer timeout;

    public LeshuaPaymentClientImpl(String agentId, String serverUrl, String privateKey) {
        this.agentId = agentId;
        this.serverUrl = serverUrl;
        this.privateKey = privateKey;
    }

    public LeshuaPaymentClientImpl(String agentId, String serverUrl, String privateKey, Integer timeout) {
        this(agentId, serverUrl, privateKey);
        this.timeout = timeout;
    }

    @Override
    public MerchantSettlementOrderResponse querySettlementOrder(MerchantSettlementOrderRequest request, LeshuaSettlementApiEnum apiEnum) {
        ValidateUtil.validateWithThrow(request);
        MerchantSettlementOrderResponse response = new MerchantSettlementOrderResponse();
        Map<String, String> requestMap = new ConcurrentSkipListMap<>();
        requestMap.put("agentId", agentId);
        requestMap.put("billId", request.getBillId());
        requestMap.put("type", request.getType());
        String sign = LeshuaSignature.getMD5Sign(requestMap, null, privateKey);
        if (sign != null) {
            sign = sign.toLowerCase();
        }
        requestMap.put("sign", sign);
        try {
            String url = serverUrl + apiEnum.getValue();
            String responseStr = FsHttpUtil.post(url, requestMap);
            log.info("querySettlementOrder >> 乐刷结算单查询 >> url={},request={},response={}", url, JSONObject.toJSONString(requestMap), responseStr);
            JSONObject jsonObject = JSON.parseObject(responseStr);
            response = jsonObject.toJavaObject(MerchantSettlementOrderResponse.class);
            response.setErrorCode(jsonObject.getString("error_code"));
            response.setErrorMsg(jsonObject.getString("error_msg"));
            return response;
        } catch (IOException e) {
            log.error("执行http请求异常,e={}", ExceptionUtils.getStackTrace(e));
        }
        return response;
    }

    @Override
    public <Req, Res> MerchantSettlementOrderBaseResponse<Res> execute(Req request, LeshuaSettlementApiEnum apiEnum, Class<Res> resClass) throws LeshuaException {
        ValidateUtil.validateWithThrow(request);
        MerchantSettlementOrderBaseResponse<Res> response = new MerchantSettlementOrderBaseResponse<>();
        String url = serverUrl + apiEnum.getValue();
        String responseStr = executeHttp(JSON.toJSONString(request), url);
        JSONObject jsonObject = JSON.parseObject(responseStr);
        response.setErrorMsg(jsonObject.getString("error_msg"));
        response.setPageNo(jsonObject.getInteger("pageNo"));
        response.setPageSize(jsonObject.getInteger("pageSize"));
        response.setErrorCode(jsonObject.getString("error_code"));
        response.setTotalCount(jsonObject.getInteger("totalCount"));
        JSONObject dataJson = jsonObject.getJSONObject("data");
        JSONArray jsonArray = dataJson.getJSONArray("list");
        List<Res> orders = jsonArray.toJavaList(resClass);
        response.setList(orders);
        return response;

    }

    private String executeHttp(String request, String url) throws LeshuaException {
        //获取签名值
        String sign = SignUtil.getSettlementOrderSign(request, privateKey);
        RequestConfig requestConfig = null;
        if (timeout == null || timeout <= 0) {
            requestConfig = RequestConfig.custom().setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT).setConnectTimeout(CONNECT_TIMEOUT).build();
        } else {
            requestConfig = RequestConfig.custom().setConnectionRequestTimeout(timeout)
                    .setSocketTimeout(timeout).setConnectTimeout(timeout).build();
        }
        HttpPost httppost = new HttpPost(url);
        httppost.setConfig(requestConfig);
        MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();
        multipartEntityBuilder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
        multipartEntityBuilder.setCharset(Charset.forName(SignUtil.DEFAULT_CHARSET));
        String reqSerialNo = ReqSerialNoUtil.getReqSerialNo();
        //代理商编号
        multipartEntityBuilder.addPart("agentId", new StringBody(agentId, ContentType.APPLICATION_JSON));
        //版本
        multipartEntityBuilder.addPart("version", new StringBody(SignUtil.DEFAULT_VERSION, ContentType.APPLICATION_JSON));
        //请求流水号
        multipartEntityBuilder.addPart("reqSerialNo", new StringBody(reqSerialNo, ContentType.APPLICATION_JSON));
        //数据内容
        multipartEntityBuilder.addPart("data", new StringBody(request, ContentType.APPLICATION_JSON));
        //签名值
        multipartEntityBuilder.addPart("sign", new StringBody(sign, ContentType.APPLICATION_JSON));
        HttpEntity reqEntity = multipartEntityBuilder.build();
        httppost.setEntity(reqEntity);
        try {
            CloseableHttpResponse httpResponse = httpclient.execute(httppost);
            HttpEntity resEntity = httpResponse.getEntity();
            String responseStr = EntityUtils.toString(resEntity, SignUtil.DEFAULT_CHARSET);
            log.info("请求乐刷接口 >> url={},request={},response={}", url, EntityUtils.toString(reqEntity, SignUtil.DEFAULT_CHARSET), responseStr);
            EntityUtils.consume(resEntity);
            httpResponse.close();
            return responseStr;
        } catch (Exception e) {
            throw new LeshuaException("调用失败", e);
        }
    }
}
