/*
 * ailike.com
 * Copyright (C) 2022-2025 All Rights Reserved.
 */
package com.fshows.lakala.client.impl;

import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.lakala.client.base.*;
import com.fshows.lakala.constant.LakalaConstant;
import com.fshows.lakala.exception.LakalaApiException;
import com.fshows.lakala.request.base.LakalaBaseOtherRequest;
import com.fshows.lakala.request.base.LakalaBizRequest;
import com.fshows.lakala.response.base.LakalaBaseOtherResponse;
import com.fshows.lakala.response.base.LakalaBizResponse;
import com.fshows.lakala.util.ReqIdUtil;

/**
 * @author zhangling
 * @version OtherLakalaApiClientImpl.java, v 0.1 2025-07-31 09:42 zhangling
 */
public class OtherLakalaApiClientImpl extends AbstractLakalaApiClientImpl<LakalaBaseOtherRequest, LakalaBaseOtherResponse> {
    private static SerializeConfig serializeConfig = new SerializeConfig();

    static {
        serializeConfig.propertyNamingStrategy = PropertyNamingStrategy.CamelCase;
    }

    public OtherLakalaApiClientImpl(ApiClientConfig apiClientConfig) throws LakalaApiException {
        super(apiClientConfig);
    }

    public OtherLakalaApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws LakalaApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends LakalaBizResponse, R extends ILakalaApiDefinition> LakalaBaseOtherResponse<T> execute(LakalaBizRequest<T, R> request, R merchantApiDefinitionEnum) throws LakalaApiException {
        return super.doExecute(request, merchantApiDefinitionEnum);
    }

    /**
     * 添加公共参数构建完整的请求
     *
     * @param bizRequest                业务请求参数
     * @param merchantApiDefinitionEnum
     * @return
     */
    @Override
    protected LakalaBaseOtherRequest buildLakalaBizRequest(LakalaBizRequest bizRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        LakalaBaseOtherRequest request = new LakalaBaseOtherRequest();
        request.setTimestamp(System.currentTimeMillis());
        request.setRnd(RandomUtil.randomString(12));
        request.setVer(merchantApiDefinitionEnum.getVersion());
        request.setReqId(ReqIdUtil.getId());
        request.setReqData(bizRequest);
        return request;
    }

    @Override
    protected SignParam buildSignParam(LakalaBaseOtherRequest lakalaBaseRequest, String bodyStr) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setSerialNo(apiClientConfig.getSerialNo());
        signParam.setBody(bodyStr);
        signParam.setNonceStr(lakalaBaseRequest.getRnd());
        signParam.setTimeStamp(lakalaBaseRequest.getTimestamp() / 1000L);
        return signParam;
    }

    @Override
    protected String serializableRequest(LakalaBaseOtherRequest lakalaBaseRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        return JSONObject.toJSONString(lakalaBaseRequest, serializeConfig);
    }

    @Override
    protected LakalaBaseOtherResponse parseResponse(String resBody, ILakalaApiDefinition merchantApiDefinitionEnum) {
        JSONObject resJson = JSONObject.parseObject(resBody);

        // 业务响应结果
        LakalaBizResponse bizResData = (LakalaBizResponse) resJson.getObject(LakalaConstant.LAKALA_TRADE_RESPONSE_BIZ_KEY, merchantApiDefinitionEnum.getResponseClass());
        resJson.remove(LakalaConstant.LAKALA_TRADE_RESPONSE_BIZ_KEY);

        // 完整响应结果
        LakalaBaseOtherResponse response = resJson.toJavaObject(LakalaBaseOtherResponse.class);
        response.setRespData(bizResData);
        return response;
    }
}