/*
 * ailike.com
 * Copyright (C) 2022-2025 All Rights Reserved.
 */
package com.fshows.lakala.client.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.lakala.client.base.ApiClientConfig;
import com.fshows.lakala.client.base.IHttpRequest;
import com.fshows.lakala.client.base.ILakalaApiDefinition;
import com.fshows.lakala.client.base.ISigner;
import com.fshows.lakala.client.base.SignParam;
import com.fshows.lakala.constant.LakalaConstant;
import com.fshows.lakala.exception.LakalaApiException;
import com.fshows.lakala.request.base.LakalaBaseEcRequest;
import com.fshows.lakala.request.base.LakalaBizRequest;
import com.fshows.lakala.response.base.LakalaBaseEcResponse;
import com.fshows.lakala.response.base.LakalaBizResponse;

import java.util.Date;

/**
 * 电子合同相关接口client
 *
 * @author zhaoxumin
 * @version EcLakalaApiClientImpl.java, v 0.1 2025-09-12 14:36 zhaoxumin
 */
public class EcLakalaApiClientImpl extends AbstractLakalaApiClientImpl<LakalaBaseEcRequest, LakalaBaseEcResponse> {
    private static SerializeConfig serializeConfig = new SerializeConfig();

    static {
        serializeConfig.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
    }

    public EcLakalaApiClientImpl(ApiClientConfig apiClientConfig) throws LakalaApiException {
        super(apiClientConfig);
    }

    public EcLakalaApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws LakalaApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends LakalaBizResponse, R extends ILakalaApiDefinition> LakalaBaseEcResponse<T> execute(LakalaBizRequest<T, R> request, R merchantApiDefinitionEnum) throws LakalaApiException {
        return super.doExecute(request, merchantApiDefinitionEnum);
    }

    /**
     * 添加公共参数构建完整的请求
     *
     * @param bizRequest                业务请求参数
     * @param merchantApiDefinitionEnum
     * @return
     */
    @Override
    protected LakalaBaseEcRequest buildLakalaBizRequest(LakalaBizRequest bizRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        LakalaBaseEcRequest request = new LakalaBaseEcRequest();
        Date now = new Date();
        request.setReqTime(DateUtil.format(now, "yyyyMMddHHmmss"));
        request.setTimestamp(now.getTime());
        request.setRnd(RandomUtil.randomString(12));
        request.setVersion(merchantApiDefinitionEnum.getVersion());
        request.setReqData(bizRequest);
        return request;
    }

    @Override
    protected SignParam buildSignParam(LakalaBaseEcRequest lakalaBaseRequest, String bodyStr) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setSerialNo(apiClientConfig.getSerialNo());
        signParam.setBody(bodyStr);
        signParam.setNonceStr(lakalaBaseRequest.getRnd());
        signParam.setTimeStamp(lakalaBaseRequest.getTimestamp() / 1000L);
        return signParam;
    }

    @Override
    protected String serializableRequest(LakalaBaseEcRequest lakalaBaseRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        return JSONObject.toJSONString(lakalaBaseRequest, serializeConfig);
    }

    @Override
    protected LakalaBaseEcResponse parseResponse(String resBody, ILakalaApiDefinition merchantApiDefinitionEnum) {
        JSONObject resJson = JSONObject.parseObject(resBody);

        // 业务响应结果
        LakalaBizResponse bizResData = (LakalaBizResponse) resJson.getObject(LakalaConstant.LAKALA_TRADE_RESPONSE_BIZ_KEY, merchantApiDefinitionEnum.getResponseClass());
        resJson.remove(LakalaConstant.LAKALA_TRADE_RESPONSE_BIZ_KEY);

        // 完整响应结果
        LakalaBaseEcResponse response = resJson.toJavaObject(LakalaBaseEcResponse.class);
        response.setRespData(bizResData);
        return response;
    }
}