/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package com.fshows.lakala.client.impl;

import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import com.fshows.lakala.client.base.ApiClientConfig;
import com.fshows.lakala.client.base.IHttpRequest;
import com.fshows.lakala.client.base.ILakalaApiDefinition;
import com.fshows.lakala.client.base.ISigner;
import com.fshows.lakala.client.base.LakalaHttpResult;
import com.fshows.lakala.client.base.SignParam;
import com.fshows.lakala.constant.LakalaConstant;
import com.fshows.lakala.exception.LakalaApiException;
import com.fshows.lakala.request.base.LakalaBaseMerchantRequest;
import com.fshows.lakala.request.base.LakalaBizRequest;
import com.fshows.lakala.response.base.LakalaBaseMerchantResponse;
import com.fshows.lakala.response.base.LakalaBaseResponse;
import com.fshows.lakala.response.base.LakalaBizResponse;
import com.fshows.lakala.util.LogUtil;
import com.fshows.lakala.util.ReqIdUtil;
import lombok.extern.slf4j.Slf4j;

/**
 * @author zhubo
 * @version UploadFileClientImpl.java, v 0.1 2023-02-24 14:56 zhubo
 */
@Slf4j
public class LakalaUploadFileClientImpl extends AbstractLakalaApiClientImpl<LakalaBaseMerchantRequest, LakalaBaseMerchantResponse> {

    private static SerializeConfig serializeConfig = new SerializeConfig();

    static {
        serializeConfig.propertyNamingStrategy = PropertyNamingStrategy.CamelCase;
    }

    public LakalaUploadFileClientImpl(ApiClientConfig apiClientConfig) throws LakalaApiException {
        super(apiClientConfig);
    }

    public LakalaUploadFileClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws LakalaApiException {
        super(apiClientConfig, httpRequest, signer);
    }

    @Override
    public <T extends LakalaBizResponse, R extends ILakalaApiDefinition> LakalaBaseResponse<T> execute(LakalaBizRequest<T, R> bizRequest, R merchantApiDefinitionEnum) throws LakalaApiException {
        String url = null;
        try {
            // 参数校验
            checkParam(bizRequest, merchantApiDefinitionEnum);
            // 补全公共参数
            LakalaBaseMerchantRequest lakalaBaseRequest = buildLakalaBizRequest(bizRequest, merchantApiDefinitionEnum);
            // 序列化请求参数
            String body = serializableRequest(lakalaBaseRequest, merchantApiDefinitionEnum);
            // 构建签名参数
            SignParam signParam = buildSignParam(lakalaBaseRequest, body);
            // 执行加签
            String authorization = signer.sign(signParam, this.apiClientConfig);
            // 执行加签获取签名
            url = getServerURL(merchantApiDefinitionEnum);
            //执行HTTP post请求
            LakalaHttpResult result = doRequest(url, authorization, signParam.getBody());
            //反序列化相应参数
            return parseResponse(result.getBody(), merchantApiDefinitionEnum);
        } catch (LakalaApiException e) {
            LogUtil.error(log, "【lakala-sdk】拉卡拉请求异常 >> url={}", e, url);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【lakala-sdk】拉卡拉请求异常 >> url={}", e, url, bizRequest);
            throw new LakalaApiException(e.getMessage(), e);
        }

    }

    private LakalaHttpResult doRequest(String url, String authorization, String requestBody) throws LakalaApiException {
        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.info(log, "【lakala-sdk】请求开始 >> doRequest >> url={},begin={}", url, beginTime);
            LakalaHttpResult result = httpRequest.post(url, authorization, requestBody, apiClientConfig);
            if (null == result) {
                throw new LakalaApiException("lakala相应结果为空");
            }
            LogUtil.info(log, "【lakala-sdk】请求结束 >> doRequest >> url={},response={},lakalTraceId={},cost={}", url,
                    result.getBody(), result.getLakalaTraceId(), (System.currentTimeMillis() - beginTime));
            return result;
        } catch (LakalaApiException e) {
            LogUtil.error(log, "【lakala-sdk】拉卡拉请求异常 >> doRequest >> url={}, cost={}", url, (System.currentTimeMillis() - beginTime));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【lakala-sdk】拉卡拉请求异常 >> doRequest >> url={}, cost={}", url, (System.currentTimeMillis() - beginTime));
            throw new LakalaApiException("请求拉卡拉接口异常", e);
        }
    }

    @Override
    protected LakalaBaseMerchantRequest buildLakalaBizRequest(LakalaBizRequest bizRequest, ILakalaApiDefinition merchantApiDefinitionEnum) {
        LakalaBaseMerchantRequest baseMerchantRequest = new LakalaBaseMerchantRequest();
        baseMerchantRequest.setTimestamp(System.currentTimeMillis());
        baseMerchantRequest.setRnd(RandomUtil.randomString(12));
        baseMerchantRequest.setVer(merchantApiDefinitionEnum.getVersion());
        baseMerchantRequest.setReqId(ReqIdUtil.getId());
        baseMerchantRequest.setReqData(bizRequest);
        return baseMerchantRequest;
    }

    @Override
    protected SignParam buildSignParam(LakalaBaseMerchantRequest lakalaBaseRequest, String bodyStr) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setSerialNo(apiClientConfig.getSerialNo());
        signParam.setBody(bodyStr);
        signParam.setNonceStr(lakalaBaseRequest.getRnd());
        signParam.setTimeStamp(lakalaBaseRequest.getTimestamp() / 1000L);
        return signParam;
    }

    @Override
    protected String serializableRequest(LakalaBaseMerchantRequest lakalaBaseRequest, ILakalaApiDefinition tradeApiDefinitionEnum) {
        return JSONObject.toJSONString(lakalaBaseRequest, serializeConfig);
    }

    @Override
    protected LakalaBaseMerchantResponse parseResponse(String resBody, ILakalaApiDefinition merchantApiDefinitionEnum) {
        JSONObject resJson = JSONObject.parseObject(resBody);
        LakalaBizResponse bizResData = (LakalaBizResponse) resJson.getObject(LakalaConstant.LAKALA_RESPONSE_BIZ_KEY, merchantApiDefinitionEnum.getResponseClass());
        resJson.remove(LakalaConstant.LAKALA_RESPONSE_BIZ_KEY);
        LakalaBaseMerchantResponse response = resJson.toJavaObject(LakalaBaseMerchantResponse.class);
        response.setRespData(bizResData);
        return response;
    }
}
