/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.lakala.client.impl;

import com.fshows.lakala.client.base.ApiClientConfig;
import com.fshows.lakala.client.base.IHttpRequest;
import com.fshows.lakala.client.base.ILakalaApiClient;
import com.fshows.lakala.client.base.ILakalaApiDefinition;
import com.fshows.lakala.client.base.ISigner;
import com.fshows.lakala.client.base.LakalaHttpResult;
import com.fshows.lakala.client.base.SignParam;
import com.fshows.lakala.exception.LakalaApiException;
import com.fshows.lakala.request.base.LakalaBaseRequest;
import com.fshows.lakala.request.base.LakalaBizRequest;
import com.fshows.lakala.response.base.LakalaBaseResponse;
import com.fshows.lakala.util.LogUtil;
import com.fshows.lakala.util.ValidateUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;

/**
 * 默认拉卡拉api客户端实现
 *
 * @author liluqing
 * @version DefaultLakalaApiClient.java, v 0.1 2022-03-02 11:59
 */
@Setter
@Slf4j
public abstract class AbstractLakalaApiClientImpl<T extends LakalaBaseRequest, R extends LakalaBaseResponse> implements ILakalaApiClient {

    /**
     * 客户端配置
     */
    protected ApiClientConfig apiClientConfig;

    /**
     * http请求器
     */
    protected IHttpRequest httpRequest;

    /**
     * 加签
     */
    protected ISigner signer;

    public AbstractLakalaApiClientImpl(ApiClientConfig apiClientConfig) throws LakalaApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = new DefaultHttpRequestImpl();
        this.signer = new DefaultSignerImpl();
    }

    public AbstractLakalaApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws LakalaApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = httpRequest;
        this.signer = signer;
    }

    /**
     * 执行拉卡拉请求
     *
     * @param bizRequest
     * @param tradeApiDefinitionEnum
     * @return
     * @throws LakalaApiException
     */
    protected R doExecute(LakalaBizRequest bizRequest, ILakalaApiDefinition tradeApiDefinitionEnum) throws LakalaApiException {
        try {
            // 参数校验
            checkParam(bizRequest, tradeApiDefinitionEnum);
            // 补全全局公共参数
            T lakalaBaseRequest = buildLakalaBizRequest(bizRequest, tradeApiDefinitionEnum);
            // 序列化请求参数
            String body = serializableRequest(lakalaBaseRequest, tradeApiDefinitionEnum);
            // 构建签名参数
            SignParam signParam = buildSignParam(lakalaBaseRequest, body);
            // 执行加签动作
            String authorization = signer.sign(signParam, this.apiClientConfig);
            // 执行加签操作获取签名
            String url = getServerURL(tradeApiDefinitionEnum);
            // 执行HTTP post请求
            LakalaHttpResult httpResult = requestPost(url, authorization, signParam.getBody());
            // 根据是否验签配置判断是否需要验签
            if (this.apiClientConfig.isVrifySignResponse()) {
                boolean bo = signer.verifySign(httpResult.getBody(), httpResult.getVerifySignParam(), this.apiClientConfig);
                if (!bo) {
                    LogUtil.error(log, "【lakala-sdk】响应结果验签失败 >> httpResult={}", httpResult);
                    throw new LakalaApiException("[lakala-sdk]响应结果验签失败");
                }
            }
            // 反序列化响应参数
            return parseResponse(httpResult.getBody(), tradeApiDefinitionEnum);
        } catch (LakalaApiException e) {
            LogUtil.error(log, "【lakala-sdk】拉卡拉请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【lakala-sdk】拉卡拉请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw new LakalaApiException(e.getMessage(), e);
        }
    }

    /**
     * 执行post请求
     *
     * @param url
     * @param authorization
     * @param requestBody
     * @return
     */
    protected LakalaHttpResult requestPost(String url, String authorization, String requestBody) throws IOException, LakalaApiException {
        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.info(log, "【lakala-sdk】请求开始 >> url={}, authorization={}, request={}, begin={}", url, authorization, requestBody, beginTime);
            LakalaHttpResult httpResult = httpRequest.post(url, authorization, requestBody, apiClientConfig);
            if (httpResult == null) {
                throw new LakalaApiException("lakala响应结果为空");
            }
            LogUtil.info(log, "【lakala-sdk】请求结束 >> url={}, request={}, response={}, lakalaTraceId={}, cost={}ms", url, requestBody,
                    httpResult.getBody(), httpResult.getLakalaTraceId(), System.currentTimeMillis() - beginTime);
            return httpResult;
        } catch (LakalaApiException e) {
            LogUtil.error(log, "【lakala-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【lakala-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw new LakalaApiException("请求lakala接口异常", e);
        }
    }

    /**
     * 请求参数校验
     *
     * @param bizRequest
     * @return
     */
    protected void checkParam(LakalaBizRequest bizRequest, ILakalaApiDefinition lakalaApiDefinition) throws LakalaApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        System.out.println();
        boolean checkRequestClass = StringUtils.equals(bizRequest.getClass().getCanonicalName(), lakalaApiDefinition.getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new LakalaApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.apiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    /**
     * 获取请求地址
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected String getServerURL(ILakalaApiDefinition tradeApiDefinitionEnum) {
        return apiClientConfig.getApiParentURL() + tradeApiDefinitionEnum.getApiSubURI();
    }


    /**
     * 构建完整的请求参数
     *
     * @param request 业务请求参数
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract T buildLakalaBizRequest(
            LakalaBizRequest request,
            ILakalaApiDefinition tradeApiDefinitionEnum);

    /**
     * 构建加签的请求参数
     *
     * @param lakalaBaseRequest 实际请求的完整参数
     * @param bodyStr 序列化之后的请求参数
     * @return
     */
    protected abstract SignParam buildSignParam(
            T lakalaBaseRequest,
            String bodyStr);

    /**
     * 序列化请求参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract String serializableRequest(
            T lakalaBaseRequest,
            ILakalaApiDefinition tradeApiDefinitionEnum);

    /**
     * 解析响应参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract R parseResponse(
            String resBody,
            ILakalaApiDefinition tradeApiDefinitionEnum);
}
