/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package com.fshows.lakala.client.impl;

import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.lakala.client.base.ApiClientConfig;
import com.fshows.lakala.client.base.IHttpRequest;
import com.fshows.lakala.client.base.ILakalaApiDefinition;
import com.fshows.lakala.client.base.ISigner;
import com.fshows.lakala.client.base.LakalaHttpResult;
import com.fshows.lakala.client.base.SignParam;
import com.fshows.lakala.constant.CommonConstant;
import com.fshows.lakala.constant.LakalaConstant;
import com.fshows.lakala.exception.LakalaApiException;
import com.fshows.lakala.request.base.LakalaBaseMerchantRequest;
import com.fshows.lakala.request.common.UploadPictureRequest;
import com.fshows.lakala.response.base.LakalaBaseMerchantResponse;
import com.fshows.lakala.response.base.LakalaBizResponse;
import com.fshows.lakala.response.common.UploadPictureReponse;
import com.fshows.lakala.util.FsHttpResult;
import com.fshows.lakala.util.FsHttpUtil;
import com.fshows.lakala.util.LogUtil;
import com.fshows.lakala.util.ReqIdUtil;
import com.fshows.lakala.util.ValidateUtil;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * @author zhubo
 * @version UploadFileClientImpl.java, v 0.1 2023-02-24 14:56 zhubo
 */
@Slf4j
public class UploadFileClient {


    private ApiClientConfig apiClientConfig;

    private IHttpRequest httpRequest;

    private static final String URL = "api/v2/mms/openApi/uploadFile";

    private ISigner signer;

    private String orgCode;

    /**
     * 上传照片
     *
     * @param request 请求
     * @return {@link UploadPictureReponse}
     */
    public LakalaBaseMerchantResponse uploadPicture(UploadPictureRequest request) throws LakalaApiException{
        // 参数校验
        if (null == request){
            LogUtil.error(log, "uploadPicture >> 参数为空");
            throw new LakalaApiException("图片上传参数为空");
        }
        ValidateUtil.validateWithThrow(request);
        // 参数填充
        request.setVersion(LakalaConstant.MERCHANT_API_VERSION_V2);
        request.setOrderNo(ReqIdUtil.getOrderNo());
        request.setOrgCode(orgCode);

        LakalaBaseMerchantRequest baseMerchantRequest = buildRequest(request);
        String body = buildBody(baseMerchantRequest);
        String reqUrl = apiClientConfig.getApiParentURL() + URL;
        SignParam signParam = buildSignParam(baseMerchantRequest, body);
        String authorization = signer.sign(signParam, this.apiClientConfig);
        Map<String, String> headers = new HashMap<>();
        headers.put("Authorization", authorization);

        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.info(log, "【lakala-sdk】请求开始 >> url={}", reqUrl, headers, beginTime);
            LakalaHttpResult result = httpRequest.post(reqUrl, authorization, signParam.getBody(), apiClientConfig);
            LogUtil.info(log, "【lakala-sdk】请求结束 >> url={}, response={}, lakalaTraceId={}, cost={}ms", reqUrl,
                    result.getBody(), result.getLakalaTraceId(), System.currentTimeMillis() - beginTime);
            if (null == result){
                throw new LakalaApiException("lakala响应结果为空");
            }
            JSONObject jsonObject = JSON.parseObject(result.getBody());
            UploadPictureReponse reponse = JSON.parseObject(jsonObject.getString("respData"), UploadPictureReponse.class);
            return parseResponse(result.getBody());
        } catch (IOException e) {
            LogUtil.error(log, "【lakala-sdk】网络请求异常 >> url={}, cost={}ms",
                    e, reqUrl, (System.currentTimeMillis() - beginTime));
            throw new LakalaApiException("请求lakala接口异常", e);
        }
    }

    private LakalaBaseMerchantRequest buildRequest(UploadPictureRequest request) {
        LakalaBaseMerchantRequest baseMerchantRequest = new LakalaBaseMerchantRequest();
        baseMerchantRequest.setTimestamp(System.currentTimeMillis());
        baseMerchantRequest.setRnd(RandomUtil.randomString(12));
        baseMerchantRequest.setVer(LakalaConstant.MERCHANT_API_VERSION_V2);
        baseMerchantRequest.setReqId(ReqIdUtil.getId());
        baseMerchantRequest.setReqData(request);
        return baseMerchantRequest;
    }

    protected String buildBody(LakalaBaseMerchantRequest request) {
        return JSONObject.toJSONString(request);
    }

    private SignParam buildSignParam(LakalaBaseMerchantRequest lakalaBaseRequest, String bodyStr) {
        SignParam signParam = new SignParam();
        signParam.setAppid(apiClientConfig.getAppId());
        signParam.setSerialNo(apiClientConfig.getSerialNo());
        signParam.setBody(bodyStr);
        signParam.setNonceStr(lakalaBaseRequest.getRnd());
        signParam.setTimeStamp(lakalaBaseRequest.getTimestamp() / 1000L);
        return signParam;
    }

    private LakalaBaseMerchantResponse parseResponse(String resBody) {
        JSONObject resJson = JSONObject.parseObject(resBody);

        // 业务响应结果
        LakalaBizResponse bizResData = resJson.getObject(LakalaConstant.LAKALA_RESPONSE_BIZ_KEY,
                UploadPictureReponse.class);
        resJson.remove(LakalaConstant.LAKALA_RESPONSE_BIZ_KEY);

        // 完整响应结果
        LakalaBaseMerchantResponse response = resJson.toJavaObject(LakalaBaseMerchantResponse.class);
        response.setRespData(bizResData);
        return response;
    }
}
