/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.hxb;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.hxb.contant.HxbpayContant;
import com.fshows.hxb.request.HxbpayBaseReq;
import com.fshows.hxb.request.HxbpayBizReq;
import com.fshows.hxb.response.HxbpayBizRes;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.PostHttpRequestHandler;
import com.fshows.sdk.core.client.component.serializable.JsonSerializableHandler;
import com.fshows.sdk.core.client.component.signer.RsaSignHandlerImpl;
import com.fshows.sdk.core.client.tempalte.nest.apienum.NestApiDefinitionEnum;
import com.fshows.sdk.core.client.tempalte.nest.request.NestBaseRequest;
import com.fshows.sdk.core.client.tempalte.nest.request.NestBizRequest;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.hxb.apienum.HxbpayTradeApiEnum;
import com.fshows.hxb.response.HxbpayBaseRes;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.RequestParamUtils;
import com.kayak.enve.EnvApplication;
import com.kayak.sign.KKAES2;
import com.kayak.sign.SignVer;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.Date;

/**
 * 嵌套参数客户端模版
 * @author liluqing
 * @version NestTemplateApiClinet.java, v 0.1 2023-12-12 15:40
 */
@Slf4j
public class HxbTradeApiClinet extends AbstractApiClient<HxbpayBizReq, HxbpayBaseRes, HxbpayTradeApiEnum> {


    /**
     * 序列化处理器
     */
    protected ISerializableHandler paramSerializable = new JsonSerializableHandler();

    /**
     * 请求执行器
     */
    protected IHttpRequestHandler httpRequestHandler = new PostHttpRequestHandler();

    /**
     * 华夏银行加密解密工具
     */
    private EnvApplication envApplication = new EnvApplication();

    public HxbTradeApiClinet(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
        String cert = envApplication.getCert(apiClientConfig.getPayCompanyPublicKey());
        apiClientConfig.setPayCompanyPublicKey(cert);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param merchantConfigModel
     * @return
     * @throws FsApiException
     */
    protected HxbpayBaseRes doExecute(HxbpayBizReq request, HxbpayTradeApiEnum iApiDefinition, DefaultClientConfigModel merchantConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, merchantConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束[密文] >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime);
            // 反序列化构建响应结果
            HxbpayBaseRes baseRes = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, cost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, JSONObject.toJSONString(request),
                    JSONObject.toJSONString(baseRes), System.currentTimeMillis() - beginTime);

            return baseRes;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求参数
     *
     * @param request
     * @param context
     * @return
     */
    @Override
    protected ApiRequestModel buildApiRequestModel(HxbpayBizReq request, DefaultRequestContext context) {
        // 请求内容
        String bodyJson = "";
        try {
            // sdk客户端配置
            DefaultClientConfigModel configModel = context.getApiClientConfig();

            // 参数补充,一级服务商ID
            request.setMerchantId(configModel.getAgentId());
            // 请求时间
            request.setPcsDate(DateUtil.format(new Date(), DatePattern.PURE_DATETIME_PATTERN));

            ApiRequestModel apiRequestModel = new ApiRequestModel();
            // 设置网关地址
            apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + context.getIApiDefinition().getApiURI());
            // 请求内容
            bodyJson = JSONObject.toJSONString(request);

            //4 签名:商户使用自己的私钥签名
            // 参数说明：(1)待签名数据(响应body内容)；(2)商户的私钥；(3)密钥口令；
            String retSigned = SignVer.signJson(bodyJson, configModel.getFubeiPrivateKey(), configModel.getPassword());
            // 业务报文加密
            String sRetBody = encode(bodyJson);
            apiRequestModel.setRequestSign(retSigned);

            // 补充参数构建实际的请求入参
            HxbpayBaseReq baseReq = new HxbpayBaseReq();
            baseReq.setAppid(configModel.getAppId());
            baseReq.setSignature(retSigned);
            baseReq.setVersion(context.getIApiDefinition().getVersion());
            baseReq.setBody(sRetBody);
            apiRequestModel.setRequest(baseReq);
            // 参数序列化
            String requestBody = paramSerializable.serializeObject(apiRequestModel, context);
            apiRequestModel.setRequestBody(requestBody);
            return apiRequestModel;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求华夏异常 >> iApiDefinition={}, request={}, bodyJson={}", e,
                    getClientInfo().getClientDesc(), context.getIApiDefinition(), request, bodyJson);
            throw new RuntimeException(e);
        }
    }

    /**
     * 处理客户端信息
     *
     * @return
     */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientCode("hxb-sdk");
        clientInfoModel.setClientName("华夏银行");
        return clientInfoModel;
    }

    @Override
    protected HxbpayBaseRes buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        // 1.  2. 请求参数填充  3. 请求加签  4. 参数反序列化
        String body = null;
        try {
            // 响应内容
            String responseBody = apiResponseModel.getResponseBody();

            JSONObject resJson = JSONObject.parseObject(responseBody);
            // 业务响应内容，密文
            String bodyCiphertext = (String) resJson.remove("body");

            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();

            HxbpayBaseRes hxbpayBaseRes = resJson.toJavaObject(HxbpayBaseRes.class);
            // 请求报文解密
            body = decode(bodyCiphertext);
            log.info(" 解密之后的报文 >> body={}", body);
            if (StringUtils.isNotBlank(body)) {
                hxbpayBaseRes.setBody(JSONObject.parseObject(body, requestContext.getIApiDefinition().getResponseClass()));
            }
            return hxbpayBaseRes;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 华夏响应异常 >> iApiDefinition={}, request={}, 解密后的body={}", e,
                    getClientInfo().getClientDesc(), requestContext.getIApiDefinition(), apiRequestModel.getRequest(), body);
            throw new RuntimeException(e);
        }
    }

    /**
     * 输入密文,返回明文
     *
     * @param ciphertext
     * @return
     */
    public String decode(String ciphertext) {
        byte[] decryption = envApplication.openEnvelope(apiClientConfig.getFubeiPrivateKey(), apiClientConfig.getPassword(), ciphertext);
        return StrUtil.str(decryption, "utf-8");
    }

    /**
     * 输入明文,返回密文
     *
     * @param plaintext
     * @return
     */
    public String encode(String plaintext) {
        return envApplication.makeEnvelope(apiClientConfig.getPayCompanyPublicKey(), KKAES2.AES, StrUtil.bytes(plaintext, ("UTF-8")));
    }

    /**
     * 验签
     *
     * @param body
     * @param signed
     * @return
     */
    public boolean verifySign(String body, String signed) {
        try {
            return SignVer.verifyJson(body, signed, apiClientConfig.getPayCompanyPublicKey());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }

    @Override
    public HxbpayBaseRes execute(HxbpayBizReq request, HxbpayTradeApiEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public HxbpayBaseRes execute(HxbpayBizReq request, HxbpayTradeApiEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

}