/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fuiou.util;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ReflectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.steward.annotation.NoSign;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author zhaoyi
 * @version LeshuaUtils.java, v 0.1 2020-07-28 09:47 zhaoyi
 */
@Slf4j
public class FuiouRequestUtils {

    /**
     * 缓存反射字段
     */
    private static final Map<Class, Field[]> CACHE_FIELD_MAP = new ConcurrentHashMap<>();

    private static final String XML = "xml";

    private static final String DEFAULT_CHARSET = "GBK";

    private static final String EMPTY = "";

    private static final String BENE_INFOS = "bene_infos";

    private static final String BENE_INFO = "bene";

    private static final String SHAREHOLDER_INFOS = "shareholder_infos";

    private static final String SHARE_HOLDER = "shareholder";

    private static final String BENE_IMG_IDS = "bene_img_ids";

    private static final String BENE_IMG = "bene_img";

    private static final String SHAREHOLD_IMG_IDS = "sharehold_img_ids";

    private static final String SHAREHOLD_IMG = "sharehold_img";




    /**
     * 忽略字段
     */
    private static final String[] IGNORE_FIELD_ARR = {"serialVersionUID"};

    private static Pattern UNDERLINE_PATTERN = Pattern.compile("[A-Z]");

    public static InputStream getStringStream(String sInputString) throws UnsupportedEncodingException {
        return getStringStream(sInputString, DEFAULT_CHARSET);
    }

    public static InputStream getStringStream(String sInputString, String charsetName) throws UnsupportedEncodingException {
        ByteArrayInputStream tInputStringStream = null;
        if (sInputString != null && !sInputString.trim().equals(EMPTY)) {
            tInputStringStream = new ByteArrayInputStream(sInputString.getBytes(charsetName));
        }
        return tInputStringStream;
    }

    public static <Res> Res getObjectFromXML(String xml, Class<Res> tClass) throws Exception {
        Map<String, String> mapFromXML = getMapFromXML(xml);
        return BeanUtil.toBean(mapFromXML, tClass);
    }

    /**
     * map 转对象
     *
     * @param mapFromXML
     * @param tClass
     * @param <Res>
     * @return
     * @throws Exception
     */
    public static <Res> Res getObjectFromMap(Map<String, String> mapFromXML, Class<Res> tClass) {
        if (mapFromXML == null) {
            return ReflectUtil.newInstance(tClass);
        }
        return BeanUtil.toBean(mapFromXML, tClass);
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值，值为null的属性排除
     *
     * @param obj
     * @return
     */
    public static Map<String, String> toMap(Object obj) {
        return toResultMap(obj, false);
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值，值为null的属性排除
     *
     * @param obj
     * @param isHump
     * @return
     */
    public static Map<String, String> toResultMap(Object obj, boolean isHump) {
        Field[] fields = CACHE_FIELD_MAP.get(obj.getClass());
        if (fields == null) {
            fields = ReflectUtil.getFields(obj.getClass());
            CACHE_FIELD_MAP.put(obj.getClass(), fields);
        }
        Map<String, String> map = new TreeMap<>();
        for (Field field : fields) {
            try {
                field.setAccessible(true);
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                if (ArrayUtil.contains(IGNORE_FIELD_ARR, field.getName())) {
                    continue;
                }
                String key = field.getName();
                Object val = field.get(obj);
                key = isHump ? key : camelToUnderline(key);
                if (val != null) {
                    map.put(key, dealSpecialCharacter(String.valueOf(val)));
                } else {
                    map.put(key, null);
                }
            } catch (IllegalArgumentException | IllegalAccessException e) {
                log.error(e.getMessage(), e);
            }
        }
        return map;
    }

    private static String dealSpecialCharacter(String str) {
        return str.replaceAll("&", "")
                .replaceAll("<", "")
                .replaceAll(">", "");
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值，值为null的属性排除
     *
     * @param obj
     * @return
     */
    public static Map<String, String> toMap(Object obj, boolean isHump) {
        return toMap(obj, isHump, false);
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值
     * isHump=true key保持驼峰，否则转小写下划线
     * isContainNull=true 保留value为null的值，否则过滤
     * @param obj
     * @return
     */
    public static Map<String, String> toMap(Object obj, boolean isHump, boolean isContainNull) {
        Field[] fields = CACHE_FIELD_MAP.get(obj.getClass());
        if (fields == null) {
            fields = ReflectUtil.getFields(obj.getClass());
            CACHE_FIELD_MAP.put(obj.getClass(), fields);
        }
        Map<String, String> map = new TreeMap<>();
        for (Field field : fields) {
            try {
                field.setAccessible(true);
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                if (ArrayUtil.contains(IGNORE_FIELD_ARR, field.getName())) {
                    continue;
                }
                String key = field.getName();
                Object val = field.get(obj);
                if (isContainNull || val != null) {
                    map.put(isHump ? key : camelToUnderline(key), dealSpecialCharacter((val == null) ? "" : val.toString()));
                }
            } catch (IllegalArgumentException | IllegalAccessException e) {
                log.error(e.getMessage(), e);
            }
        }
        return map;
    }

    /**
     * 计算失效时间
     *
     * @param now
     * @param expireInMinutes
     * @param dateFormat
     * @return
     */
    public static String calcExpireTime(Date now, String expireInMinutes, SimpleDateFormat dateFormat) {
        return dateFormat.format(now.getTime() + Long.parseLong(expireInMinutes) * 60 * 1000);
    }

    /**
     * 驼峰转下划线
     */
    public static String camelToUnderline(String str) {
        Matcher matcher = UNDERLINE_PATTERN.matcher(str);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, "_" + matcher.group(0).toLowerCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * xml报文转 map
     *
     * @param xmlString
     * @return
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    public static Map<String, String> getMapFromXML(String xmlString)
            throws ParserConfigurationException, IOException, SAXException {
        return getMapFromXML(xmlString, DEFAULT_CHARSET);
    }

    public static Map<String, String> getMapFromXML(String xmlString, String charsetName)
            throws ParserConfigurationException, IOException, SAXException {
        //xmlString = xmlString.replaceFirst("encoding=\".*\"", "encoding=\"GBK\"");
        // 这里用Dom的方式解析回包的最主要目的是防止API新增回包字段
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        InputStream is = getStringStream(xmlString, charsetName);
        Document document = builder.parse(is);

        // 获取到document里面的全部结点
        NodeList allNodes = document.getFirstChild().getChildNodes();
        Node node;
        Map<String, String> map = new TreeMap<>();
        int i = 0;
        while (i < allNodes.getLength()) {
            node = allNodes.item(i);
            if (node instanceof Element) {
                map.put(node.getNodeName(), node.getTextContent());
            }
            i++;
        }
        return map;
    }

    /**
     * xml报文转 map
     *
     * @param xmlString
     * @return
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    public static Map<String, Object> getMapFromXMLGBK(String xmlString)
            throws ParserConfigurationException, IOException, SAXException {
//        xmlString = xmlString.replaceFirst("encoding=\".*\"", "encoding=\"GBK\"");
        if (StringUtils.startsWith(xmlString, "<xml>")) {
            xmlString = "<?xml version=\"1.0\" encoding=\"GBK\" standalone=\"yes\"?>" + xmlString;
        }
        // 这里用Dom的方式解析回包的最主要目的是防止API新增回包字段
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        InputStream is = getStringStream(xmlString);
        Document document = builder.parse(is);

        // 获取到document里面的全部结点
        NodeList allNodes = document.getFirstChild().getChildNodes();
        Node node;
        Map<String, Object> map = new TreeMap<>();
        int i = 0;
        while (i < allNodes.getLength()) {
            node = allNodes.item(i);
            if (node instanceof Element) {
                map.put(node.getNodeName(), node.getTextContent());
            }
            i++;
        }
        return map;
    }


    /**
     * xml文件转map
     * @param xmlString
     * @return {@link Map}<{@link String}, {@link String}>
     */
    public static Map<String, String> xmlStringToMap(String xmlString) {
        Map<String, String> resultMap = new HashMap<>();

        try {
            InputStream inputStream = new ByteArrayInputStream(xmlString.getBytes());
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document document = builder.parse(inputStream);

            Element root = document.getDocumentElement();
            NodeList nodeList = root.getChildNodes();

            for (int i = 0; i < nodeList.getLength(); i++) {
                Node node = nodeList.item(i);
                if (node.getNodeType() == Node.ELEMENT_NODE) {
                    String nodeName = node.getNodeName();
                    String nodeValue = node.getTextContent().trim();
                    resultMap.put(nodeName, nodeValue);
                }
            }
        } catch (Exception e) {
            LogUtil.error(log,"【fuiou-sdk-fusteward】 xml文件解析异常 ,xmlStr:{}",xmlString);
            throw new FsApiException("【fuiou-sdk-fusteward】xml文件解析异常");
        }

        return resultMap;
    }


    /**
     * map 转对象
     *
     * @param mapFromXML
     * @param tClass
     * @param <Res>
     * @return
     * @throws Exception
     */
    public static <Res> Res mapToObject(Map<String, Object> mapFromXML, Class<Res> tClass) {
        String jsonStr = JSON.toJSONString(mapFromXML);
        return JSON.parseObject(jsonStr,tClass);
    }
    /**
     * 通过map生成xml
     *
     * @param data
     * @return
     */
    public static String generateXml(Map<String, String> data) {
        StringBuilder xmlBuilder = new StringBuilder();
        xmlBuilder.append("<?xml version=\"1.0\" encoding=\"GBK\" standalone=\"yes\"?>");
        xmlBuilder.append("<xml>");
        for (Map.Entry<String, String> entry : data.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (value != null) {
                xmlBuilder.append("<").append(key).append(">").append(value).append("</").append(key).append(">");
            }
        }
        xmlBuilder.append("</xml>");
        return xmlBuilder.toString();
    }


    /**
     * 通过map生成xml
     *
     * @param data
     * @return
     */
    public static String generateXmlByObjectMap(Map<String, Object> data) {
        StringBuilder xmlBuilder = new StringBuilder();
        //xmlBuilder.append("<?xml version=\"1.0\" encoding=\"GBK\" standalone=\"yes\"?>");
        xmlBuilder.append("<xml>");

        for (Map.Entry<String, Object> entry : data.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();

            // 判断是否为List
            if (value instanceof List) {
                handleListValue(xmlBuilder, key, (List<?>) value);
            } else {
                handleScalarValue(xmlBuilder, key, value);
            }
        }
        xmlBuilder.append("</xml>");
        return xmlBuilder.toString();
    }


    /**
     * 组装普通数据
     * @param xmlBuilder
     * @param key
     * @param value
     */
    private static void handleScalarValue(StringBuilder xmlBuilder, String key, Object value) {
        if (value != null && !value.toString().isEmpty()) {
            xmlBuilder.append("<").append(key).append(">").append(value).append("</").append(key).append(">");
        } else {
            xmlBuilder.append("<").append(key).append("/>");
        }
    }

    /**
     * 组装List数据
     * @param xmlBuilder
     * @param key
     * @param list
     */
    private static void handleListValue(StringBuilder xmlBuilder, String key, List<?> list) {
        xmlBuilder.append("<").append(key).append(">");
        // 遍历List组装
        for (Object listItem : list) {
            if(key.equalsIgnoreCase(BENE_INFOS)){
                xmlBuilder.append("<").append(BENE_INFO).append(">");
            }else if(key.equalsIgnoreCase(SHAREHOLDER_INFOS)){
                xmlBuilder.append("<").append(SHARE_HOLDER).append(">");
            } else if(key.equalsIgnoreCase(BENE_IMG_IDS)){
                xmlBuilder.append("<").append(BENE_IMG).append(">");
            }else if(key.equalsIgnoreCase(SHAREHOLD_IMG_IDS)){
                xmlBuilder.append("<").append(SHAREHOLD_IMG).append(">");
            } else{
                xmlBuilder.append("<").append(key).append(">");
            }
            if (listItem instanceof String
                    || listItem instanceof Integer
                    || listItem instanceof Long
                    || listItem instanceof Double
                    || listItem instanceof BigDecimal
                    || listItem instanceof Float) {
                xmlBuilder.append(listItem);
            } else {
                Map<String,Object> map = (Map<String, Object>) listItem;
                Map<String,Object> treeMap = new TreeMap(map);

                for (Map.Entry<String, Object> entry : treeMap.entrySet()) {
                    String itemKey = entry.getKey();
                    Object itemValue = entry.getValue();
                    if (itemValue != null && !itemValue.toString().isEmpty()) {
                        xmlBuilder.append("<").append(itemKey).append(">").append(itemValue).append("</").append(itemKey).append(">");
                    } else {
                        xmlBuilder.append("<").append(itemKey).append("/>");
                    }
                }
            }
            if(key.equalsIgnoreCase(BENE_INFOS)){
                xmlBuilder.append("</").append(BENE_INFO).append(">");
            }else if(key.equalsIgnoreCase(SHAREHOLDER_INFOS)){
                xmlBuilder.append("</").append(SHARE_HOLDER).append(">");
            }else if(key.equalsIgnoreCase(BENE_IMG_IDS)){
                xmlBuilder.append("</").append(BENE_IMG).append(">");
            }else if(key.equalsIgnoreCase(SHAREHOLD_IMG_IDS)){
                xmlBuilder.append("</").append(SHAREHOLD_IMG).append(">");
            }else{
                xmlBuilder.append("</").append(key).append(">");
            }

        }
        xmlBuilder.append("</").append(key).append(">");
    }




    /**
     * 请求转map对象
     * @param object
     * @return {@link Map}<{@link String}, {@link Object}>
     */
    public static  Map<String, Object> ObjectToMap(Object object){

        String jsonString = JSON.toJSONString(object, SerializerFeature.WriteMapNullValue);
        Map<String, Object> resultMap = JSON.parseObject(jsonString, new TypeReference<TreeMap<String, Object>>() {});
        return resultMap;
    }

    /**
     * 请求转map对象
     * @param object
     * @return {@link Map}<{@link String}, {@link Object}>
     */
    public static  Map<String, Object> ObjectToMapFgj(Object object){

        String jsonString = JSON.toJSONString(object, SerializerFeature.WriteMapNullValue);
        //这里只能特殊处理一下了
        if(jsonString.contains("insCd")){
            jsonString = jsonString.replace("insCd","ins_cd");
        }
        Map<String, Object> resultMap = JSON.parseObject(jsonString, new TypeReference<TreeMap<String, Object>>() {});
        return resultMap;
    }


    /**
     * 根据类获取需要过滤的字段
     * @param clazz
     * @return {@link String[]}
     */
    public static String[] getNoNeedSign(Class<?> clazz) {
        List<String> noSignFields = new ArrayList<>();
        processFields(clazz, noSignFields);
        return noSignFields.toArray(new String[0]);
    }

    /**
     * 获取无须注解的字段
     * @param clazz
     * @param noSignFields
     */
    private static void processFields(Class<?> clazz, List<String> noSignFields) {
        // 获取所有字段，包括父类的字段
        List<Field> allFields = getAllFields(clazz);

        // 遍历所有字段，检查是否带有 @NoSign 注解
        for (Field field : allFields) {
            if (field.isAnnotationPresent(NoSign.class)) {
                noSignFields.add(field.getName());
            }

            // 如果字段是 List 类型，且泛型是对象类型，则递归处理该对象类型
            if (List.class.isAssignableFrom(field.getType())) {
                Class<?> genericType = getGenericType(field);
                if (genericType != null) {
                    processFields(genericType, noSignFields);
                }
            }
        }
    }

    private static List<Field> getAllFields(Class<?> clazz) {
        List<Field> fields = new ArrayList<>();

        // 遍历当前类及其父类，获取所有字段
        while (clazz != null) {
            for (Field field : clazz.getDeclaredFields()) {
                fields.add(field);
            }
            clazz = clazz.getSuperclass();
        }

        return fields;
    }

    /**
     * 获取泛型的实际类型
     * @param field
     * @return {@link Class}<{@link ?}>
     */
    private static Class<?> getGenericType(Field field) {
        // 获取字段的泛型类型
        java.lang.reflect.Type genericType = field.getGenericType();
        if (genericType instanceof java.lang.reflect.ParameterizedType) {
            java.lang.reflect.ParameterizedType paramType = (java.lang.reflect.ParameterizedType) genericType;
            java.lang.reflect.Type[] actualTypes = paramType.getActualTypeArguments();
            if (actualTypes.length > 0 && actualTypes[0] instanceof Class) {
                return (Class<?>) actualTypes[0];
            }
        }
        return null;
    }


    public static void main(String[] args) {
        // 示例用法
        Map<String, String> data = new HashMap<>();
        data.put("ins_cd", "08A9999999");
        data.put("mchnt_cd", "0002900F0370542");
        data.put("term_id", "12345678");
        data.put("random_str", "d0194c1024f180065d2434fa8b6a2f82");
        data.put("txn_begin_ts", "20201201151802");
        data.put("order_type", "WECHAT");
        String xml = generateXml(data);
        System.out.println(xml);
    }
}

