/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.steward.component;

import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.URLUtil;
import cn.hutool.core.util.XmlUtil;
import com.fshows.fuiou.util.FuiouRequestUtils;
import com.fshows.fuiou.util.LogUtil;
import com.fshows.fuiou.util.RsaUtils;
import com.fshows.fuiou.util.XmlConvertUtil;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.steward.apienum.FuStewardApiDefinitionEnum;
import com.fshows.steward.enums.FuStewardApiTypeEnum;
import com.fshows.steward.request.FuStewardBaseRequest;
import com.fshows.steward.response.FuStewardBaseResponse;
import com.fshows.steward.response.FuStewardBizResponse;
import lombok.extern.slf4j.Slf4j;
import java.util.HashMap;
import java.util.Map;

/**
 * 序列化处理
 *
 * @author liluqing
 * @version FuStewardSerializableHandler.java, v 0.1 2023-12-25 15:49
 */
@Slf4j
public class FuStewardSerializableHandler implements ISerializableHandler {

    @Override
    public String serializeObject(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        if (!(requestContext.getIApiDefinition() instanceof  FuStewardApiDefinitionEnum)) {
            throw new FsApiException("参数序列化失败");
        }
        FuStewardApiDefinitionEnum apiDefinitionEnum = (FuStewardApiDefinitionEnum) requestContext.getIApiDefinition();
        if (FuStewardApiTypeEnum.ENCRYPT_API.equals(apiDefinitionEnum.getApiTypeEnum())) {
            return encryptApiSerializeObject(apiRequestModel, requestContext);
        }
        return normalApiSerializeObject(apiRequestModel, requestContext);
    }

    /**
     * 普通接口参数序列化
     *
     * @return
     */
    private String normalApiSerializeObject(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        String xmlStr = FuiouRequestUtils.generateXmlByObjectMap(apiRequestModel.getParamMap());

        String enCodeStr = URLUtil.encodeAll(xmlStr, CharsetUtil.CHARSET_GBK);
        // 生成表单参数
        Map<String, String> params = new HashMap<>();
        params.put("req",enCodeStr);
        apiRequestModel.setRequestForm(params);
        return enCodeStr;
    }

    /**
     *  加密接口参数序列化
     *
     * @return
     */
    private String encryptApiSerializeObject(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {

        FuStewardBaseRequest request = (FuStewardBaseRequest) apiRequestModel.getRequest();
        // 客户端配置信息
        DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig(); 
        // 生成xml格式报文
        String xmlStr = FuiouRequestUtils.generateXmlByObjectMap(apiRequestModel.getParamMap());
        LogUtil.info(log, "{} >> 请求参数加密 >> url={}, method={}, 请求明文xmlStr={}", requestContext.getClientInfoModel().getClientDesc(), apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), xmlStr);
        // xml报文加密
        String encrypt = RsaUtils.encrypt(xmlStr, apiClientConfig.getPayCompanyPublicKey(), CharsetUtil.CHARSET_GBK);
        // 组装参数
        Map<String, String> params = new HashMap<>();
        params.put("mchntCd", request.getMchntCd());
        params.put("message", encrypt);
        // form表单提交
        apiRequestModel.setRequestForm(params);
        return null;
    }

    @Override
    public IResponseDefinition deserializationResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        if (!(requestContext.getIApiDefinition() instanceof  FuStewardApiDefinitionEnum)) {
            throw new FsApiException("参数序列化失败");
        }
        FuStewardApiDefinitionEnum apiDefinitionEnum = (FuStewardApiDefinitionEnum) requestContext.getIApiDefinition();
        if (FuStewardApiTypeEnum.ENCRYPT_API.equals(apiDefinitionEnum.getApiTypeEnum())) {
            return encryptApiDeserializationResponse(apiResponseModel, apiRequestModel, requestContext);
        }
        return normalApiDeserializationResponse(apiResponseModel, apiRequestModel, requestContext);
    }

    /**
     * 普通接口反序列化
     *
     * @return
     */
    private IResponseDefinition normalApiDeserializationResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext)  {
        String deCodeStr = null;
        try {
            FuStewardBaseResponse baseResponse = new FuStewardBaseResponse();
            deCodeStr = URLUtil.decode(apiResponseModel.getResponseBody(),"GBK");

            // 业务参数
            FuStewardBizResponse bizResponse = XmlConvertUtil.xml2Bean(deCodeStr, requestContext.getIApiDefinition().getResponseClass());
            baseResponse.setData(bizResponse);

            // 签名并不参与验签
            baseResponse.setMchntCd(bizResponse.getMchntCd());
            apiResponseModel.setResponseSign(bizResponse.getSignature());

            // 存放返回参数map
            apiResponseModel.setResponseMap(FuiouRequestUtils.ObjectToMap(bizResponse));
            return baseResponse;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk-fusteward】响应结果反序列化异常 >> deCodeResBody={},  originalResBody={}", e, deCodeStr, apiResponseModel.getResponseBody());
            throw new FsApiException("[fuiou-sdk-fusteward]响应结果反序列化异常");
        }
    }

    /**
     * 加密接口反序列化
     *
     * @return
     */
    private IResponseDefinition encryptApiDeserializationResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        // 业务参数
        try {
            FuStewardBaseResponse baseResponse = new FuStewardBaseResponse();
            DefaultClientConfigModel configModel = requestContext.getApiClientConfig();
            // xml 转 map
            Map<String, Object> resMap =  XmlUtil.xmlToMap(apiResponseModel.getResponseBody());

            // 商户号8
            String mchntCd = resMap.get("mchntCd").toString();
            baseResponse.setMchntCd(mchntCd);
            Object messageObj = resMap.get("message");
            if (messageObj == null) {
                Object respDescObj = resMap.get("respDesc");
                if (respDescObj != null) {
                    throw new FsApiException("[fuiou-sdk-fusteward]富友接口处理异常：" + respDescObj);
                }
                throw new FsApiException("[fuiou-sdk-fusteward]富友响应结果为空");
            }

            // message解密
            String encrypt = RsaUtils.decrypt(messageObj.toString(), configModel.getFubeiPrivateKey(), CharsetUtil.CHARSET_GBK);

            LogUtil.info(log, "{} >> 响应结果解密 >> url={}, method={}, 响应密文={}, 响应明文={}",
                    requestContext.getClientInfoModel().getClientDesc(), apiRequestModel.getApiURL(), requestContext.getIApiDefinition(),
                    messageObj, encrypt);
            FuStewardBizResponse bizResponse = XmlConvertUtil.xml2Bean(encrypt, requestContext.getIApiDefinition().getResponseClass());

            // 签名不参与验签
            apiResponseModel.setResponseSign(bizResponse.getSignature());
            apiResponseModel.setResponseMap(FuiouRequestUtils.ObjectToMap(bizResponse));

            baseResponse.setData(bizResponse);
            return baseResponse;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk-fusteward】响应结果反序列化异常 >> resBody={}", e, apiResponseModel.getResponseBody());
            throw new FsApiException("[fuiou-sdk-fusteward]响应结果反序列化异常");
        }
    }


}