/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows.steward.component;

import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.URLUtil;
import cn.hutool.core.util.XmlUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fuiou.exception.FuiouApiException;
import com.fshows.fuiou.response.base.FuiouBizResponse;
import com.fshows.fuiou.response.settlement.FuiouQuerySettlementResponse;
import com.fshows.fuiou.util.FuiouRequestUtils;
import com.fshows.fuiou.util.LogUtil;
import com.fshows.fuiou.util.RsaUtils;
import com.fshows.fuiou.util.XmlConvertUtil;
import com.fshows.sdk.core.client.base.definition.IResponseDefinition;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.tempalte.nest.request.NestBaseRequest;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.steward.apienum.FuStewardApiDefinitionEnum;
import com.fshows.steward.enums.FuStewardApiTypeEnum;
import com.fshows.steward.request.FuStewardBaseRequest;
import com.fshows.steward.response.FuStewardBaseResponse;
import com.fshows.steward.response.FuStewardBizResponse;
import lombok.extern.slf4j.Slf4j;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * 序列化处理
 *
 * @author liluqing
 * @version FuStewardSerializableHandler.java, v 0.1 2023-12-25 15:49
 */
@Slf4j
public class FuStewardSerializableHandler implements ISerializableHandler {

    @Override
    public String serializeObject(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        if (!(requestContext.getIApiDefinition() instanceof  FuStewardApiDefinitionEnum)) {
            throw new FsApiException("参数序列化失败");
        }
        FuStewardApiDefinitionEnum apiDefinitionEnum = (FuStewardApiDefinitionEnum) requestContext.getIApiDefinition();
        if (FuStewardApiTypeEnum.ENCRYPT_API.equals(apiDefinitionEnum.getApiTypeEnum())) {
            return encryptApiSerializeObject(apiRequestModel, requestContext);
        }
        return normalApiSerializeObject(apiRequestModel, requestContext);
    }

    /**
     * 普通接口参数序列化
     *
     * @return
     */
    private String normalApiSerializeObject(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        String xmlStr = FuiouRequestUtils.generateXmlByObjectMap(apiRequestModel.getParamMap());
        String enCodeStr = URLUtil.encodeAll(xmlStr, CharsetUtil.CHARSET_GBK);
        // 生成表单参数
        Map<String, String> params = new HashMap<>();
        params.put("req",enCodeStr);
        apiRequestModel.setRequestForm(params);
        return enCodeStr;
    }

    /**
     *  加密接口参数序列化
     *
     * @return
     */
    private String encryptApiSerializeObject(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {

        FuStewardBaseRequest request = (FuStewardBaseRequest) apiRequestModel.getRequest();
        // 客户端配置信息
        DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig(); 
        // 生成xml格式报文
        String xmlStr = FuiouRequestUtils.generateXmlByObjectMap(apiRequestModel.getParamMap());
        // xml报文加密
        String encrypt = RsaUtils.encrypt(xmlStr, apiClientConfig.getPayCompanyPublicKey());
        // 组装参数
        Map<String, String> params = new HashMap<>();
        params.put("mchntCd", request.getMchntCd());
        params.put("message", encrypt);
        return JSON.toJSONString(params);
    }

    @Override
    public IResponseDefinition deserializationResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        if (!(requestContext.getIApiDefinition() instanceof  FuStewardApiDefinitionEnum)) {
            throw new FsApiException("参数序列化失败");
        }
        FuStewardApiDefinitionEnum apiDefinitionEnum = (FuStewardApiDefinitionEnum) requestContext.getIApiDefinition();
        if (FuStewardApiTypeEnum.ENCRYPT_API.equals(apiDefinitionEnum.getApiTypeEnum())) {
            return encryptApiDeserializationResponse(apiResponseModel, apiRequestModel, requestContext);
        }
        return normalApiDeserializationResponse(apiResponseModel, apiRequestModel, requestContext);
    }

    /**
     * 普通接口反序列化
     *
     * @return
     */
    private IResponseDefinition normalApiDeserializationResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext)  {
        try {
            FuStewardBaseResponse baseResponse = new FuStewardBaseResponse();
            String deCodeStr = URLUtil.decode(apiResponseModel.getResponseBody(),"GBK");

            // 业务参数
            FuStewardBizResponse bizResponse = XmlConvertUtil.xml2Bean(deCodeStr, requestContext.getIApiDefinition().getResponseClass());
            baseResponse.setData(bizResponse);

            // 签名并不参与验签
            baseResponse.setMchntCd(bizResponse.getMchntCd());
            apiResponseModel.setResponseSign(bizResponse.getSignature());

            // 存放返回参数map
            apiResponseModel.setResponseMap(FuiouRequestUtils.ObjectToMap(bizResponse));
            return baseResponse;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk-fusteward】响应结果反序列化异常 >> resBody={}", e, apiResponseModel.getResponseBody());
            throw new FsApiException("[fuiou-sdk-fusteward]响应结果反序列化异常");
        }
    }

    public static void main(String[] args) {
        String xml = "<xml><count>1</count><ins_cd>08M0031681</ins_cd><mchnt_cd>0003310F6813298</mchnt_cd><random_str>4ZSSIW7I5QPZZ5F69XVJPCKVHWDYLUIU</random_str><result_code>000000</result_code><result_msg>SUCCESS</result_msg><sign>Z9Y4NlhB2EG2quzJPY2uBoGHl5V1uswOmR4BD0byUJUWRuVocr/Z+Kh79VkQTsoeigMs9xv472ds6IJp6iPTuHqni/V8Vhck0tiNspBO3xuCy4T+Ov/E/WqQfZtPAteJtwjb8dJLLXUFv0B1QmaP20WcXEI1Z0QJQt7B5FnTsFw=</sign><txnList><txn><acnt_nm>王青</acnt_nm><acnt_no>623091***4142</acnt_no><dest_txn_amt>149</dest_txn_amt><iss_bank_nm>中国邮政储蓄银行总行</iss_bank_nm><kbps_src_settle_dt>20240118</kbps_src_settle_dt><kbps_trace_no>181100857374</kbps_trace_no><not_settle_amt>0</not_settle_amt><out_order_no>20240118110029512320FUS</out_order_no><pay_msg></pay_msg><pay_st>1</pay_st><pay_st_desc>成功</pay_st_desc><src_mchnt_cd>0003310F6813298</src_mchnt_cd><txn_fee_amt>0</txn_fee_amt><txn_rcv_ts>20240118110030</txn_rcv_ts></txn></txnList></xml>";
        Map<String, Object> resMap = XmlUtil.xmlToMap(xml);
        System.out.println(JSONObject.toJSONString(resMap, true));
        System.out.println(JSONObject.parseObject(JSONObject.toJSONString(resMap), FuiouQuerySettlementResponse.class));

        try {
            FuiouQuerySettlementResponse response = XmlConvertUtil.xml2Bean(xml, FuiouQuerySettlementResponse.class);
            System.out.println(response);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

    }

    /**
     * 加密接口反序列化
     *
     * @return
     */
    private IResponseDefinition encryptApiDeserializationResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        FuStewardBaseResponse baseResponse = new FuStewardBaseResponse();

        DefaultClientConfigModel configModel = requestContext.getApiClientConfig();
        // xml 转 map
        Map<String, Object> resMap = null;
        try {
            resMap =  XmlUtil.xmlToMap(apiResponseModel.getResponseBody());
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk-fusteward】响应结果反序列化异常 >> resBody={}", e, apiResponseModel.getResponseBody());
            throw new FsApiException("[fuiou-sdk-fusteward]响应结果反序列化异常");
        }
        // 商户号
        String mchntCd = resMap.get("mchntCd").toString();
        baseResponse.setMchntCd(mchntCd);
        String message = resMap.get("message").toString();

        // message解密
        String encrypt = RsaUtils.decrypt(message, configModel.getFubeiPrivateKey());

        // xml 转 map
        Map<String, String> bizMap = null;
        try {
            bizMap = FuiouRequestUtils.getMapFromXML(encrypt);
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk-fusteward】响应结果反序列化异常 >> resBody={}", e, encrypt);
            throw new FsApiException("[fuiou-sdk-fusteward]响应结果反序列化异常");
        }
        // 业务参数转换
        FuStewardBizResponse bizResponse = FuiouRequestUtils.mapToObject(resMap, requestContext.getIApiDefinition().getResponseClass());

        // 签名不参与验签
        String signature = bizMap.remove("signature");
        apiResponseModel.setResponseSign(signature);
        baseResponse.setData(bizResponse);
        return baseResponse;
    }


}