/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.fuiou.client.impl;

import com.fshows.fuiou.client.base.ApiClientConfig;
import com.fshows.fuiou.client.base.IHttpRequest;
import com.fshows.fuiou.client.base.IFuiouApiClient;
import com.fshows.fuiou.client.base.IFuiouApiDefinition;
import com.fshows.fuiou.client.base.ISigner;
import com.fshows.fuiou.client.base.FuiouHttpResult;
import com.fshows.fuiou.client.base.SignParam;
import com.fshows.fuiou.exception.FuiouApiException;
import com.fshows.fuiou.request.base.FuiouBizRequest;
import com.fshows.fuiou.response.base.FuiouBizResponse;
import com.fshows.fuiou.util.LogUtil;
import com.fshows.fuiou.util.ValidateUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;

/**
 * 默认富友api客户端实现
 *
 * @author liluqing
 * @version AbstractFuiouApiClientImpl.java, v 0.1 2022-03-02 11:59
 */
@Setter
@Slf4j
public abstract class AbstractFuiouApiClientImpl implements IFuiouApiClient {

    /**
     * 客户端配置
     */
    protected ApiClientConfig apiClientConfig;

    /**
     * http请求器
     */
    protected IHttpRequest httpRequest;

    /**
     * 加签
     */
    protected ISigner signer;

    public AbstractFuiouApiClientImpl(ApiClientConfig apiClientConfig) throws FuiouApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = new DefaultHttpRequestImpl();
        this.signer = new DefaultSignerImpl();
    }

    public AbstractFuiouApiClientImpl(ApiClientConfig apiClientConfig, IHttpRequest httpRequest, ISigner signer) throws FuiouApiException {
        ValidateUtil.validateWithThrow(apiClientConfig);
        this.apiClientConfig = apiClientConfig;
        this.httpRequest = httpRequest;
        this.signer = signer;
    }

    /**
     * 执行富友请求
     *
     * @param bizRequest
     * @param tradeApiDefinitionEnum
     * @return
     * @throws FuiouApiException
     */
    protected FuiouBizResponse doExecute(FuiouBizRequest bizRequest, IFuiouApiDefinition tradeApiDefinitionEnum) throws FuiouApiException {
        try {
            // 参数校验
            checkParam(bizRequest, tradeApiDefinitionEnum);
            // 补充全局参数
            buildFuiouBizRequest(bizRequest, tradeApiDefinitionEnum);
            // 构建签名参数
            SignParam signParam = buildSignParam(bizRequest);
            // 获取签名
            String authorization = signer.sign(signParam, this.apiClientConfig);
            // 设置签名
            bizRequest.setSign(authorization);
            // 序列化请求参数
            String body = serializableRequest(bizRequest, authorization, tradeApiDefinitionEnum);
            // 执行加签操作获取签名
            String url = getServerURL(tradeApiDefinitionEnum);
            // 执行HTTP post请求
            FuiouHttpResult httpResult = requestPost(url, authorization, body);
            // 反序列化响应结果
            FuiouBizResponse response = parseResponse(httpResult.getBody(), tradeApiDefinitionEnum);
            LogUtil.info(log, "【fuiou-sdk】响应结果映射结束 >> url={}, request={}, response={}", url, bizRequest, response);
            // 根据是否验签配置判断是否需要验签
            // 只有响应成功才进行验签处理，因为富友在响应失败时不会返回签名
            if (this.apiClientConfig.isVrifySignResponse() && response != null
                    && StringUtils.equals(response.getResultCode(), "000000")) {
                boolean bo = signer.verifySign(response.getResMap(), null, this.apiClientConfig);
                if (!bo) {
                    LogUtil.error(log, "【fuiou-sdk】响应结果验签失败 >> httpResult={}", httpResult);
                    throw new FuiouApiException("[fuiou-sdk]响应结果验签失败");
                }
            }
            return response;
        } catch (FuiouApiException e) {
            LogUtil.error(log, "【fuiou-sdk】富友请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】富友请求异常 >> tradeApiDefinition={}, bizRequest={}", e, tradeApiDefinitionEnum, bizRequest);
            throw new FuiouApiException(e.getMessage(), e);
        }
    }

    /**
     * 执行post请求
     *
     * @param url
     * @param authorization
     * @param requestBody
     * @return
     */
    protected FuiouHttpResult requestPost(String url, String authorization, String requestBody) throws IOException, FuiouApiException {
        long beginTime = System.currentTimeMillis();
        try {
            LogUtil.info(log, "【fuiou-sdk】请求开始 >> url={}, authorization={}, request={}", url, authorization, requestBody);
            FuiouHttpResult httpResult = httpRequest.post(url, authorization, requestBody, apiClientConfig);
            if (httpResult == null) {
                throw new FuiouApiException("fuiou响应结果为空");
            }
            LogUtil.info(log, "【fuiou-sdk】请求结束 >> url={}, request={}, response={}, cost={}ms", url, requestBody,
                    httpResult.getBody(), System.currentTimeMillis() - beginTime);
            return httpResult;
        } catch (FuiouApiException e) {
            LogUtil.error(log, "【fuiou-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "【fuiou-sdk】网络请求异常 >> url={}, request={}, cost={}ms",
                    e, url, requestBody, (System.currentTimeMillis() - beginTime));
            throw new FuiouApiException("请求fuiou接口异常", e);
        }
    }

    /**
     * 请求参数校验
     *
     * @param bizRequest
     * @return
     */
    protected void checkParam(FuiouBizRequest bizRequest, IFuiouApiDefinition fuiouApiDefinition) throws FuiouApiException {
        ValidateUtil.notNull(bizRequest, "request请求参数不能为空");
        boolean checkRequestClass = StringUtils.equals(bizRequest.getClass().getCanonicalName(), fuiouApiDefinition.getRequestClass().getCanonicalName());
        if (!checkRequestClass) {
            throw new FuiouApiException("请求参数类型不正确");
        }
        // 当校验开关打开时才开启主动前置参数校验
        if (this.apiClientConfig.isCheckParam()) {
            ValidateUtil.validateWithThrow(bizRequest);
        }
    }

    /**
     * 获取请求地址
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected String getServerURL(IFuiouApiDefinition tradeApiDefinitionEnum) {
        return apiClientConfig.getApiParentURL() + tradeApiDefinitionEnum.getApiSubURI();
    }


    /**
     * 构建完整的请求参数
     *
     * @param request 业务请求参数
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract FuiouBizRequest buildFuiouBizRequest(
            FuiouBizRequest request,
            IFuiouApiDefinition tradeApiDefinitionEnum);

    /**
     * 构建加签的请求参数
     *
     * @param bizRequest 实际请求的完整参数
     * @return
     */
    protected abstract SignParam buildSignParam(
            FuiouBizRequest bizRequest);

    /**
     * 序列化请求参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract String serializableRequest(
            FuiouBizRequest fuiouBaseRequest,
            String sign,
            IFuiouApiDefinition tradeApiDefinitionEnum);

    /**
     * 解析响应参数
     *
     * @param tradeApiDefinitionEnum
     * @return
     */
    protected abstract FuiouBizResponse parseResponse(
            String body,
            IFuiouApiDefinition tradeApiDefinitionEnum) throws FuiouApiException;
}