/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package com.fshows.fuiou.client.impl;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fuiou.client.base.IFuiouUploadPicClient;
import com.fshows.fuiou.exception.FuiouApiException;
import com.fshows.fuiou.response.merchant.FuiouPicUploadResponse;
import com.fshows.fuiou.util.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Map;

/**
 * @author zhubo
 * @version FuiouUploadPicClientImpl.java, v 0.1 2023-09-20 19:40 zhubo
 */
@Slf4j
public class FuiouUploadPicClientImpl implements IFuiouUploadPicClient {

    private String insId;

    private String postUrl;

    private String fuiouMerchantPrivateKey;

    private String charset;

    public FuiouUploadPicClientImpl(String insId, String postUrl, String fuiouMerchantPrivateKey, String charset) {
        this.insId = insId;
        this.postUrl = postUrl;
        this.fuiouMerchantPrivateKey = fuiouMerchantPrivateKey;
        this.charset = charset;
    }

    @Override
    public FuiouPicUploadResponse uploadPic(Map<String, String> parameters, File file) throws FuiouApiException {
        String responseString = "";
        CloseableHttpClient httpclient = HttpClients.createDefault();
        if (null == parameters || parameters.isEmpty()) {
            return null;
        }

        parameters.put("ins_cd", this.insId);
        parameters.put("sign", getSign(parameters));
        try {
            HttpPost httpPost = new HttpPost(this.postUrl);
            MultipartEntityBuilder entity = MultipartEntityBuilder.create();
            // 请求参数
            for (String key : parameters.keySet()) {
                entity.addTextBody(key, parameters.get(key), ContentType.DEFAULT_TEXT);
            }
            entity.addPart("file", new FileBody(file));
            httpPost.setEntity(entity.build());
            RequestConfig requestConfig = RequestConfig.custom().setSocketTimeout(50000).setConnectTimeout(30000)
                    .build();// 设置请求和传输超时时间
            httpPost.setConfig(requestConfig);
            log.info("【fuiou-sdk】图片上传入参 >> request = " + httpPost.getRequestLine());
            CloseableHttpResponse response = httpclient.execute(httpPost);
            if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                HttpEntity httpEntity = response.getEntity();
                log.info("【fuiou-sdk】图片上传出参 >> response = " + response.getStatusLine());
                try {
                    if (httpEntity != null) {
                        responseString = EntityUtils.toString(httpEntity);
                        log.info("fuiou-sdk 图片上传出参 >> response content:" + responseString);
                        EntityUtils.consume(httpEntity);
                    }
                } finally {
                    response.close();
                }
            }
        } catch (Exception e) {
            log.error("http请求异常", e);
        } finally {
            if (httpclient != null) {
                try {
                    httpclient.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return StrUtil.isBlank(responseString) ? null : JSONObject.parseObject(responseString, FuiouPicUploadResponse.class);
    }

    private String getSign(Map<String, String> dataMap) throws FuiouApiException {
        try {
            String waitSignStr = getWaitSignStr(dataMap);
            LogUtil.info(log, "【fuiou-sdk】待加签字符串 >> waitSignStr={}", waitSignStr);
            //加签
            String md5Hex = DigestUtils.md5Hex(waitSignStr.getBytes(this.charset));
            return md5Hex.toLowerCase();
        } catch (UnsupportedEncodingException e) {
            LogUtil.error(log, "【fuiou-sdk】fuiou请求加签失败 >> signParam={}", e, dataMap);
            throw new FuiouApiException("请求加签失败", e);
        }
    }

    private String getWaitSignStr(Map<String, String> dataMap) {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry : dataMap.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (StringUtils.equals(key, "sign")
                    || StringUtils.startsWith(key, "reserved")) {
                continue;
            }
            if (StrUtil.isBlank(value)) {
                continue;
            }
            if (sb.length() > 0) {
                sb.append("&");
            }
            sb.append(key).append("=").append(value);
        }
        sb.append("&key=").append(this.fuiouMerchantPrivateKey);
        return sb.toString();
    }
}
