package com.frt.fubeiopenapisdk.client;

import cn.hutool.core.util.StrUtil;
import cn.hutool.http.Header;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.frt.fubeiopenapisdk.apienum.FubeiPayApiDefinitionEnum;
import com.frt.fubeiopenapisdk.client.model.FubeiApiRequestModel;
import com.frt.fubeiopenapisdk.client.model.FubeiPayClientConfigModel;
import com.frt.fubeiopenapisdk.constant.FubeiPayConstant;
import com.frt.fubeiopenapisdk.request.FubeiPayBizRequest;
import com.frt.fubeiopenapisdk.response.FubeiPayBizResponse;
import com.frt.fubeiopenapisdk.utils.MD5;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

@Slf4j
public class FubeiPayApiClient extends AbstractApiClient<FubeiPayBizRequest, FubeiPayBizResponse, FubeiPayApiDefinitionEnum> {

    public FubeiPayApiClient(FubeiPayClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
    }

    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName(FubeiPayConstant.ClientInfo.CLIENT_NAME);
        clientInfoModel.setClientCode(FubeiPayConstant.ClientInfo.CLIENT_CODE);
        return clientInfoModel;
    }

    @Override
    protected void checkParam(FubeiPayBizRequest bizRequest, DefaultRequestContext defaultRequestContext) throws FsApiException {
        super.checkParam(bizRequest, defaultRequestContext);
        if (StrUtil.isEmpty(bizRequest.getVendorSn())) {
            throw new FsApiException("请求异常vendorSn不能为空");
        }
        if (StrUtil.isEmpty(bizRequest.getAppSecret())) {
            throw new FsApiException("请求异常appSecret不能为空");
        }
    }

    @Override
    protected DefaultRequestContext buildRequestContext(FubeiPayApiDefinitionEnum iApiDefinition,
                                                        FubeiPayBizRequest request,
                                                        DefaultClientConfigModel merchantConfig
    ) {

        FubeiPayClientConfigModel clientConfig = (FubeiPayClientConfigModel) this.apiClientConfig;

        FubeiPayClientConfigModel config = new FubeiPayClientConfigModel();
        config.setApiParentURL(clientConfig.getApiParentURL());
        config.setCharset(clientConfig.getCharset());
        config.setTimeout(clientConfig.getTimeout());
        config.setReadTimeout(clientConfig.getReadTimeout());

        DefaultRequestContext context = new DefaultRequestContext();
        context.setIApiDefinition(iApiDefinition);
        context.setApiClientConfig(config);
        context.setClientInfoModel(getClientInfo());
        return context;
    }

    @Override
    protected ApiRequestModel buildApiRequestModel(FubeiPayBizRequest request, DefaultRequestContext context) {
        FubeiPayClientConfigModel configModel = (FubeiPayClientConfigModel) context.getApiClientConfig();
        FubeiApiRequestModel apiRequestModel = new FubeiApiRequestModel();
        apiRequestModel.setApiURL(configModel.getApiParentURL());
        apiRequestModel.setContentType(FubeiPayConstant.Http.CONTENT_TYPE_JSON);
        String requestJson = JSON.toJSONString(request);
        Map<String, String> map = JSON.parseObject(requestJson, new TypeReference<Map<String, String>>() {
        });
        apiRequestModel.setVendorSn(map.get("vendorSn"));
        apiRequestModel.setAppSecret(map.get("appSecret"));
        map.remove("appSecret");
        map.remove("vendorSn");
        apiRequestModel.setRequestBody(JSONUtil.toJsonStr(map));
        return apiRequestModel;
    }

    @Override
    protected FubeiPayBizResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
        JSONObject dataJson = null;
        try {
            String responseBody = apiResponseModel.getResponseBody();
            JSONObject resJson = JSONObject.parseObject(responseBody);
            try {
                dataJson = resJson.getJSONObject("data");
                resJson.remove("data");
                if (dataJson == null) {
                    dataJson = new JSONObject();
                }
                dataJson.putAll(resJson);
            } catch (ClassCastException e) {
                JSONArray jsonArray = resJson.getJSONArray("data");
                resJson.remove("data");
                if (dataJson == null) {
                    dataJson = new JSONObject();
                }
                dataJson.putAll(resJson);
                dataJson.put("list", jsonArray);
            }
            return JSONObject.parseObject(dataJson.toJSONString(), requestContext.getIApiDefinition().getResponseClass());
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 响应异常 >> iApiDefinition={}, request={}, 业务响应报文dataJson={}", e,
                    getClientInfo().getClientDesc(), requestContext.getIApiDefinition(), apiRequestModel.getRequest(), dataJson);
            throw new RuntimeException(e);
        }
    }

    @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {

        FubeiApiRequestModel fubeiApiRequestModel = (FubeiApiRequestModel) apiRequestModel;
        String appSecret = fubeiApiRequestModel.getAppSecret();
        String vendorSn = fubeiApiRequestModel.getVendorSn();

        JSONObject jsonParam = new JSONObject(new TreeMap<>());//组装公共参数
        jsonParam.put("nonce", System.currentTimeMillis());
        //服务商维度
        jsonParam.put("vendor_sn", vendorSn);
        //服务商维度
        jsonParam.put("version", "1.0");
        //调用方法名称
        jsonParam.put("method", requestContext.getIApiDefinition().getApiURI());
        jsonParam.put("format", "json");
        jsonParam.put("sign_method", "md5");
        jsonParam.put("biz_content", apiRequestModel.getRequestBody());
        Set<String> keys = jsonParam.keySet();
        StringBuilder signStr = new StringBuilder();
        for (String key : keys) {
            signStr.append(key).append("=").append(jsonParam.get(key)).append("&");
        }
        String signWithSecret = signStr.substring(0, signStr.length() - 1) + appSecret;
        String sign = MD5.sign(signWithSecret, "utf-8");
        jsonParam.put("sign", sign.toUpperCase());

        DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
        String body = HttpUtil.createPost(apiClientConfig.getApiParentURL())
                .header(Header.CONTENT_TYPE, apiRequestModel.getContentType())
                .charset(apiClientConfig.getCharset())
                .timeout(apiClientConfig.getReadTimeout())
                .body(JSONUtil.toJsonStr(jsonParam), apiRequestModel.getContentType())
                .execute().body();

        ApiResponseModel apiResponseModel = new ApiResponseModel();
        apiResponseModel.setResponseBody(body);
        return apiResponseModel;
    }

    @Override
    public FubeiPayBizResponse execute(FubeiPayBizRequest request, FubeiPayApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public FubeiPayBizResponse execute(FubeiPayBizRequest request, FubeiPayApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    @Override
    protected FubeiPayBizResponse doExecute(FubeiPayBizRequest request, FubeiPayApiDefinitionEnum iApiDefinition, DefaultClientConfigModel customConfig) throws FsApiException {
        long beginTime = System.currentTimeMillis();

        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, customConfig);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;

        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}", getClientInfo().getClientDesc(), iApiDefinition, apiRequestModel.getRequestBody());
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();
            // 构建响应数据
            FubeiPayBizResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);
            LogUtil.info(log, "{} >> 执行请求结束 >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), apiRequestModel.getRequestBody(),
                    JSONObject.toJSONString(response), System.currentTimeMillis() - beginTime, reqEndTime - reqBeginTime);
            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求业务异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求未知异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    iApiDefinition, JSONObject.toJSONString(request), JSONObject.toJSONString(apiRequestModel), JSONObject.toJSONString(apiResponseModel));
            throw new FsApiException("请求未知异常", e);
        }
    }
}