/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.sdk.core.util;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import com.fshows.sdk.core.exception.FsApiException;
import lombok.extern.slf4j.Slf4j;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author zhaoyi
 * @version LeshuaUtils.java, v 0.1 2020-07-28 09:47 zhaoyi
 */
@Slf4j
public class RequestParamUtils {

    /**
     * 缓存反射字段
     */
    private static final Map<Class, Field[]> CACHE_FIELD_MAP = new ConcurrentHashMap<>();

    private static final String XML = "xml";

    private static final String DEFAULT_CHARSET = "GBK";

    private static final String EMPTY = "";

    /**
     * 忽略字段
     */
    private static final String[] IGNORE_FIELD_ARR = {"serialVersionUID"};

    private static Pattern UNDERLINE_PATTERN = Pattern.compile("[A-Z]");

    public static InputStream getStringStream(String sInputString) throws UnsupportedEncodingException {
        ByteArrayInputStream tInputStringStream = null;
        if (sInputString != null && !sInputString.trim().equals(EMPTY)) {
            tInputStringStream = new ByteArrayInputStream(sInputString.getBytes(DEFAULT_CHARSET));
        }
        return tInputStringStream;
    }

    public static <Res> Res getObjectFromXML(String xml, Class<Res> tClass) throws Exception {
        Map<String, Object> mapFromXML = getMapFromXML(xml);
        return BeanUtil.toBean(mapFromXML, tClass);
    }

    /**
     * map 转对象
     *
     * @param mapFromXML
     * @param tClass
     * @param <Res>
     * @return
     * @throws Exception
     */
    public static <Res> Res getObjectFromMap(Map mapFromXML, Class<Res> tClass) {
        return BeanUtil.toBean(mapFromXML, tClass);
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值，值为null的属性排除
     *
     * @param obj
     * @return
     */
    public static Map<String, String> toMap(Object obj) {
        Field[] fields = CACHE_FIELD_MAP.get(obj.getClass());
        if (fields == null) {
            fields = ReflectUtil.getFields(obj.getClass());
            CACHE_FIELD_MAP.put(obj.getClass(), fields);
        }
        Map<String, String> map = new TreeMap<>();
        for (Field field : fields) {
            try {
                field.setAccessible(true);
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                if (ArrayUtil.contains(IGNORE_FIELD_ARR, field.getName())) {
                    continue;
                }
                String key = field.getName();
                Object val = field.get(obj);
                if (val != null) {
                    map.put(camelToUnderline(key), String.valueOf(val));
                } else {
                    map.put(camelToUnderline(key), null);
                }
            } catch (IllegalArgumentException | IllegalAccessException e) {
                log.error(e.getMessage(), e);
            }
        }
        return map;
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值，值为null的属性排除
     *
     * @param obj
     * @return
     */
    public static Map<String, String> toMap(Object obj, boolean isHump) {
        Field[] fields = CACHE_FIELD_MAP.get(obj.getClass());
        if (fields == null) {
            fields = ReflectUtil.getFields(obj.getClass());
            CACHE_FIELD_MAP.put(obj.getClass(), fields);
        }
        Map<String, String> map = new TreeMap<>();
        for (Field field : fields) {
            try {
                field.setAccessible(true);
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                if (ArrayUtil.contains(IGNORE_FIELD_ARR, field.getName())) {
                    continue;
                }
                String key = field.getName();
                Object val = field.get(obj);
                if (val != null) {
                    map.put(isHump ? key : camelToUnderline(key), String.valueOf(val));
                }
            } catch (IllegalArgumentException | IllegalAccessException e) {
                log.error(e.getMessage(), e);
            }
        }
        return map;
    }

    /**
     * 将对象转换成map，key为属性名，value为属性值，值为null的属性排除
     *
     * @param obj
     * @param isHump
     * @return
     */
    public static Map<String, Object> toMapObj(Object obj, boolean isHump, boolean isIgnoreNull) {
        Field[] fields = CACHE_FIELD_MAP.get(obj.getClass());
        if (fields == null) {
            fields = ReflectUtil.getFields(obj.getClass());
            CACHE_FIELD_MAP.put(obj.getClass(), fields);
        }
        Map<String, Object> map = new TreeMap<>();
        for (Field field : fields) {
            try {
                field.setAccessible(true);
                if (Modifier.isStatic(field.getModifiers())) {
                    continue;
                }
                if (ArrayUtil.contains(IGNORE_FIELD_ARR, field.getName())) {
                    continue;
                }
                String key = field.getName();
                Object val = field.get(obj);
                if ((val == null) && isIgnoreNull) {
                    continue;
                }
                map.put(isHump ? key : camelToUnderline(key), ObjectUtil.isNull(val) ?  "": val.toString());

            } catch (IllegalArgumentException | IllegalAccessException e) {
                log.error(e.getMessage(), e);
            }
        }
        return map;
    }

    /**
     * 计算失效时间
     *
     * @param now
     * @param expireInMinutes
     * @param dateFormat
     * @return
     */
    public static String calcExpireTime(Date now, String expireInMinutes, SimpleDateFormat dateFormat) {
        return dateFormat.format(now.getTime() + Long.parseLong(expireInMinutes) * 60 * 1000);
    }

    /**
     * 驼峰转下划线
     */
    public static String camelToUnderline(String str) {
        Matcher matcher = UNDERLINE_PATTERN.matcher(str);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, "_" + matcher.group(0).toLowerCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * xml报文转 map
     *
     * @param xmlString
     * @return
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    public static Map<String, Object> getMapFromXML(String xmlString) {
        try {
            //xmlString = xmlString.replaceFirst("encoding=\".*\"", "encoding=\"GBK\"");
            // 这里用Dom的方式解析回包的最主要目的是防止API新增回包字段
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            InputStream is = getStringStream(xmlString);
            Document document = builder.parse(is);

            // 获取到document里面的全部结点
            NodeList allNodes = document.getFirstChild().getChildNodes();
            Node node;
            Map<String, Object> map = new TreeMap<>();
            int i = 0;
            while (i < allNodes.getLength()) {
                node = allNodes.item(i);
                if (node instanceof Element) {
                    map.put(node.getNodeName(), node.getTextContent());
                }
                i++;
            }
            return map;
        } catch (Exception e) {
            throw new FsApiException("响应结果反序列化异常！", e);
        }
    }

    /**
     * 通过map生成xml
     *
     * @param data
     * @return
     */
    public static String generateXml(Map<String, String> data) {
        StringBuilder xmlBuilder = new StringBuilder();
        xmlBuilder.append("<?xml version=\"1.0\" encoding=\"GBK\" standalone=\"yes\"?>");
        xmlBuilder.append("<xml>");
        for (Map.Entry<String, String> entry : data.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (value != null) {
                xmlBuilder.append("<").append(key).append("><![CDATA[").append(value).append("]]></").append(key).append(">");
            }
        }
        xmlBuilder.append("</xml>");
        return xmlBuilder.toString();
    }

    public static void main(String[] args) {
        // 示例用法
        Map<String, String> data = new HashMap<>();
        data.put("ins_cd", "08A9999999");
        data.put("mchnt_cd", "0002900F0370542");
        data.put("term_id", "12345678");
        data.put("random_str", "d0194c1024f180065d2434fa8b6a2f82");
        data.put("txn_begin_ts", "20201201151802");
        data.put("order_type", "WECHAT");
        String xml = generateXml(data);
        System.out.println(xml);
    }
}

