/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.sdk.core.client.component.signer;


import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.SignerUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;

import java.util.Map;

/**
 * MD5加签默认实现（基于对于Map<String, Object> requestMap的加签验签实现）
 * （如果参与加签验签的属性只有一层，没有多层嵌套的结果，可直接使用该实现）
 *
 * @author liluqing
 * @version DefaultSignerImpl.java, v 0.1 2022-03-02 18:32
 */
@Slf4j
public class Md5SignHandlerImpl implements IApiSignHandler {

    public String sign(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws FsApiException {
        String waitSignStr = null;
        try {
            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
            // 获取待加签字符串
            Map<String, Object> data = apiRequestModel.getParamMap();
            // 获取待加签字符串
            waitSignStr = SignerUtil.getWaitSignStr(data, apiClientConfig.getMd5SignKey());
            LogUtil.info(log, "{} >> 待加签字符串 >> waitSignStr={}", requestContext.getClientInfoModel().getClientDesc() , waitSignStr);
            //加签
            String md5Hex = DigestUtils.md5Hex(waitSignStr.getBytes(apiClientConfig.getCharset()));
            return md5Hex.toLowerCase();
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求加签失败 >> apiRequestModel={}, waitSignStr={}", e, requestContext.getClientInfoModel().getClientDesc(), apiRequestModel, waitSignStr);
            throw new FsApiException("请求加签失败", e);
        }
    }

    public Boolean verifySign(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws FsApiException {
        String waitSignStr = null;
        try {
            DefaultClientConfigModel apiClientConfig = requestContext.getApiClientConfig();
            // 响应签名
            String resSign = apiResponseModel.getResponseSign();
            // 获取待加签字符串
            waitSignStr = SignerUtil.getWaitSignStr(apiResponseModel.getResponseMap(), apiClientConfig.getMd5SignKey());
            //加签
            String md5Hex = DigestUtils.md5Hex(waitSignStr.getBytes(apiClientConfig.getCharset())).toLowerCase();
            //加签
            return resSign.equalsIgnoreCase(md5Hex);
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 响应结果验签失败 >> apiRequestModel={}, waitSignStr={}", e, apiRequestModel, waitSignStr);
            throw new FsApiException("响应结果验签失败", e);
        }
    }
}