/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.fsframework.web.service;

import com.alibaba.csp.sentinel.Entry;
import com.alibaba.csp.sentinel.EntryType;
import com.alibaba.csp.sentinel.SphU;
import com.alibaba.csp.sentinel.slots.block.BlockException;
import com.fshows.fsframework.common.exception.CommonException;
import com.fshows.fsframework.core.BaseParam;
import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.web.bean.ApplicationContextHelper;
import com.fshows.fsframework.web.domain.ApiContainer;
import com.fshows.fsframework.web.domain.ApiDescriptor;
import com.fshows.fsframework.web.exception.ApiInvokeException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
 * 带限流器实现的api调用器
 *
 * @author liluqing
 * @version AbstractLimitApiInvokeService.java, v 0.1 2022-09-20 14:08
 */
@Component
@Slf4j
public abstract class SentinelLimitApiInvokeService {

    /**
     * API调用类
     */
    @Autowired
    protected IApiClient apiClient;

    @Autowired
    protected ApiContainer apiContainer;

    /**
     * 限流总开关
     */
    @Value("${rate.limit.switch:true}")
    private String rateLimitSwitch;

    /**
     * 调用API
     *
     * @param appId
     * @param sign
     * @param apiMethodName
     * @param content
     * @param <P>
     * @param <R>
     * @return
     */
    @SuppressWarnings("unchecked")
    public <P extends BaseParam, R> R invoke(String appId, String sign, String apiMethodName, Map<String, Object> params, String content) throws Throwable {
        // 获得方法信息
        ApiDescriptor apiDescriptor = apiContainer.get(apiMethodName);
        if (null == apiDescriptor) {
            throw ApiInvokeException.API_NOT_EXIST;
        }
        // 获得 bean
        String beanName = apiDescriptor.getBeanName();
        Object bean = ApplicationContextHelper.getBean(beanName);
        if (null == bean) {
            throw ApiInvokeException.API_NOT_EXIST;
        }
        Boolean hasPermission = checkPermission(appId, apiMethodName);
        if (!hasPermission) {
            throw ApiInvokeException.DO_NOT_HAS_PERMISSION;
        }
        Boolean checkSign = checkSign(params, appId);
        if (!checkSign) {
            throw ApiInvokeException.API_INVALID_SIGIN;
        }
        // 调用方法
        return doInvoke(appId, apiMethodName, content, (ApiInvoker<P, R>) (method, param) ->
                (R) apiDescriptor.getMethod().invoke(bean, param));
    }

    /**
     * 执行函数方法
     *
     * @param method
     * @param paramJsonStr
     * @param invoker
     * @param <P>
     * @param <R>
     * @return
     */
    protected <P extends BaseParam, R> R doInvoke(String appId, String method, String paramJsonStr, ApiInvoker<P, R> invoker) throws Throwable {
        // 是否开启限流
        boolean rateLimit = StringUtils.equalsIgnoreCase(this.rateLimitSwitch, StringPool.TRUE);
        Object[] hotParam = null;
        Entry entry = null;
        try {
            // 只有在限流总开关打开的情况下才开启限流
            if (rateLimit) {
                // 获取热点参数
                hotParam = getHotParam(appId, method, paramJsonStr);
                if (hotParam == null) {
                    hotParam = new Object[0];
                }
                // 获取限流对象
                entry = SphU.entry(method, EntryType.IN, 1, hotParam);
            }
            return apiClient.invoke(method, paramJsonStr, invoker);
        } catch (BlockException ex) {
            LogUtil.warn(log, "网关请求限流 >> method={}, hotParam={}", ex, method, hotParam);
            throw CommonException.RATE_LIMIT_EXCEPTION;
        } finally {
            // 资源退出
            if (rateLimit && entry != null) {
                entry.exit(1, hotParam);
            }
        }
    }

    /**
     * 获取热点参数
     *
     * @param appId
     * @param apiMethodName
     * @return
     */
    protected abstract Object[] getHotParam(String appId, String apiMethodName, String paramJsonStr);


    /**
     * 权限验证
     *
     * @param appId
     * @param apiMethodName
     * @return
     */
    protected abstract boolean checkPermission(String appId, String apiMethodName);

    /**
     * 签名检查
     *
     * @param params
     * @param appId
     * @return
     */
    protected abstract boolean checkSign(Map<String, Object> params, String appId);
}