/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.oss;

import com.alibaba.fastjson.JSON;
import com.aliyun.oss.ClientConfiguration;
import com.aliyun.oss.ClientException;
import com.aliyun.oss.HttpMethod;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.comm.Protocol;
import com.aliyun.oss.model.GeneratePresignedUrlRequest;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectResult;
import com.fshows.fsframework.common.exception.CommonException;
import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.io.File;
import java.net.URL;
import java.util.Date;

import static com.aliyun.oss.internal.OSSConstants.DEFAULT_OBJECT_CONTENT_TYPE;

/**
 * @author liujing01
 * @version AliyunOssUtil.java, v 0.1 2018-12-19 15:23
 */
@Slf4j
public class AliyunOssUtil {
    private static final String BACKSLASH = "/";
    /**
     * 使用volatile关键字保其可见性
     */
    private static volatile AliyunOssUtil instance = null;
    private OSSClient ossClient;
    private String bucketName;
    /**
     * oss 外网地址
     */
    private static final String OSS_ENDPOINT_OUT = "http://oss-cn-hangzhou.aliyuncs.com";
    /**
     * oss 内网地址
     */
    private static final String OSS_ENDPOINT_INNER = "http://oss-cn-hangzhou-internal.aliyuncs.com";


    private AliyunOssUtil() {
    }

    private AliyunOssUtil(String endpoint, String bucketName, String accessKey, String secretKey) {
        this.bucketName = bucketName;
        // 创建ClientConfiguration。ClientConfiguration是OSSClient的配置类，可配置代理、连接超时、最大连接数等参数。
        ClientConfiguration conf = new ClientConfiguration();
        // 设置OSSClient允许打开的最大HTTP连接数，默认为1024个。
        conf.setMaxConnections(200);
        // 设置Socket层传输数据的超时时间，默认为50000毫秒。
        conf.setSocketTimeout(10000);
        // 设置建立连接的超时时间，默认为50000毫秒。
        conf.setConnectionTimeout(10000);
        // 设置从连接池中获取连接的超时时间（单位：毫秒），默认不超时。
        conf.setConnectionRequestTimeout(1000);
        // 设置连接空闲超时时间。超时则关闭连接，默认为60000毫秒。
        conf.setIdleConnectionTime(10000);
        // 设置失败请求重试次数，默认为3次。
        conf.setMaxErrorRetry(3);
        // 设置是否支持将自定义域名作为Endpoint，默认支持。
        conf.setSupportCname(true);
        // 设置是否开启二级域名的访问方式，默认不开启。
        conf.setSLDEnabled(false);
        // 设置连接OSS所使用的协议（HTTP/HTTPS），默认为HTTP。
        conf.setProtocol(Protocol.HTTP);
        ossClient = new OSSClient(endpoint, accessKey, secretKey, conf);
    }

    public static AliyunOssUtil getInstance(String endpoint, String bucketName, String accessKey, String secretKey) {
        if (instance == null) {
            synchronized (AliyunOssUtil.class) {
                if (instance == null) {
                    instance = new AliyunOssUtil(endpoint, bucketName, accessKey, secretKey);
                }
            }
        }
        return instance;
    }

    /**
     * 文件上传
     *
     * @param folder demo:"/a/b/c" or "a/b/c"
     * @param file   最终生成文件 "/a/b/c/file.getName()"
     * @return
     */
    public boolean uploadFile(String folder, File file) {
        if (file == null) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("file must be not null");
        }
        if (StringUtils.isEmpty(folder)) {
            LogUtil.warn(log, "oss上传的objectName:{},localFile：{}", folder, file.getName());
            throw CommonException.INVALID_PARAM_ERROR.newInstance("folder param must be not null");
        }
        if (file.isDirectory()) {
            LogUtil.warn(log, "oss上传的objectName:{},localFile：{}", folder, file.getName());
            throw CommonException.INVALID_PARAM_ERROR.newInstance("file must be one file");
        }
        long now = SystemClock.millisClock().now();
        String objectName = folder.lastIndexOf("/") != -1 ? folder : folder + "/";
        objectName = checkObjectName(objectName);
        objectName = objectName + file.getName();
        LogUtil.info(log, "oss上传的objectName:{},localFile：{}", objectName, file.getAbsolutePath());
        PutObjectResult result = ossClient.putObject(bucketName, objectName, file);
        LogUtil.info(log, "upload objectName:{},oss返回结果:{},耗时:{}ms", objectName, JSON.toJSONString(result), (SystemClock.millisClock().now() - now));
        if (result != null) {
            return true;
        }
        return false;
    }

    /**
     * 上传文件
     *
     * @param folder    demo:"/a/b/c" or "a/b/c"
     * @param localFile 最终生成文件 "/a/b/c/file.getName()"
     * @return
     */
    public boolean uploadFile(String folder, String localFile) {
        if (StringUtils.isEmpty(folder)) {
            LogUtil.warn(log, "oss上传的objectName:{},localFile：{}", folder, localFile);
            throw CommonException.INVALID_PARAM_ERROR.newInstance("folder param must be not null");
        }
        if (StringUtils.isEmpty(localFile)) {
            LogUtil.warn(log, "oss上传的objectName:{},localFile：{}", folder, localFile);
            throw CommonException.INVALID_PARAM_ERROR.newInstance("localFile param must be not null");
        }
        File file = new File(localFile);
        if (file.isDirectory()) {
            LogUtil.warn(log, "oss上传的objectName:{},localFile：{}", folder, localFile);
            throw CommonException.INVALID_PARAM_ERROR.newInstance("localFile must be one file");
        }
        long now = SystemClock.millisClock().now();
        String objectName = folder.lastIndexOf("/") != -1 ? folder : folder + "/";
        objectName = checkObjectName(objectName);
        objectName = objectName + file.getName();
        LogUtil.info(log, "oss上传的objectName:{},localFile：{}", objectName, localFile);
        PutObjectResult result = ossClient.putObject(bucketName, objectName, file);
        LogUtil.info(log, "upload objectName:{},oss返回结果:{},耗时:{}ms", objectName, JSON.toJSONString(result), (SystemClock.millisClock().now() - now));
        if (result != null) {
            return true;
        }
        return false;
    }

    /**
     * 下载文件
     *
     * @param objectName
     * @param localFile
     * @return
     */
    public boolean downloadFile(String objectName, String localFile) {
        if (StringUtils.isEmpty(objectName)) {
            LogUtil.warn(log, "oss下载的objectName:{},localFile：{}", objectName, localFile);
            throw CommonException.INVALID_PARAM_ERROR.newInstance("objectName param must be not null");
        }
        if (StringUtils.isEmpty(localFile)) {
            LogUtil.warn(log, "oss下载的objectName:{},localFile：{}", objectName, localFile);
            throw CommonException.INVALID_PARAM_ERROR.newInstance("localFile param must be not null");
        }
        objectName = checkObjectName(objectName);
        long now = SystemClock.millisClock().now();
        ObjectMetadata metadata = ossClient.getObject(new GetObjectRequest(bucketName, objectName), new File(localFile));
        LogUtil.info(log, "download objectName:{},Request-ID:{},耗时:{}ms", objectName, metadata.getRequestId(), (SystemClock.millisClock().now() - now));
        if (metadata != null) {
            return true;
        }
        return false;

    }

    /**
     * 文件下载
     *
     * @param objectName
     * @param file
     * @return
     */
    public boolean downloadFile(String objectName, File file) {
        if (file == null) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("file must be not null");
        }
        if (StringUtils.isEmpty(objectName)) {
            LogUtil.warn(log, "oss下载的objectName:{},localFile：{}", objectName, file.getName());
            throw CommonException.INVALID_PARAM_ERROR.newInstance("objectName param must be not null");
        }
        objectName = checkObjectName(objectName);
        long now = SystemClock.millisClock().now();
        ObjectMetadata metadata = ossClient.getObject(new GetObjectRequest(bucketName, objectName), file);
        LogUtil.info(log, "download objectName:{},Request-ID:{},耗时:{}ms", objectName, metadata.getRequestId(), (SystemClock.millisClock().now() - now));
        if (metadata != null) {
            return true;
        }
        return false;
    }

    private String checkObjectName(String objectName) {
        if (objectName.startsWith(BACKSLASH)) {
            objectName = objectName.substring(1, objectName.length());
        }
        return objectName;
    }

    /**
     * 删除文件
     *
     * @param objectName
     */
    public void deleteFile(String objectName) {
        if (StringUtils.isEmpty(objectName)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("objectName param must be not null");
        }
        objectName = checkObjectName(objectName);
        long now = SystemClock.millisClock().now();
        ossClient.deleteObject(bucketName, objectName);
        LogUtil.info(log, "delete objectName:{},耗时:{}ms", objectName, (SystemClock.millisClock().now() - now));
    }


    /**
     * 生成下载url get的方式访问
     *
     * @param key           OSS里面的文件路径+文件名
     * @param expireSeconds 当前时间+过期秒数
     * @return
     */
    public String getExpireUrl(String key, Integer expireSeconds) {
        if (StringUtils.isEmpty(key) || expireSeconds == null) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance(" key or expireSeconds  must be not null");
        }
        Long expireMills = (long) expireSeconds * 1000;
        Long expireTime = System.currentTimeMillis() + expireMills;
        // 设置URL过期时间
        Date time = new Date(expireTime);
        return getExpireUrl(key, time);
    }

    /**
     * 生成下载url get的方式访问
     *
     * @param key  OSS里面的文件路径+文件名
     * @param time 失效时间
     * @return
     */
    public String getExpireUrl(String key, Date time) {
        if (StringUtils.isEmpty(key) || time == null) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance(" key or time  must be not null");
        }
        String downloadUrl = "";
        try {
            URL url = ossClient.generatePresignedUrl(bucketName, key, time);
            downloadUrl = url.toString();
            return downloadUrl;
        } catch (ClientException e) {
            LogUtil.error(log, "AliyunOssUtil >> fileUrl 下载地址异常", e);
            return downloadUrl;
        }
    }

    /**
     * 生成上传url put的方式访问
     *
     * @param key           OSS里面的文件路径+文件名
     * @param expireSeconds 失效时间
     * @return
     */
    public String getExpirePutUrl(String key, Integer expireSeconds) {
        if (StringUtils.isBlank(key) || expireSeconds == null) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance(" key or time  must not be null");
        }
        String uploadUrl = StringPool.EMPTY;
        try {
            GeneratePresignedUrlRequest request = new GeneratePresignedUrlRequest(bucketName, key, HttpMethod.PUT);
            Long expireMills = (long) expireSeconds * 1000;
            Long expireTime = System.currentTimeMillis() + expireMills;
            // 设置URL过期时间
            Date time = new Date(expireTime);
            request.setExpiration(time);
            // 设置ContentType
            request.setContentType(DEFAULT_OBJECT_CONTENT_TYPE);
            // 生成PUT方式的签名URL
            URL signedUrl = ossClient.generatePresignedUrl(request);
            uploadUrl = signedUrl.toString();
            return uploadUrl;
        } catch (ClientException e) {
            LogUtil.error(log, "AliyunOssUtil >> getExpirePutUrl 生成上传url异常 >> Ex={}", ExceptionUtils.getStackTrace(e));
            return uploadUrl;
        }
    }

    /**
     * 获得外网配置
     *
     * @param ossAk
     * @param ossSk
     * @return
     */
    public static OSSClient getAliOssConfigOut(String ossAk, String ossSk) {
        return new OSSClient(OSS_ENDPOINT_OUT, ossAk,
                ossSk);
    }

    /**
     * 获得内网配置
     *
     * @param ossAk
     * @param ossSk
     * @return
     */
    public static OSSClient getAliOssConfigInner(String ossAk, String ossSk) {
        return new OSSClient(OSS_ENDPOINT_INNER, ossAk,
                ossSk);
    }
}
