/**
 * fshows.com
 * Copyright (C) 2013-2019 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.loadbalance;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.beust.jcommander.internal.Lists;
import com.fshows.fsframework.core.utils.LogUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.RpcContext;
import org.apache.dubbo.rpc.cluster.loadbalance.RandomLoadBalance;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author buhao
 * @version SitLoadBalance.java, v 0.1 2019-11-26 21:57 buhao
 */
@Slf4j
public class StableLoadBalance extends RandomLoadBalance {

    public static final String NAME = "stable";

    /**
     * 上下文中，路由 tab 的 key
     */
    public static final String ROUTE_TAG_CONTEXT_KEY = "ROUTE_TAG_CONTEXT_KEY";
    /**
     * stable 环境路由 tab key
     */
    public static final String STABLE_ROUTE_TAG_KEY = "stable.route.tag";
    /**
     * 默认路由 tab key
     */
    public static final String DEFAULT_ROUTE_TAG_KEY = "dubbo.provider.owner";
    /**
     * dubbo 中保存的路由标签 key
     */
    private static final String DUBBO_ROUTE_TAG_KEY = "default.owner";

    @Override
    public <T> Invoker<T> select(List<Invoker<T>> invokers, URL url, Invocation invocation) {
        return doSelect(invokers, url, invocation);
    }

    @Override
    protected <T> Invoker<T> doSelect(List<Invoker<T>> invokers, URL url, Invocation invocation) {
        LogUtil.info(log, "doSelect >> 路由过滤开始 >> invokers = {}, invocation = {}", JSON.toJSONString(invokers), invocation);
        // 获得路由 TAG 根据上下文
        String sourceOwnerByContext = getRouteTagByContext();
        // 根据上下文中路由 TAG 过滤到的可调用节点
        List<Invoker<T>> filterInvokersByGroup = filterInvokersByContextRouteTag(invokers, sourceOwnerByContext);
        // 如果都为空取 stable 节点
        if (StrUtil.isBlank(sourceOwnerByContext) || filterInvokersByGroup.isEmpty()) {
            filterInvokersByGroup = filterInvokersByStableRouteTag(invokers);
        }
        // 如果过滤结果不为空，覆盖结果
        if (!filterInvokersByGroup.isEmpty()) {
            invokers = filterInvokersByGroup;
        }
        LogUtil.info(log, "doSelect >> 路由过滤结束 >> invokers = {}, invocation = {}", JSON.toJSONString(invokers), invocation);
        return super.doSelect(invokers, url, invocation);
    }

    /**
     * 根据 stable 路由标签过滤调用列表
     *
     * @param invokers
     * @param <T>
     * @return
     */
    private <T> List<Invoker<T>> filterInvokersByStableRouteTag(List<Invoker<T>> invokers) {
        final List<Invoker<T>> result = invokers.stream().filter(it -> StrUtil.equalsIgnoreCase(it.getUrl().getParameter(DUBBO_ROUTE_TAG_KEY), System.getProperty(STABLE_ROUTE_TAG_KEY))).collect(Collectors.toList());
        LogUtil.info(log, "filterInvokersByStableRouteTag >> 根据 stable route tag 过滤 >> invokers = {}", JSON.toJSONString(invokers));
        return result;
    }

    /**
     * 根据上下文路由标签过滤调用列表
     *
     * @param invokers
     * @param sourceOwnerByContext
     * @param <T>
     * @return
     */
    private <T> List<Invoker<T>> filterInvokersByContextRouteTag(List<Invoker<T>> invokers, String sourceOwnerByContext) {
        // 获得源 owner 节点过滤
        List<Invoker<T>> result = Lists.newArrayList();
        if (StrUtil.isNotBlank(sourceOwnerByContext)) {
            result = invokers.stream().filter(it -> StrUtil.equalsIgnoreCase(it.getUrl().getParameter(DUBBO_ROUTE_TAG_KEY), sourceOwnerByContext)).collect(Collectors.toList());
        }
        LogUtil.info(log, "doSelect >> 根据上下文路由 TAG 过滤调用列表 >> sourceOwnerByContext = {}， result = {}",
                JSON.toJSONString(sourceOwnerByContext), JSON.toJSONString(result));
        return result;
    }

    /**
     * 获得上下文中的路由标签
     *
     * @return
     */
    private String getRouteTagByContext() {
        // 获得上下文路由标签
        String result = RpcContext.getContext().getAttachment(ROUTE_TAG_CONTEXT_KEY);
        LogUtil.info(log, "doSelect >> 获得上下文路由 TAG >> result = {}", result);
        // 如果上下文没有源路由标签，就从环境变量取默认标签
        if (StrUtil.isBlank(result)) {
            result = System.getProperty(DEFAULT_ROUTE_TAG_KEY);
            if (StrUtil.isNotBlank(result)) {
                // 将默认路由标签存入上下文
                RpcContext.getContext().setAttachment(ROUTE_TAG_CONTEXT_KEY, result);
                LogUtil.info(log, "doSelect >> 上下文路由 TAG 为空,从环境变量中取，并设置到上下文中 >> result = {}", result);
            }
        }
        return result;
    }
}