/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.mq.config;

import com.fshows.fsframework.extend.util.ConfigUtil;
import com.fshows.fsframework.extend.util.FsAESUtil;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.validation.constraints.NotNull;
import java.util.HashSet;
import java.util.Set;

/**
 * MQ配置类
 *
 * @author liluqing
 * @version FsMqConfig.java, v 0.1 2025-01-10 15:30
 */
@Setter
@Slf4j
public class FsMqConfig implements ApplicationContextAware {

    /**
     * spring上下文
     */
    private ApplicationContext applicationContext;

    public FsMqConfig() {
    }

    /**
     * 【必填参数】AES密码
     */
    @NotNull
    private String aesPasswordKey = "${fs.aliyun.key.aes.encry.password}";

    /**
     * 【必填参数】加密的MQ AKSK
     */
    @NotNull
    private String accessKeyKey = "${fs.aliyun.mq.access.key}";
    @NotNull
    private String secretKeyKey = "${fs.aliyun.mq.secret.key}";

    /**
     * 【可选】Producer关闭旧实例的延迟时间，单位秒
     */
    private String producerCloseDelayTime = "${fs.aliyun.mq.producer.close.delay.time:30}";

    /**
     * 【可选】Consumer重启的最大延迟时间，单位秒
     */
    private String consumerRestartDelayTime = "${fs.aliyun.mq.consumer.restart.delay.time:30}";

    /**
     * 获取AES密码
     */
    public String getAesPassword() {
        return ConfigUtil.getProperty(aesPasswordKey, applicationContext);
    }

    /**
     * 获取解密后的AccessKey
     */
    public String getDecryptedAccessKey() {
        String encryptedKey = ConfigUtil.getProperty(accessKeyKey, applicationContext);
        return decryptAkAndSk(encryptedKey, getAesPassword());
    }

    /**
     * 获取解密后的SecretKey
     */
    public String getDecryptedSecretKey() {
        String encryptedKey = ConfigUtil.getProperty(secretKeyKey, applicationContext);
        return decryptAkAndSk(encryptedKey, getAesPassword());
    }

    /**
     * 获取Producer关闭延迟时间
     */
    public Integer getProducerCloseDelayTime() {
        return ConfigUtil.getProperty(producerCloseDelayTime, Integer.class, applicationContext);
    }

    /**
     * 获取Consumer重启延迟时间
     */
    public Integer getConsumerRestartDelayTime() {
        return ConfigUtil.getProperty(consumerRestartDelayTime, Integer.class, applicationContext);
    }

    /**
     * 获取需要监听的Apollo配置key集合
     */
    public Set<String> getWatchKeys() {
        Set<String> keys = new HashSet<>();
        keys.add(ConfigUtil.getDynamicKey(aesPasswordKey));
        keys.add(ConfigUtil.getDynamicKey(accessKeyKey));
        keys.add(ConfigUtil.getDynamicKey(secretKeyKey));
        return keys;
    }

    /**
     * 获取AES密码的Apollo key
     */
    public String getApolloKeyForAesPassword() {
        return ConfigUtil.getDynamicKey(aesPasswordKey);
    }

    /**
     * 获取AccessKey的Apollo key
     */
    public String getApolloKeyForAccessKey() {
        return ConfigUtil.getDynamicKey(accessKeyKey);
    }

    /**
     * 获取SecretKey的Apollo key
     */
    public String getApolloKeyForSecretKey() {
        return ConfigUtil.getDynamicKey(secretKeyKey);
    }

    /**
     * 判断是否启用MQ客户端动态更新
     */
    public Boolean getMqClientDynamicUpdate() {
        return ConfigUtil.isDynamicProperty(accessKeyKey) && ConfigUtil.isDynamicProperty(secretKeyKey);
    }

    /**
     * 解密AK和SK
     *
     * @param ciphertext 密文
     * @param pwd        密码
     * @return 明文
     */
    private String decryptAkAndSk(String ciphertext, String pwd) {
        // 小于32位的，说明是明文，则直接返回
        if (StringUtils.length(ciphertext) < 32) {
            return ciphertext;
        }
        return FsAESUtil.decryptKey(ciphertext, pwd);
    }

    /**
     * 实现ApplicationContextAware接口
     * @param applicationContext the ApplicationContext object to be used by this object
     * @throws BeansException
     */
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
