/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.mq.decorator;

import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.ONSFactory;
import com.aliyun.openservices.ons.api.Producer;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.SendCallback;
import com.aliyun.openservices.ons.api.SendResult;
import com.aliyun.openservices.ons.api.exception.ONSClientException;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.aliyun.mq.config.FsMqConfig;
import com.fshows.fsframework.extend.aliyun.mq.core.FsMqInstanceManager;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PreDestroy;
import java.util.Properties;
import java.util.concurrent.ExecutorService;

/**
 * 增强的Producer装饰器
 * 在原有ProducerBean基础上增加Apollo配置变更时的无损实例更新功能
 *
 * @author liluqing
 * @version FsProducerBean.java, v 0.1 2025-01-10 15:30
 */
@Slf4j
public class FsProducerBean implements Producer {

    /**
     * 需要注入该字段，指定构造{@code Producer}实例的属性，具体支持的属性详见{@link PropertyKeyConst}
     */
    private Properties properties;

    /**
     * 当前活跃的Producer实例
     */
    private volatile Producer producer;

    /**
     * 实例唯一标识
     */
    private String instanceKey;

    /**
     * MQ实例管理器
     */
    @Autowired
    private FsMqInstanceManager instanceManager;

    /**
     * MQ配置
     */
    @Autowired
    private FsMqConfig fsMqConfig;

    /**
     * 启动该{@code Producer}实例，建议配置为Bean的init-method
     */
    @Override
    public void start() {
        if (null == this.instanceManager) {
            throw new ONSClientException("当前 FsProducerBean 对象，未通过 @Bean 注解注册到Spring容器中, 启动失败");
        }
        if (null == this.fsMqConfig) {
            throw new ONSClientException("当前 FsProducerBean 对象，未通过 @Bean 注解注册到Spring容器中, 启动失败");
        }
        if (null == this.properties) {
            throw new ONSClientException("properties not set");
        }

        // 校验AK/SK不能由外部设置
        validateCredentials();

        // 更新Properties中的AK/SK（如果配置了动态更新）
        Properties updatedProperties = updatePropertiesWithDecryptedKeys();

        this.producer = ONSFactory.createProducer(updatedProperties);
        this.producer.start();

        // 生成实例唯一标识并注册到实例管理器
        this.instanceKey = generateInstanceKey();
        if (instanceManager != null && fsMqConfig != null && fsMqConfig.getMqClientDynamicUpdate()) {
            instanceManager.registerProducer(this.instanceKey, this, this.properties);
        }

        LogUtil.info(log, "FsProducerBean启动完成: {}", instanceKey);
    }

    /**
     * 替换Producer实例（原子操作）
     * 用于配置变更时的无损更新
     *
     * @param newProducer 新的Producer实例
     * @return 旧的Producer实例
     */
    public synchronized Producer replaceProducerInstance(Producer newProducer) {
        Producer oldProducer = this.producer;
        this.producer = newProducer;
        LogUtil.info(log, "Producer实例已替换: {}", instanceKey);
        return oldProducer;
    }

    /**
     * 关闭该{@code Producer}实例，建议配置为Bean的destroy-method
     */
    @Override
    public void shutdown() {
        // 从实例管理器中注销
        if (instanceManager != null && instanceKey != null) {
            instanceManager.unregisterProducer(instanceKey);
        }

        if (this.producer != null) {
            this.producer.shutdown();
        }
        LogUtil.info(log, "FsProducerBean已关闭: {}", instanceKey);
    }

    /**
     * 预销毁方法
     */
    @PreDestroy
    public void preDestroy() {
        shutdown();
    }

    @Override
    public void updateCredential(Properties credentialProperties) {
        if (this.producer != null) {
            this.producer.updateCredential(credentialProperties);
        }
    }

    @Override
    public SendResult send(Message message) {
        return this.producer.send(message);
    }

    @Override
    public void sendOneway(Message message) {
        this.producer.sendOneway(message);
    }

    @Override
    public void sendAsync(Message message, SendCallback sendCallback) {
        this.producer.sendAsync(message, sendCallback);
    }

    @Override
    public void setCallbackExecutor(final ExecutorService callbackExecutor) {
        this.producer.setCallbackExecutor(callbackExecutor);
    }

    @Override
    public boolean isStarted() {
        return this.producer != null && this.producer.isStarted();
    }

    @Override
    public boolean isClosed() {
        return this.producer == null || this.producer.isClosed();
    }

    /**
     * 获取Properties
     */
    public Properties getProperties() {
        return properties;
    }

    /**
     * 设置Properties
     */
    public void setProperties(Properties properties) {
        this.properties = properties;
    }

    /**
     * 更新Properties，解密AK/SK
     *
     * @return 更新后的Properties
     */
    private Properties updatePropertiesWithDecryptedKeys() {
        Properties updatedProperties = new Properties();
        updatedProperties.putAll(this.properties);

        // 如果配置了动态更新，则使用解密后的AK/SK
        if (fsMqConfig != null && fsMqConfig.getMqClientDynamicUpdate()) {
            updatedProperties.setProperty(PropertyKeyConst.AccessKey, fsMqConfig.getDecryptedAccessKey());
            updatedProperties.setProperty(PropertyKeyConst.SecretKey, fsMqConfig.getDecryptedSecretKey());
        }

        return updatedProperties;
    }

    /**
     * 校验AK/SK不能由外部设置
     */
    private void validateCredentials() {
        String accessKey = this.properties.getProperty(PropertyKeyConst.AccessKey);
        String secretKey = this.properties.getProperty(PropertyKeyConst.SecretKey);
        
        if (accessKey != null || secretKey != null) {
            String producerId = properties.getProperty(PropertyKeyConst.ProducerId);
            if (StringUtils.isBlank(producerId)) {
                producerId = properties.getProperty(PropertyKeyConst.GROUP_ID);
            }
            String errorMsg = String.format("%s生产者实例启动失败，阿里云ak和sk设置必须为空，系统默认使用统一配置ak/sk配置", producerId);
            LogUtil.error(log, errorMsg);
            throw new ONSClientException(errorMsg);
        }
    }

    /**
     * 生成实例唯一标识
     *
     * @return 实例标识
     */
    private String generateInstanceKey() {
        return "producer_" + System.identityHashCode(this) + "_" + System.currentTimeMillis();
    }
}
