/**
 * fshows.com
 * Copyright (C) 2013-2025 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.mq.core;

import com.aliyun.openservices.ons.api.Consumer;
import com.aliyun.openservices.ons.api.MessageListener;
import com.aliyun.openservices.ons.api.ONSFactory;
import com.aliyun.openservices.ons.api.Producer;
import com.aliyun.openservices.ons.api.PropertyKeyConst;
import com.aliyun.openservices.ons.api.bean.Subscription;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.aliyun.mq.config.FsMqConfig;
import com.fshows.fsframework.extend.aliyun.mq.decorator.FsConsumerBean;
import com.fshows.fsframework.extend.aliyun.mq.decorator.FsProducerBean;
import lombok.Data;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * MQ实例管理器
 * 负责管理所有Producer和Consumer实例，处理配置变更时的实例更新
 *
 * @author liluqing
 * @version FsMqInstanceManager.java, v 0.1 2025-01-10 15:30
 */
@Slf4j
public class FsMqInstanceManager {

    /**
     * 线程安全的Producer实例注册表
     */
    private final ConcurrentHashMap<String, ProducerInstanceInfo> producerRegistry = new ConcurrentHashMap<>();

    /**
     * 线程安全的Consumer实例注册表
     */
    private final ConcurrentHashMap<String, ConsumerInstanceInfo> consumerRegistry = new ConcurrentHashMap<>();

    /**
     * 定时任务执行器
     */
    private final ScheduledExecutorService executorService = Executors.newScheduledThreadPool(4);

    /**
     * 随机数生成器，用于Consumer随机延迟重启
     */
    private final Random random = new Random();

    /**
     * MQ配置
     */
    @Setter
    private FsMqConfig fsMqConfig;

    /**
     * 注册Producer实例
     *
     * @param instanceKey  实例唯一标识
     * @param producerBean Producer装饰器
     * @param properties   原始配置
     */
    public void registerProducer(String instanceKey, FsProducerBean producerBean, Properties properties) {
        ProducerInstanceInfo info = new ProducerInstanceInfo(producerBean, properties);
        producerRegistry.put(instanceKey, info);
        LogUtil.info(log, "注册Producer实例: {}", instanceKey);
    }

    /**
     * 注册Consumer实例
     *
     * @param instanceKey       实例唯一标识
     * @param consumerBean      Consumer装饰器
     * @param properties        原始配置
     * @param subscriptionTable 订阅表
     */
    public void registerConsumer(String instanceKey, FsConsumerBean consumerBean, Properties properties,
                                Map<Subscription, MessageListener> subscriptionTable) {
        ConsumerInstanceInfo info = new ConsumerInstanceInfo(consumerBean, properties, subscriptionTable);
        consumerRegistry.put(instanceKey, info);
        LogUtil.info(log, "注册Consumer实例: {}", instanceKey);
    }

    /**
     * 注销Producer实例
     *
     * @param instanceKey 实例唯一标识
     */
    public void unregisterProducer(String instanceKey) {
        producerRegistry.remove(instanceKey);
        LogUtil.info(log, "注销Producer实例: {}", instanceKey);
    }

    /**
     * 注销Consumer实例
     *
     * @param instanceKey 实例唯一标识
     */
    public void unregisterConsumer(String instanceKey) {
        consumerRegistry.remove(instanceKey);
        LogUtil.info(log, "注销Consumer实例: {}", instanceKey);
    }

    /**
     * 更新所有Producer实例
     */
    public void updateAllProducers() {
        LogUtil.info(log, "开始更新所有Producer实例，共{}个", producerRegistry.size());

        for (Map.Entry<String, ProducerInstanceInfo> entry : producerRegistry.entrySet()) {
            String instanceKey = entry.getKey();
            ProducerInstanceInfo info = entry.getValue();

            try {
                updateProducerInstance(instanceKey, info);
            } catch (Exception e) {
                LogUtil.error(log, "更新Producer实例失败: {}", instanceKey, e);
            }
        }
    }

    /**
     * 更新所有Consumer实例
     */
    public void updateAllConsumers() {
        LogUtil.info(log, "开始更新所有Consumer实例，共{}个", consumerRegistry.size());

        for (Map.Entry<String, ConsumerInstanceInfo> entry : consumerRegistry.entrySet()) {
            String instanceKey = entry.getKey();
            ConsumerInstanceInfo info = entry.getValue();

            try {
                updateConsumerInstance(instanceKey, info);
            } catch (Exception e) {
                LogUtil.error(log, "更新Consumer实例失败: {}", instanceKey, e);
            }
        }
    }

    /**
     * 更新单个Producer实例 - 无损策略
     * 1. 创建新实例
     * 2. 原子性替换
     * 3. 延迟关闭旧实例
     *
     * @param instanceKey 实例标识
     * @param info        实例信息
     */
    private void updateProducerInstance(String instanceKey, ProducerInstanceInfo info) {
        FsProducerBean producerBean = info.getProducerBean();

        // 创建新的Properties，更新AK/SK
        Properties newProperties = updateProperties(info.getOriginalProperties());

        // 创建新的Producer实例
        Producer newProducerImpl = ONSFactory.createProducer(newProperties);
        newProducerImpl.start();

        // 原子性替换实例
        Producer oldProducerImpl = producerBean.replaceProducerInstance(newProducerImpl);

        // 延迟关闭旧实例
        if (oldProducerImpl != null) {
            int delayTime = fsMqConfig.getProducerCloseDelayTime();
            executorService.schedule(() -> {
                try {
                    oldProducerImpl.shutdown();
                    LogUtil.info(log, "延迟关闭旧Producer实例: {}", instanceKey);
                } catch (Exception e) {
                    LogUtil.error(log, "关闭旧Producer实例失败: {}", instanceKey, e);
                }
            }, delayTime, TimeUnit.SECONDS);
        }

        LogUtil.info(log, "Producer实例更新完成: {}", instanceKey);
    }

    /**
     * 更新单个Consumer实例 - 随机延迟重启策略
     * 1. 随机延迟1-N秒
     * 2. 关闭旧实例
     * 3. 创建并启动新实例
     *
     * @param instanceKey 实例标识
     * @param info        实例信息
     */
    private void updateConsumerInstance(String instanceKey, ConsumerInstanceInfo info) {
        // 随机延迟1-N秒
        int maxDelay = fsMqConfig.getConsumerRestartDelayTime();
        int randomDelay = 1 + random.nextInt(Math.max(1, maxDelay));

        executorService.schedule(() -> {
            try {
                FsConsumerBean consumerBean = info.getConsumerBean();

                // 关闭旧实例
                consumerBean.shutdown();

                // 创建新的Properties，更新AK/SK
                Properties newProperties = updateProperties(info.getOriginalProperties());

                // 创建并启动新实例
                Consumer newConsumerImpl = ONSFactory.createConsumer(newProperties);
                consumerBean.replaceConsumerInstance(newConsumerImpl, info.getSubscriptionTable());
                consumerBean.start();

                LogUtil.info(log, "Consumer实例重启完成: {}", instanceKey);
            } catch (Exception e) {
                LogUtil.error(log, "Consumer实例重启失败: {}", instanceKey, e);
            }
        }, randomDelay, TimeUnit.SECONDS);

        LogUtil.info(log, "Consumer实例已安排{}秒后重启: {}", randomDelay, instanceKey);
    }

    /**
     * 更新Properties中的AK/SK
     *
     * @param originalProperties 原始配置
     * @return 更新后的配置
     */
    private Properties updateProperties(Properties originalProperties) {
        Properties newProperties = new Properties();
        newProperties.putAll(originalProperties);

        // 更新AK/SK
        newProperties.setProperty(PropertyKeyConst.AccessKey, fsMqConfig.getDecryptedAccessKey());
        newProperties.setProperty(PropertyKeyConst.SecretKey, fsMqConfig.getDecryptedSecretKey());

        return newProperties;
    }

    /**
     * 获取当前注册的Producer数量
     */
    public int getProducerCount() {
        return producerRegistry.size();
    }

    /**
     * 获取当前注册的Consumer数量
     */
    public int getConsumerCount() {
        return consumerRegistry.size();
    }

    /**
     * 关闭实例管理器
     */
    public void shutdown() {
        LogUtil.info(log, "关闭MQ实例管理器");
        executorService.shutdown();
        try {
            if (!executorService.awaitTermination(30, TimeUnit.SECONDS)) {
                executorService.shutdownNow();
            }
        } catch (InterruptedException e) {
            executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    /**
     * Producer实例信息
     */
    @Data
    private static class ProducerInstanceInfo {
        private final FsProducerBean producerBean;
        private final Properties originalProperties;

        public ProducerInstanceInfo(FsProducerBean producerBean, Properties originalProperties) {
            this.producerBean = producerBean;
            this.originalProperties = new Properties();
            this.originalProperties.putAll(originalProperties);
        }
    }

    /**
     * Consumer实例信息
     */
    @Data
    private static class ConsumerInstanceInfo {
        private final FsConsumerBean consumerBean;
        private final Properties originalProperties;
        private final Map<Subscription, MessageListener> subscriptionTable;

        public ConsumerInstanceInfo(FsConsumerBean consumerBean, Properties originalProperties,
                                   Map<Subscription, MessageListener> subscriptionTable) {
            this.consumerBean = consumerBean;
            this.originalProperties = new Properties();
            this.originalProperties.putAll(originalProperties);
            this.subscriptionTable = subscriptionTable;
        }
    }
}
