/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcException;
import com.alibaba.fastjson.JSONObject;
import com.fshows.fsframework.core.constants.StringPool;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.dubbo.model.DubboThreadPoolMonitorConfigConvertModel;
import com.fshows.fsframework.extend.dubbo.model.DubboThreadPoolMonitorConfigModel;
import com.fshows.fsframework.extend.util.SpringContextUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * dubbo线程池占用方法监控
 *
 * @author zhaoxumin
 * @version GlobalDubboThreadPoolMonitor.java, v 0.1 2024-08-06 17:52 zhaoxumin
 */
@Slf4j
@Activate(group = {Constants.PROVIDER}, order = -9999)
public class GlobalDubboThreadPoolMonitor implements Filter {

    /**
     * dubbo线程池占用方法监控开关. 0-关闭 1-开启 apollo中配置
     */
    private final static String DUBBO_THREAD_POOL_MONITOR_SWITCH = "dubbo.thread.pool.monitor.switch";

    /**
     * dubbo线程池监控参数配置. apollo中配置
     */
    private final static String DUBBO_THREAD_POOL_MONITOR_CONFIG = "dubbo.thread.pool.monitor.config";

    /**
     * dubbo线程池线程数量. apollo中配置
     */
    private final static String DUBBO_PROTOCOL_THREADS = "dubbo.protocol.threads";

    /**
     * 调度任务执行器
     */
    private final static ScheduledExecutorService SCHEDULER = Executors.newScheduledThreadPool(1);

    /**
     * 调度任务集合
     */
    private final Queue<ScheduledFuture<?>> scheduledFutures = new ConcurrentLinkedQueue<>();

    /**
     * 保存dubbo方法的调用计数
     */
    private final Map<String, AtomicInteger> methodAndCountMap = new ConcurrentHashMap<>();

    /**
     * 上一次告警时间
     */
    private static final AtomicLong LAST_ALERT_TIME = new AtomicLong(0);

    /**
     * dubbo协议默认线程数量
     */
    private final static String DEFAULT_DUBBO_PROTOCOL_THREADS = "200";

    /**
     * 线程池紧急告警默认比例
     */
    private final static BigDecimal EMERGENCY_THRESHOLD_RATIO = new BigDecimal("0.6");

    /**
     * 数值100
     */
    private final static BigDecimal HUNDRED = new BigDecimal("100");

    /**
     * 警报间隔时间. 单位毫秒
     */
    private final static Long ALERT_INTERVAL_MS = 10 * 1000L;

    private static volatile DubboThreadPoolMonitorConfigConvertModel configModel = null;

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        Environment environment = SpringContextUtil.getEnvironment();
        if (environment != null) {
            if (StringPool.ONE.equals(environment.getProperty(DUBBO_THREAD_POOL_MONITOR_SWITCH))) {
                DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigModel = null;
                try {
                    threadPoolMonitorConfigModel = getThreadPoolMonitorConfigModel();
                } catch (Throwable e) {
                    LogUtil.error(log, "record thread-pool running get monitor config error >> dubbo线程池占用方法监控，加载配置参数异常", e);
                }

                if (threadPoolMonitorConfigModel != null) {
                    String methodName = invoker.getUrl().getPath() + "." + invocation.getMethodName();

                    methodAndCountMap.compute(methodName, (key, count) -> {
                        if (count == null) {
                            return new AtomicInteger(1);
                        } else {
                            count.incrementAndGet();
                            return count;
                        }
                    });
                    try {
                        return invoker.invoke(invocation);
                    } finally {
                        methodAndCountMap.compute(methodName, (key, count) -> {
                            if (count != null) {
                                if (count.decrementAndGet() == 0) {
                                    // 计数为0时，删除该方法记录
                                    return null;
                                }
                            }
                            return count;
                        });
                        // 如果占用线程数过高，紧急警报
                        alertIfEmergency(threadPoolMonitorConfigModel);
                    }
                }
            } else {
                configModel = null;
                cancelMonitorTask();
            }
        }
        return invoker.invoke(invocation);
    }

    /**
     * 紧急情况下直接告警
     *
     * @param threadPoolMonitorConfigModel
     */
    private void alertIfEmergency(DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigModel) {
        // 计算当前线程池使用数
        int usageCount = methodAndCountMap.values().stream()
                .mapToInt(AtomicInteger::get)
                .sum();
        // 计算告警阈值
        int emergencyThreshold = threadPoolMonitorConfigModel.getApplicationThreadCount()
                .multiply(EMERGENCY_THRESHOLD_RATIO)
                .setScale(2, RoundingMode.HALF_UP)
                .intValue();

        if (usageCount >= emergencyThreshold) {
            long now = System.currentTimeMillis();
            // 获取上一次告警时间，并重置为当前时间
            long lastAlertTime = LAST_ALERT_TIME.getAndSet(now);
            // 比较当前时间距离上次告警时间，是否已过10秒，是则继续告警
            if (lastAlertTime != 0 && now < lastAlertTime + ALERT_INTERVAL_MS) {
                return;
            }

            // 排序计数列表
            List<Map.Entry<String, AtomicInteger>> methodAndCountList = methodAndCountMap.entrySet().stream()
                    .sorted(Map.Entry.<String, AtomicInteger>comparingByValue(Comparator.comparingInt(AtomicInteger::get)).reversed())
                    // 兜底，最多只取前200个方法
                    .limit(200)
                    .collect(Collectors.toList());
            LogUtil.error(log, "record thread-pool running method >> 紧急警报，线程池占用{}%以上（仅输出前200个方法）>> usageCount={}, methodAndCountList={}", EMERGENCY_THRESHOLD_RATIO.multiply(HUNDRED).setScale(0, RoundingMode.HALF_UP),
                    usageCount, methodAndCountList);
        }
    }

    /**
     * 日志输出线程池占用情况
     */
    private void recordThreadPoolRunningMethod(BigDecimal applicationThreadCount, BigDecimal usageThresholdRatio) {
        try {
            // 排序计数列表
            List<Map.Entry<String, AtomicInteger>> methodAndCountList = methodAndCountMap.entrySet().stream()
                    .sorted(Map.Entry.<String, AtomicInteger>comparingByValue(Comparator.comparingInt(AtomicInteger::get)).reversed())
                    .collect(Collectors.toList());

            // 计算总使用量
            int usageCount = methodAndCountList.stream()
                    .mapToInt(entry -> entry.getValue().get())
                    .sum();

            if (usageThresholdRatio != null) {
                int threshold = applicationThreadCount.multiply(usageThresholdRatio).setScale(2, RoundingMode.HALF_UP).intValue();
                if (usageCount >= threshold) {
                    LogUtil.warn(log, "record thread-pool running method >> dubbo线程池占用方法监控，超过设定阈值 >> applicationThreadCount={}, usageCount={}, methodAndCountList={}",
                            applicationThreadCount, usageCount, methodAndCountList);
                }
            }
            LogUtil.info(log, "record thread-pool running method >> 输出dubbo线程池占用方法 >> usageCount={}, methodAndCountList={}", usageCount, methodAndCountList);
        } catch (Throwable e) {
            LogUtil.error(log, "record thread-pool running method error >> 输出dubbo线程池占用方法异常", e);
        }
    }

    /**
     * 获取线程池监控配置
     *
     * @return
     */
    private DubboThreadPoolMonitorConfigConvertModel getThreadPoolMonitorConfigModel() {
        if (configModel == null) {
            synchronized (this) {
                if (configModel == null) {
                    Environment environment = SpringContextUtil.getEnvironment();
                    if (environment == null) {
                        return null;
                    }
                    DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigConvertModel = new DubboThreadPoolMonitorConfigConvertModel();
                    // 配置dubbo协议线程数
                    String dubboProtocolThreads = StrUtil.blankToDefault(environment.getProperty(DUBBO_PROTOCOL_THREADS), DEFAULT_DUBBO_PROTOCOL_THREADS);
                    threadPoolMonitorConfigConvertModel.setApplicationThreadCount(new BigDecimal(dubboProtocolThreads));

                    // 配置监控点
                    String threadPoolMonitorConfigStr = environment.getProperty(DUBBO_THREAD_POOL_MONITOR_CONFIG);
                    if (StrUtil.isNotBlank(threadPoolMonitorConfigStr)) {
                        DubboThreadPoolMonitorConfigModel config = JSONObject.parseObject(threadPoolMonitorConfigStr, DubboThreadPoolMonitorConfigModel.class);
                        if (config != null) {
                            List<DubboThreadPoolMonitorConfigModel.ThreadPoolMonitorPointConfigModel> threadPoolMonitorPointConfigList = config.getThreadPoolMonitorPointConfigModelList();
                            if (CollUtil.isNotEmpty(threadPoolMonitorPointConfigList)) {
                                List<DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel> threadPoolMonitorPointConfigConvertModelList = Lists.newArrayList();
                                for (DubboThreadPoolMonitorConfigModel.ThreadPoolMonitorPointConfigModel threadPoolMonitorPointConfigModel : threadPoolMonitorPointConfigList) {
                                    if (StrUtil.isNotBlank(threadPoolMonitorPointConfigModel.getMonitorPeriod())) {
                                        String[] split = threadPoolMonitorPointConfigModel.getMonitorPeriod().split(StringPool.TILDA);
                                        if (split.length == 2) {
                                            DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel threadPoolMonitorPointConfigConvertModel = new DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel();
                                            threadPoolMonitorPointConfigConvertModel.setMonitorPeriodStartTime(LocalTime.parse(split[0], DateTimeFormatter.ofPattern("HH:mm:ss")));
                                            threadPoolMonitorPointConfigConvertModel.setMonitorPeriodEndTime(LocalTime.parse(split[1], DateTimeFormatter.ofPattern("HH:mm:ss")));
                                            threadPoolMonitorPointConfigConvertModel.setMonitorFrequency(threadPoolMonitorPointConfigModel.getMonitorFrequency());
                                            threadPoolMonitorPointConfigConvertModel.setUsageThresholdRatio(new BigDecimal(threadPoolMonitorPointConfigModel.getUsageThresholdRatio()));
                                            threadPoolMonitorPointConfigConvertModelList.add(threadPoolMonitorPointConfigConvertModel);
                                        }
                                    }
                                }
                                threadPoolMonitorConfigConvertModel.setThreadPoolMonitorPointConfigList(threadPoolMonitorPointConfigConvertModelList);
                                // 初始化参数完成，开启监控
                                startMonitorTask(threadPoolMonitorConfigConvertModel);
                            }
                        }
                    }
                    configModel = threadPoolMonitorConfigConvertModel;
                }
            }
        }
        return configModel;
    }

    /**
     * 开启监控
     *
     * @param threadPoolMonitorConfigConvertModel
     */
    private void startMonitorTask(DubboThreadPoolMonitorConfigConvertModel threadPoolMonitorConfigConvertModel) {
        if (threadPoolMonitorConfigConvertModel != null && CollUtil.isNotEmpty(threadPoolMonitorConfigConvertModel.getThreadPoolMonitorPointConfigList())) {
            for (DubboThreadPoolMonitorConfigConvertModel.ThreadPoolMonitorPointConfigConvertModel configConvertModel : threadPoolMonitorConfigConvertModel.getThreadPoolMonitorPointConfigList()) {
                LocalTime start = configConvertModel.getMonitorPeriodStartTime();
                LocalTime end = configConvertModel.getMonitorPeriodEndTime();
                long monitorFrequency = configConvertModel.getMonitorFrequency();

                long initialDelay = 0;
                LocalTime nowTime = LocalTime.now();
                if (nowTime.isBefore(start)) {
                    initialDelay = nowTime.until(start, ChronoUnit.SECONDS);
                }

                // 调度任务
                ScheduledFuture<?> scheduledFuture = SCHEDULER.scheduleAtFixedRate(() -> {
                    try {
                        if (SpringContextUtil.getEnvironment() != null && StringPool.ONE.equals(SpringContextUtil.getEnvironment().getProperty(DUBBO_THREAD_POOL_MONITOR_SWITCH))) {
                            LocalTime now = LocalTime.now();
                            if (now.isAfter(start) && now.isBefore(end)) {
                                recordThreadPoolRunningMethod(threadPoolMonitorConfigConvertModel.getApplicationThreadCount(), configConvertModel.getUsageThresholdRatio());
                            }
                        }
                    } catch (Throwable e) {
                        LogUtil.error(log, "running thread-pool monitor task error >> 运行线程池占用方法监控异常", e);
                    }
                }, initialDelay, monitorFrequency, TimeUnit.SECONDS);
                scheduledFutures.add(scheduledFuture);
            }
        }
    }

    /**
     * 取消监控
     */
    public void cancelMonitorTask() {
        if (CollUtil.isNotEmpty(scheduledFutures)) {
            synchronized (this) {
                if (CollUtil.isNotEmpty(scheduledFutures)) {
                    try {
                        ScheduledFuture<?> future;
                        while ((future = scheduledFutures.poll()) != null) {
                            if (!future.isCancelled()) {
                                future.cancel(false);
                            }
                        }
                    } catch (Throwable e) {
                        LogUtil.error(log, "cancel thread-pool monitor task error >> 取消线程池占用方法监控异常", e);
                    }
                }
            }
        }
    }
}