/**
 * fshows.com
 * Copyright (C) 2013-2024 All Rights Reserved.
 */
package com.fshows.fsframework.extend.aliyun.oss;

import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.crypto.symmetric.AES;
import com.aliyun.oss.ClientConfiguration;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.comm.Protocol;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.ObjectMetadata;
import com.ctrip.framework.apollo.enums.PropertyChangeType;
import com.ctrip.framework.apollo.model.ConfigChange;
import com.ctrip.framework.apollo.model.ConfigChangeEvent;
import com.ctrip.framework.apollo.spring.annotation.ApolloConfigChangeListener;
import com.fshows.fsframework.common.exception.CommonException;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.extend.util.FsAESUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author wangqilei
 * @version OssLocalClient.java, v 0.1 2024-12-10 3:26 PM wangqilei
 */
@Slf4j
@Service
public class FsOssClient {
    /**
     * 阿波罗配置
     */
    private static final String OSS_ACCESS_KEY_PREFIX = "fs.aliyun.oss.access.key";
    private static final String OSS_SECRET_KEY_PREFIX = "fs.aliyun.oss.secret.key";
    private static final String BACKSLASH = "/";
    /**
     * oss 外网地址
     */
    @Value("${fs.aliyun.oss.out.url:https://oss-cn-hangzhou.aliyuncs.com}")
    private String ossEndPointOut;
    /**
     * oss 内网地址
     */
    @Value("${fs.aliyun.oss.inner.url:https://oss-cn-hangzhou-internal.aliyuncs.com}")
    private String ossEndPointInner;

    /**
     * 外网链接过期时间
     */
    @Value("${fs.aliyun.oss.url.expire.time:3600}")
    private Long urlExpireTime;

    @Value("${fs.aliyun.key.aes.encry.password}")
    private String aesPassword;

    @Value("${fs.aliyun.oss.access.key}")
    private String encryAccessKey;

    @Value("${fs.aliyun.oss.secret.key}")
    private String encrySecretKey;

    private volatile OSSClient outOssClient;

    private volatile OSSClient innerOssClient;

    private final ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1);

    @PostConstruct
    public void init() {
        try {
            String decryptAccess = FsAESUtil.decryptKey(encryAccessKey, aesPassword);
            String decryptSecret = FsAESUtil.decryptKey(encrySecretKey, aesPassword);

            ClientConfiguration config = buildOssConfig();
            if (outOssClient == null) {
                outOssClient = new OSSClient(ossEndPointOut, decryptAccess, decryptSecret, config);
            }
            if (innerOssClient == null) {
                innerOssClient = new OSSClient(ossEndPointInner, decryptAccess, decryptSecret, config);
            }

            LogUtil.info(log, "FsOssClient初始化完成");
        } catch (Exception e) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("FsOssClient客户端初始化异常，请检查AKSK是否配置正确");
        }
    }

    @ApolloConfigChangeListener(value = "lifecircle.common", interestedKeys = {OSS_ACCESS_KEY_PREFIX, OSS_SECRET_KEY_PREFIX})
    public synchronized void refreshOssClient(ConfigChangeEvent changeEvent) {
        ConfigChange accessConfigChange = changeEvent.getChange(OSS_ACCESS_KEY_PREFIX);
        ConfigChange secretConfigChange = changeEvent.getChange(OSS_SECRET_KEY_PREFIX);
        LogUtil.info(log, "refreshOssClient >> 接收到AKSK变换通知 >> access={} , secret={}", accessConfigChange, secretConfigChange);

        String refreshEncryAccessKey = encryAccessKey;
        if (accessConfigChange != null
                && !accessConfigChange.getChangeType().equals(PropertyChangeType.DELETED)
                && !accessConfigChange.getOldValue().equals(accessConfigChange.getNewValue())) {
            refreshEncryAccessKey = accessConfigChange.getNewValue();
        }

        String refreshEncrySecretKey = encrySecretKey;
        if (secretConfigChange != null
                && !secretConfigChange.getChangeType().equals(PropertyChangeType.DELETED)
                && !secretConfigChange.getOldValue().equals(secretConfigChange.getNewValue())) {
            refreshEncrySecretKey = secretConfigChange.getNewValue();
        }

        if (refreshEncryAccessKey.equals(encryAccessKey) && refreshEncrySecretKey.equals(encrySecretKey)) {
            LogUtil.info(log, "refreshOssClient >> aksk没有发生变更");
            return;
        }

        // 临时客户端，用于关闭，释放资源
        final OSSClient tmpOutOssClient = outOssClient;
        final OSSClient tmpInnerOssClient = innerOssClient;
        try {
            // 对key解密
            String accessKey = FsAESUtil.decryptKey(refreshEncryAccessKey, aesPassword);
            String secretKey = FsAESUtil.decryptKey(refreshEncrySecretKey, aesPassword);

            // 创建新客户端
            ClientConfiguration config = buildOssConfig();
            outOssClient = new OSSClient(ossEndPointOut, accessKey, secretKey, config);
            innerOssClient = new OSSClient(ossEndPointInner, accessKey, secretKey, config);

            LogUtil.info(log, "refreshOssClient >> OSS动态替换aksk成功，新OSS客户端创建完成 >> accessKey={} , secretKey={}", accessKey, secretKey);
        } catch (Exception e) {
            LogUtil.error(log, "refreshOssClient >> OSS动态替换aksk异常", e);
        } finally {
            // 延迟关闭旧客户端，默认30秒
            executorService.schedule(() -> {
                // 当新客户端已经创建完成，才能关闭旧客户端
                if (!Objects.equals(tmpOutOssClient, outOssClient)) {
                    tmpOutOssClient.shutdown();
                }
                if (!Objects.equals(tmpInnerOssClient, innerOssClient)) {
                    tmpInnerOssClient.shutdown();
                }
                LogUtil.info(log, "refreshOssClient >> 旧客户端关闭完成");
            }, 30, TimeUnit.SECONDS);
        }
    }

    private ClientConfiguration buildOssConfig() {
        // 创建ClientConfiguration。ClientConfiguration是OSSClient的配置类，可配置代理、连接超时、最大连接数等参数。
        ClientConfiguration conf = new ClientConfiguration();
        // 设置OSSClient允许打开的最大HTTP连接数，默认为1024个。
        conf.setMaxConnections(200);
        // 设置Socket层传输数据的超时时间，默认为50000毫秒。
        conf.setSocketTimeout(10000);
        // 设置建立连接的超时时间，默认为50000毫秒。
        conf.setConnectionTimeout(10000);
        // 设置从连接池中获取连接的超时时间（单位：毫秒），默认不超时。
        conf.setConnectionRequestTimeout(1000);
        // 设置连接空闲超时时间。超时则关闭连接，默认为60000毫秒。
        conf.setIdleConnectionTime(10000);
        // 设置失败请求重试次数，默认为3次。
        conf.setMaxErrorRetry(3);
        // 设置是否支持将自定义域名作为Endpoint，默认支持。
        conf.setSupportCname(true);
        // 设置是否开启二级域名的访问方式，默认不开启。
        conf.setSLDEnabled(false);
        // 设置连接OSS所使用的协议（HTTP/HTTPS），默认为HTTP。
        conf.setProtocol(Protocol.HTTP);
        return conf;
    }

    /**
     * 上传文件（可选择内外网环境）
     */
    public void uploadFileWithOutFlag(String bucketName, String key, InputStream inputStream, boolean outOssFlag) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("Oss key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outOssFlag) {
            outOssClient.putObject(bucketName, key, inputStream);
        } else {
            innerOssClient.putObject(bucketName, key, inputStream);
        }
    }

    /**
     * 上传文件（可选择内外网环境）
     */
    public void uploadFileWithOutFlag(String bucketName, String key, File file, boolean outOssFlag) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("Oss key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outOssFlag) {
            outOssClient.putObject(bucketName, key, file);
        } else {
            innerOssClient.putObject(bucketName, key, file);
        }
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, String filePath) {
        File file = new File(filePath);
        this.uploadFileWithOutFlag(bucketName, key, file, false);
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, File file) {
        this.uploadFileWithOutFlag(bucketName, key, file, false);
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, InputStream inputStream) {
        this.uploadFileWithOutFlag(bucketName, key, inputStream, false);
    }

    /**
     * 上传文件
     */
    public void uploadFile(String bucketName, String key, String filename, ObjectMetadata metadata) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("Oss key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        File file = new File(filename);
        innerOssClient.putObject(bucketName, key, file, metadata);
    }

    /**
     * 下载文件（可选择内外网环境）
     */
    public void downloadFileWithOutFlag(String bucketName, String key, String filePath, boolean outFlag) {
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outFlag) {
            outOssClient.getObject(new GetObjectRequest(bucketName, key), new File(filePath));
        } else {
            innerOssClient.getObject(new GetObjectRequest(bucketName, key), new File(filePath));
        }
    }

    /**
     * 下载文件（可选择内外网环境）
     */
    public void downloadFileWithOutFlag(String bucketName, String key, File file, boolean outFlag) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("Oss key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        if (outFlag) {
            outOssClient.getObject(new GetObjectRequest(bucketName, key), file);
        } else {
            innerOssClient.getObject(new GetObjectRequest(bucketName, key), file);
        }
    }

    /**
     * 下载文件
     */
    public void downloadFile(String bucketName, String key, String filePath) {
        this.downloadFileWithOutFlag(bucketName, key, filePath, false);
    }

    /**
     * 下载文件
     */
    public void downloadFile(String bucketName, String key, File file) {
        this.downloadFileWithOutFlag(bucketName, key, file, false);
    }

    /**
     * 下载文件
     */
    public void downloadFileKeep(String bucketName, String key, File file) {
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }
        this.downloadFileWithOutFlag(bucketName, key, file, false);
    }

    /**
     * 生成下载url（可选择内外网环境）
     */
    public String generateFileUrlWithExpireAndOutFlag(String bucketName, String key, boolean outFlag, Long expireTime) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("Oss key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1);
        }

        Date time = new Date(System.currentTimeMillis() + (Objects.isNull(expireTime) ? urlExpireTime : expireTime));
        if (outFlag) {
            return outOssClient.generatePresignedUrl(bucketName, key, time).toString();
        } else {
            return innerOssClient.generatePresignedUrl(bucketName, key, time).toString();
        }
    }

    /**
     * 生成下载url（可选择内外网环境）
     */
    public String generateFileUrlWithOutFlag(String bucketName, String key, boolean outFlag) {
        return generateFileUrlWithExpireAndOutFlag(bucketName, key, outFlag, null);
    }

    /**
     * 生成下载url
     */
    public String generateFileUrl(String bucketName, String key) {
        return generateFileUrlWithExpireAndOutFlag(bucketName, key, true, null);
    }

    /**
     * 生成下载url
     */
    public String generateFileUrl(String bucketName, String key, Long expireTime) {
        return generateFileUrlWithExpireAndOutFlag(bucketName, key, true, expireTime);
    }

    /**
     * 删除文件
     */
    public void deleteFile(String bucketName, String key) {
        if (StringUtils.isEmpty(key)) {
            throw CommonException.INVALID_PARAM_ERROR.newInstance("Oss key is null");
        }
        if (key.startsWith(BACKSLASH)) {
            key = key.substring(1, key.length());
        }
        innerOssClient.deleteObject(bucketName, key);
    }

    public static void main(String[] args) {
        // 要加密的字符串
//        String before = new String(SecureUtil.generateKey(SymmetricAlgorithm.AES.getValue()).getEncoded(), StandardCharsets.UTF_8);
//        String inputString = Base64.encode(before);
//        System.out.println("password base64: " + inputString);
//
//        System.out.println("password: " + new String(HexUtil.encodeHex(Base64.decode(inputString))));

        // 进行 MD5 加密
        byte[] md5SecretKey = DigestUtil.md5(DigestUtil.md5("406c42688e7f7db4a8f3912772725a5d".getBytes(StandardCharsets.UTF_8)));
        System.out.println("password md5: " + new String(HexUtil.encodeHex(md5SecretKey)));

        AES aes = SecureUtil.aes(md5SecretKey);

        String accessKey = "LTAI5tHpkPpUwR9Pu3xHFUJV";
        String secretKey = "dOj8EqSma1moEQqmKdJb7sb0Pes8OI";

        String encryAccessKey = Base64.encode(aes.encrypt(accessKey));
        String encrySecretKey = Base64.encode(aes.encrypt(secretKey));
        System.out.println("oss accessKey encode: " + new String(HexUtil.encodeHex(Base64.decode(encryAccessKey))));
        System.out.println("oss accessKey encode: " + new String(HexUtil.encodeHex(Base64.decode(encrySecretKey))));

        String decryptAccessKey = new String(aes.decrypt(Base64.decode(encryAccessKey)));
        String decryptSecretKey = new String(aes.decrypt(Base64.decode(encrySecretKey)));


        System.out.println("oss accessKey decode: " + decryptAccessKey);
        System.out.println("oss secretKey decode: " + decryptSecretKey);
    }
}