/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.common.extension.ExtensionLoader;
import com.alibaba.dubbo.common.threadpool.ThreadPool;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcContext;
import com.alibaba.dubbo.rpc.RpcException;
import com.fshows.fsframework.core.enums.RpcTimeConsumingLevelEnum;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.ThreadPoolExecutor;

/**
 * dubbo 全局 慢facade 过滤器
 *
 * @author zhaoxumin
 * @version GlobalSlowFacadeFilter.java, v 0.1 2023-05-31 10:42 zhaoxumin
 */
@Slf4j
@Activate(group = {Constants.PROVIDER, Constants.CONSUMER}, order = -9999)
public class GlobalSlowFacadeFilter implements Filter {

    /**
     * do invoke filter.
     * <p>
     * <code>
     * // before filter
     * Result result = invoker.invoke(invocation);
     * // after filter
     * return result;
     * </code>
     *
     * @param invoker    service
     * @param invocation invocation.
     * @return invoke result.
     * @throws RpcException
     * @see Invoker#invoke(Invocation)
     */
    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        // 获得开始时间
        long rpcStartTime = SystemClock.millisClock().now();
        // 调用接口
        Result result = invoker.invoke(invocation);
        // 获得结束时间
        long rpcEndTime = SystemClock.millisClock().now();

        // rpc调用超过3秒，即为慢facade
        long timeConsuming = rpcEndTime - rpcStartTime;

        // 获得 RPC 方法名
        String methodName = invoker.getUrl().getPath() + "." + RpcContext.getContext().getMethodName();
        Object[] arguments = invocation.getArguments();
        RpcTimeConsumingLevelEnum rpcTimeConsumingLevelEnum = RpcTimeConsumingLevelEnum.getRpcTimeConsumingLevelEnum(timeConsuming);
        switch (rpcTimeConsumingLevelEnum) {
            // 不同耗时等级，可区分处理
            case MAJOR:
            case WARN:
            case ERROR:
                LogUtil.warn(log, "RPC 慢facade接口调用结束，耗时等级:【{}】，methodName = {}, agruments = {}, time = {}ms",
                        rpcTimeConsumingLevelEnum, methodName, arguments, timeConsuming);
                break;

            case COMMON:
            default:
                break;
        }

        ThreadPoolExecutor executor = (ThreadPoolExecutor) ExtensionLoader.getExtensionLoader(ThreadPool.class)
                .getAdaptiveExtension().getExecutor(invoker.getUrl());
        // 获取线程池相关信息
        int activeCount = executor.getActiveCount();
        int corePoolSize = executor.getCorePoolSize();
        int maximumPoolSize = executor.getMaximumPoolSize();
        int poolSize = executor.getPoolSize();
        int queueSize = executor.getQueue().size();
        // 输出线程池相关信息
        LogUtil.info(log, "RPC facade接口调用结束，ThreadPoolStatusFilter = {}, activeCount = {}, corePoolSize = {}, " +
                        "maximumPoolSize = {}, poolSize = {}, queueSize = {}", methodName, activeCount, corePoolSize, maximumPoolSize, poolSize, queueSize);
        return result;
    }
}