/**
 * fshows.com
 * Copyright (C) 2013-2022 All Rights Reserved.
 */
package
        com.fshows.fsframework.extend.idgen.impl;

import cn.hutool.core.util.RandomUtil;
import com.fshows.fsframework.core.utils.FsDateUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import com.fshows.fsframework.extend.idgen.exception.IdGenerateException;
import com.fshows.fsframework.extend.idgen.worker.RandomWorkerIdAssigner;
import com.fshows.fsframework.extend.idgen.worker.WorkerIdAssigner;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 付呗类雪花算法ID生成器
 *
 *      （由于历史因素,当前算法会在十进制位做运算）
 *      ID规则为：{12位时间,格式为yyyyMMddhhmmss}{N位十进制数字}
 *      {N位十进制数字}是在二进制位上将序列号和机器号按照如下顺序拼接{M bit-无用位（固定为0）}{Y bit-序列号}{X bit-机器ID}
 *      再转化为十进制而计算得来，
 *
 *      以交易订单号为例，N是{6位数字}，计算时它用long的低20bit表示，其中{44bit-无用位（固定为0）}{16bit-序列号}{4bit-机器ID}
 *      ,其能表达的最大的十进制数为 (2^4 * 2^16) - 1= 1048575.
 *
 * @author liluqing
 * @version FsSnowflakeIdWorker.java, v 0.1 2022-11-07 16:00
 */
public class FsSnowflakeIdGenerator {

    /**
     * 溢出策略/时间回退策略---抛出异常
     */
    private final static int THROW_EX_TRUE = 0;

    /**
     * 最大的十进制单个数字
     */
    private final static String MAX_DIGIT_STR = "9";

    /**
     * 当前时间戳,格式为yyyyMMddhhmmss
     */
    private volatile String timeStr = null;

    /**
     * 最后一次获取到的时间搓（秒）
     */
    private volatile long lastTimestamp = 0;

    /**
     * 节点ID
     */
    private long workId;

    /**
     * 序列号
     */
    private volatile long seq = 0;

    /**
     * 配置项
     */
    private FsSnowflakeIdConfig config;

    /**
     * 最大的序列号（十进制）
     * (如为空,则通过sequenceBits计算)
     * （如设置，则该值不能大于sequenceBits能表达的十进制数）
     */
    private long maxSequence = -1;

    /**
     * 用于补零的缓存字符串(0~18位),满足long的最大值的补零
     */
    private static String[] ZEROIZE_BUFFER = {"", "0", "00", "000", "0000", "00000",
            "000000", "0000000", "00000000", "000000000", "0000000000",
            "00000000000", "000000000000", "0000000000000", "00000000000000", "000000000000000",
            "0000000000000000", "00000000000000000", "000000000000000000"};

    /**
     * seq临界位
     */
    private volatile long tailSeq = 0;

    /**
     * 初始化
     */
    public FsSnowflakeIdGenerator(WorkerIdAssigner workerIdAssigner, FsSnowflakeIdConfig config) {
        // 参数校验
        this.workId = workerIdAssigner.assignWorkerId();
        // 机器ID能表达的最大数值
        long bitMaxWorkerId = (1L << config.getWorkerIdBits())-1;
        if (this.workId > bitMaxWorkerId) {
            throw new IdGenerateException("workerId有误,workerId不能大于"+bitMaxWorkerId);
        }
        if (config.getNumberLength() > 18) {
            throw new IdGenerateException("numberLength不能大于18位");
        }
        this.config = config;
        this.maxSequence = getMaxSequence(config);
        this.tailSeq = this.maxSequence-1;
    }

    /**
     * 计算序列号最大值
     *
     * @param config
     * @return
     */
    private long getMaxSequence(FsSnowflakeIdConfig config) {
        // 放置序列号的bit能表达的最大数值
        long bitMaxSeq = (1L << config.getSequenceBits());
        // 机器ID能表达的最大数值
        long bitMaxWorkerId = (1L << config.getWorkerIdBits());
        // 时间后数值的数字位数
        long numberLength = config.getNumberLength();
        // 最大数字
        long maxNumber = getMaxNumber(numberLength);
        // 序列号的最大值
        long maxSeq = maxNumber/bitMaxWorkerId;
        return Math.min(maxSeq, bitMaxSeq);
    }

    /**
     * 返回改位数下，十进制的最大值
     *
     * @param numberLength
     * @return
     */
    private long getMaxNumber(long numberLength) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < numberLength; i++) {
            sb.append(MAX_DIGIT_STR);
        }
        return Long.parseLong(sb.toString());
    }

    /**
     * 获取ID
     *
     * @return
     */
    public synchronized String next() {
        int currentTime = SystemClock.millisClock().seconds();
        if (lastTimestamp > currentTime) {
            if (config.getTimeFallbackStrategy() == THROW_EX_TRUE) {
                throw new IdGenerateException("发生时间回退错误，请稍后再试！当前时间=" + currentTime + ", 上次生成时间={}" + lastTimestamp);
            }
        }
        if (lastTimestamp < currentTime) {
            // 重置timeStr
            resetTime();
        }
        seq++;
        // seq的取值位一个环形,当seq的值大于最大值时,将seq重置为0
        if (seq >= this.maxSequence) {
            seq = 1;
        }
        if (seq == tailSeq) {
            if (config.getSeqOverflowStrategy() == 0) {
                throw new IdGenerateException("序列号溢出,请稍后再试");
            }
        }
        long unionId = (seq << config.getWorkerIdBits()) | workId;
        StringBuffer sb = new StringBuffer(timeStr);
        sb.append(zeroize(unionId, config.getNumberLength()));
        return sb.toString();
    }

    /**
     * 重置时间
     */
    private void resetTime() {
        this.lastTimestamp = SystemClock.millisClock().seconds();
        this.timeStr = FsDateUtil.getReqDateyyyyMMddHHmmss(new Date(lastTimestamp * 1000L));
        this.seq = RandomUtil.randomLong(1L, maxSequence);
        this.tailSeq = this.seq;
    }

    /**
     * 序列号补零
     *
     * @param seriesNumber
     * @param length
     * @return
     */
    private String zeroize(long seriesNumber, int length){
        int stringSize = stringSize(seriesNumber);
        return new StringBuilder(ZEROIZE_BUFFER[length-stringSize])
                .append(seriesNumber).toString();
    }

    /**
     * 计算数字长度
     *
     * @param x
     * @return
     */
    private int stringSize(long x) {
        long p = 10;
        for (int i=1; i<19; i++) {
            if (x < p)
                return i;
            p = 10*p;
        }
        return 19;
    }

    public static void main(String[] args) throws InterruptedException {
        WorkerIdAssigner workerIdAssigner = new RandomWorkerIdAssigner();
        FsSnowflakeIdConfig config = new FsSnowflakeIdConfig();
        config.setNumberLength(6);
        config.setSequenceBits(16);
        config.setWorkerIdBits(4);
        config.setSeqOverflowStrategy(0);

        AtomicLong eqNum = new AtomicLong();
        int threadNum = 70;

        CountDownLatch countDownLatch2 = new CountDownLatch(threadNum);
        long currentTime = System.currentTimeMillis();
        Set<String> hashSet = Collections.synchronizedSet(new HashSet<String>(300000));

        FsSnowflakeIdGenerator worker = new FsSnowflakeIdGenerator(workerIdAssigner, config);
        System.out.println(worker.next());
        System.out.println(worker.next());
    }
}