/**
 * fshows.com
 * Copyright (C) 2013-2018 All Rights Reserved.
 */
package com.fshows.fsframework.extend.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.Filter;
import com.alibaba.dubbo.rpc.Invocation;
import com.alibaba.dubbo.rpc.Invoker;
import com.alibaba.dubbo.rpc.Result;
import com.alibaba.dubbo.rpc.RpcException;
import com.fshows.fsframework.core.enums.RpcTimeConsumingLevelEnum;
import com.fshows.fsframework.core.utils.LogUtil;
import com.fshows.fsframework.core.utils.SystemClock;
import lombok.extern.slf4j.Slf4j;

/**
 * dubbo 全局 慢facade 过滤器
 *
 * @author zhaoxumin
 * @version GlobalSlowFacadeFilter.java, v 0.1 2023-05-31 10:42 zhaoxumin
 */
@Slf4j
@Activate(group = {Constants.PROVIDER, Constants.CONSUMER}, order = -9999)
public class GlobalSlowFacadeFilter implements Filter {

    /**
     * do invoke filter.
     * <p>
     * <code>
     * // before filter
     * Result result = invoker.invoke(invocation);
     * // after filter
     * return result;
     * </code>
     *
     * @param invoker    service
     * @param invocation invocation.
     * @return invoke result.
     * @throws RpcException
     * @see Invoker#invoke(Invocation)
     */
    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        // 获得开始时间
        long rpcStartTime = SystemClock.millisClock().now();
        // 调用接口
        Result result = invoker.invoke(invocation);
        // 获得结束时间
        long rpcEndTime = SystemClock.millisClock().now();

        // rpc调用超过3秒，即为慢facade
        long timeConsuming = rpcStartTime - rpcEndTime;
        String methodName = invocation.getMethodName();
        RpcTimeConsumingLevelEnum rpcTimeConsumingLevelEnum = RpcTimeConsumingLevelEnum.getRpcTimeConsumingLevelEnum(timeConsuming);
        switch (rpcTimeConsumingLevelEnum){
            // 不同等耗时级，可区分处理
            case MAJOR:
            case WARN:
            case ERROR:
                LogUtil.warn(log, "RPC 慢facade接口调用结束 methodName = {}, time = {}ms, rpcTimeConsumingLevelEnum = {}",
                        methodName, timeConsuming, rpcTimeConsumingLevelEnum);
                break;

            case COMMON:
            default:
                break;
        }
        return result;
    }
}