package com.fshows.util.fnefpay.gm;

import com.fshows.util.fnefpay.sdk.FnefPayConfig;
import com.fshows.util.fnefpay.util.CerFilter;
import com.fshows.util.fnefpay.util.FileUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 国密证书工具类
 */
public class CertUtils_SM2 {
    /**
     * 签名私钥
     */
    private static BigInteger priKey = null;
    /**
     * 签名证书序列号
     */
    private static String signCertId = null;
    /**
     * 验签公钥证书列表
     */
    private static List<String> certFiles = new ArrayList<String>();
    /**
     * 验证返回报文签名的公钥证书存储Map
     */
    private static Map<String, X509Cert_SM2> certMap = new HashMap<String, X509Cert_SM2>();
    /**
     * 外部传入私钥存储Map
     */
    private final static Map<String, Map<String, Object>> priKeyMap = new ConcurrentHashMap<String, Map<String, Object>>();

    static {
        try {
            init();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 初始化所有证书.
     */
    private static void init() throws Exception {
        initSignCert();//初始化签名私钥证书
        initVerifyCert();//初始化验签公钥
    }

    /**
     * 用配置文件配置的私钥路径和密码 加载签名证书
     */
    private static void initSignCert() throws Exception {
        String certPath = FnefPayConfig.getConfig().getGmSignCertPath();
        if (StringUtils.isBlank(certPath)) {
            throw new IllegalArgumentException("私钥证书路径为空");
        }
        String certPwd = FnefPayConfig.getConfig().getGmSignCertPwd();
        if (StringUtils.isBlank(certPwd)) {
            throw new IllegalArgumentException("证书密码为空");
        }
        if (priKey != null) {
            priKey = null;
        }
        byte[] arrayOfByte = FileUtils.getSM2File(certPath);
        PKCS12_SM2 P12 = new PKCS12_SM2(arrayOfByte);
        priKey = P12.getPrivateKey(certPwd);
        X509Cert_SM2 cert = P12.getPublicCert();
        signCertId = cert.getCertId();
    }

    /**
     * 用配置文件配置路径 加载银联公钥验签证书
     */
    private static void initVerifyCert() throws Exception {
        String certDir = FnefPayConfig.getConfig().getGmVerifyCertDir();
        if (StringUtils.isBlank(certDir)) {
            throw new IllegalArgumentException("公钥证书路径为空");
        }
        readCertFile(certDir);
        for (int i = 0; i < certFiles.size(); i++) {
            X509Cert_SM2 certSm2 = new X509Cert_SM2(certFiles.get(i));
            certMap.put(certSm2.getCertId(), certSm2);
        }
    }

    /**
     * 读取证书文件
     *
     * @param filePath 证书路径
     */
    public static void readCertFile(String filePath) {
        File file = new File(filePath);
        if (file.isDirectory()) {
            for (File f : file.listFiles(new CerFilter())) {
                readCertFile(f.getAbsolutePath());
            }
        } else if (file.exists()) {
            certFiles.add(file.getAbsolutePath());
        }
    }

    /**
     * 证书加载
     *
     * @param certFilePath 证书路径
     * @param certPwd      证书密码
     */
    private static void loadSignCert(String certFilePath, String certPwd) throws Exception {
        Map<String, Object> priMap = new HashMap<String, Object>();
        byte[] arrayOfByte = FileUtils.getSM2File(FnefPayConfig.getConfig().getGmSignCertPath());
        PKCS12_SM2 P12 = new PKCS12_SM2(arrayOfByte);
        X509Cert_SM2 cert = P12.getPublicCert();
        priMap.put("priKey", P12.getPrivateKey(FnefPayConfig.getConfig().getGmSignCertPwd()));
        priMap.put("signCertId", cert.getCertId());
        priKeyMap.put(certFilePath, priMap);
    }

    /**
     * @param certPath 证书路径
     * @param certPwd  证书密码
     * @return 私钥
     */
    public static Map<String, Object> getSignCertByStoreMap(String certPath, String certPwd) throws Exception {
        if (!priKeyMap.containsKey(certPath)) {
            loadSignCert(certPath, certPwd);
        }
        return priKeyMap.get(certPath);
    }

    public static BigInteger getPrivateKey() {
        return priKey;
    }

    public static String getSignCertId() {
        return signCertId;
    }

    public static byte[] getPublicKey(String certId) {
        return getVryCert(certId).getPublicKey();
    }

    public static X509Cert_SM2 getVryCert(String certId) {
        return certMap.get(certId);
    }

}
