package com.fshows.util.fjnx;

import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.BCUtil;
import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.SM2;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.signers.PlainDSAEncoding;

/**
 * 使用SM2算法进行数据加解密及签名验签
 */
public class SecurityTools {

    private final SM2 sm2;

    /**
     * @param certPrivateKey 商户自己的私钥
     * @param certPassword   私钥密码
     * @param certPublicKey  收银台的公钥
     */
    public SecurityTools(String certPrivateKey, String certPassword, String certPublicKey) throws Exception {
        String publicKey = Sm2Cert2HexUtil.cert2HexPublicKey(certPublicKey);
        String privateKey = Sm2Cert2HexUtil.cert2HexPrivateKey(certPrivateKey, certPassword);
        sm2 = getSM2(privateKey, publicKey);
    }

    /**
     * 签名
     * 使用SM3对明文数据进行摘要
     * 上送方使用自己的私钥对摘要进行签名
     * 收银台会通过上送方的公钥对报文进行验签
     *
     * @param data 签名源数据
     * @return Hex格式签名
     */
    public String sign(String data) {
        byte[] sign = sm2.sign(data.getBytes());
        return HexUtil.encodeHexStr(sign).toUpperCase();
    }

    /**
     * 验签
     * 收银台通过自身私钥进行签名
     * 上送方通过收银台下发的公钥进行验签
     *
     * @param signature  签名
     * @param sourceData 报文明文数据
     * @return 验签是否正确，true：正确，false：不正确
     */
    public boolean verify(String signature, String sourceData) {
        return sm2.verify(sourceData.getBytes(), HexUtil.decodeHex(signature));
    }

    /**
     * 生成数字信封，对报文加密秘钥（SM4对称秘钥）
     * 上送方使用收银台下发的公钥进行加密生成数字信封
     * 收银台会使用自身的私钥进行解密
     *
     * @param secretKey 对称秘钥
     * @return Hex格式签名
     */
    public String digitalEnvelope(String secretKey) {
        return sm2.encryptHex(secretKey, KeyType.PublicKey).toUpperCase().substring(2);
    }

    /**
     * 解密数字信封得到报文加密秘钥（SM4对称秘钥）
     * 收银台使用上送方的公钥对数据进行加密
     * 上送方通过使用自己的私钥对数据进行解密
     *
     * @param digitalEnvelope 数字信封
     * @return 报文加密秘钥（SM4对称秘钥）
     */
    public String decryptDigitalEnvelope(String digitalEnvelope) {
        return sm2.decryptStr("04".concat(digitalEnvelope), KeyType.PrivateKey).toUpperCase();
    }

    private static SM2 getSM2(String privateKey, String publicKey) {
        ECPrivateKeyParameters privateKeyParameters = BCUtil.toSm2Params(privateKey);
        String xHex = publicKey.substring(0, 64);
        String yHex = publicKey.substring(64, 128);
        ECPublicKeyParameters publicKeyParameters = BCUtil.toSm2Params(xHex, yHex);
        SM2 sm2 = new SM2(privateKeyParameters, publicKeyParameters);
        sm2.usePlainEncoding();
        sm2.setMode(SM2Engine.Mode.C1C2C3);
        return sm2;
    }
}
