/**
 * fshows.com
 * Copyright (C) 2013-2023 All Rights Reserved.
 */
package
        com.fshows;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.HexUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.crypto.digest.SM3;
import cn.hutool.crypto.symmetric.SM4;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fshows.apienum.FjnxApiDefinitionEnum;
import com.fshows.component.FjnxClientConfigModel;
import com.fshows.component.Sm3ApiSignHandler;
import com.fshows.request.FjnxBaseRequest;
import com.fshows.request.FjnxBizRequest;
import com.fshows.request.FjnxHeadRequest;
import com.fshows.response.FjnxBaseResponse;
import com.fshows.response.FjnxBizResponse;
import com.fshows.response.FjnxHeadResponse;
import com.fshows.sdk.core.client.base.AbstractApiClient;
import com.fshows.sdk.core.client.base.definition.IApiDefinition;
import com.fshows.sdk.core.client.base.handler.IApiSignHandler;
import com.fshows.sdk.core.client.base.handler.IHttpRequestHandler;
import com.fshows.sdk.core.client.base.handler.ISerializableHandler;
import com.fshows.sdk.core.client.base.model.ApiRequestModel;
import com.fshows.sdk.core.client.base.model.ApiResponseModel;
import com.fshows.sdk.core.client.base.model.ClientInfoModel;
import com.fshows.sdk.core.client.base.model.DefaultClientConfigModel;
import com.fshows.sdk.core.client.base.model.DefaultRequestContext;
import com.fshows.sdk.core.client.component.http.PostHttpRequestHandler;
import com.fshows.sdk.core.client.component.serializable.JsonSerializableHandler;
import com.fshows.sdk.core.exception.FsApiException;
import com.fshows.sdk.core.util.LogUtil;
import com.fshows.sdk.core.util.ReqIdUtil;
import com.fshows.util.SM2Util;
import com.fshows.util.SMSaltSigner;
import com.fshows.util.StringUtil;
import com.fshows.util.fjnx.SM2Utils;
import com.fshows.util.fjnx.SecurityTools;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;


/**
 * 福建农信客户端
 *
 * @author liluqing
 * @version NestTemplateApiClinet.java, v 0.1 2023-12-12 15:40
 */
@Slf4j
public class FjnxPayApiClinet extends AbstractApiClient<FjnxBizRequest, FjnxBaseResponse, FjnxApiDefinitionEnum> {

     /**
      * 加签验签处理器
      */
     protected IApiSignHandler iApiSignHandler = new Sm3ApiSignHandler();

    /**
     * 加密
     */
    private SMSaltSigner signer = new SMSaltSigner();

    private SecurityTools securityTools;

     /**
      * 序列化处理器
      */
     protected ISerializableHandler paramSerializable = new JsonSerializableHandler();

     /**
      * 请求执行器
      */
     protected IHttpRequestHandler httpRequestHandler = new PostHttpRequestHandler();

    public FjnxPayApiClinet(DefaultClientConfigModel apiClientConfig) throws FsApiException {
        super(apiClientConfig);
        try {
            securityTools = new SecurityTools(apiClientConfig.getFubeiPrivateKey(), apiClientConfig.getPassword(), apiClientConfig.getPayCompanyPublicKey());
        } catch (Exception e) {
            LogUtil.error(log, "FjnxPayApiClinet >> 福建农商银行公私钥加载失败! ", e);
            throw new FsApiException("加载公私钥失败", e);
        }
    }

    @Override
    public FjnxBaseResponse execute(FjnxBizRequest request, FjnxApiDefinitionEnum apiDefinition) throws FsApiException {
        return doExecute(request, apiDefinition);
    }

    @Override
    public FjnxBaseResponse execute(FjnxBizRequest request, FjnxApiDefinitionEnum apiDefinition, DefaultClientConfigModel configModel) throws FsApiException {
        return doExecute(request, apiDefinition, configModel);
    }

    /**
     * 执行请求
     *
     * @param request
     * @param iApiDefinition
     * @param merchantConfigModel
     * @return
     * @throws FsApiException
     */
    @Override
    protected FjnxBaseResponse doExecute(FjnxBizRequest request, FjnxApiDefinitionEnum iApiDefinition, DefaultClientConfigModel merchantConfigModel) throws FsApiException {
        long beginTime = System.currentTimeMillis();
        LogUtil.info(log, "{} >> 执行请求开始 >> iApiDefinition={}, request={}",
                getClientInfo().getClientDesc(), iApiDefinition, request);
        // 构建请求上下文
        DefaultRequestContext requestContext = buildRequestContext(iApiDefinition, request, merchantConfigModel);
        ApiRequestModel apiRequestModel = null;
        ApiResponseModel apiResponseModel = null;
        try {
            // 入参数校验
            checkParam(request, requestContext);
            // 构建请求数据
            apiRequestModel = buildApiRequestModel(request, requestContext);
            // 请求开始时间
            long reqBeginTime = System.currentTimeMillis();
            // 执行post请求
            apiResponseModel = httpRequest(apiRequestModel, requestContext);
            // 请求结束时间
            long reqEndTime = System.currentTimeMillis();
            Object requestBody = ObjectUtils.defaultIfNull(apiRequestModel.getRequestBody(), apiRequestModel.getRequestForm());
            LogUtil.info(log, "{} >> 请求结束[密文] >> url={}, method={}, request={}, response={}, cost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), iApiDefinition, requestBody,
                    apiResponseModel.getResponseBody(), System.currentTimeMillis() - beginTime, reqEndTime-reqBeginTime);

            FjnxBaseResponse response = buildApiResponse(apiResponseModel, apiRequestModel, requestContext);

            LogUtil.info(log, "{} >> 请求结束[明文] >> url={}, method={}, request={}, response={}, totalcost={}ms, reqcost={}ms",
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel.getApiURL(), requestContext.getIApiDefinition(), JSONObject.toJSONString(apiRequestModel.getRequest()),
                    JSONObject.toJSONString(apiResponseModel.getResponse()), System.currentTimeMillis() - beginTime, reqEndTime-reqBeginTime);
            // 反序列化构建响应结果
            return response;
        } catch (FsApiException e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            throw e;
        } catch (Exception e) {
            LogUtil.error(log, "{} >> 请求异常 >> apiDefinition={}, bizRequest={}, apiRequestModel={}, apiResponseModel={}", e,
                    requestContext.getClientInfoModel().getClientDesc() ,
                    iApiDefinition, request, apiRequestModel, apiResponseModel);
            LogUtil.info(log, "{} >> 请求异常[明文] >> url={}, method={}, request={}, cost={}ms", e,
                    requestContext.getClientInfoModel().getClientDesc(),
                    apiRequestModel == null ? "" : apiRequestModel.getApiURL(),
                    requestContext.getIApiDefinition(),
                    JSONObject.toJSONString(apiRequestModel == null ? "" : apiRequestModel.getRequest()),
                    System.currentTimeMillis() - beginTime);
            throw new FsApiException(e.getMessage(), e);
        }
    }

    /**
     * 构建请求上下文
     *
     * @param tradeApiDefinitionEnum
     * @param request
     * @return
     */
    @Override
    protected DefaultRequestContext buildRequestContext(FjnxApiDefinitionEnum tradeApiDefinitionEnum, FjnxBizRequest request, DefaultClientConfigModel merchantConfig) {

        DefaultRequestContext context = new DefaultRequestContext();
        context.setRequestId(ReqIdUtil.getId());
        context.setIApiDefinition(tradeApiDefinitionEnum);
        // 复制默认参数
        FjnxClientConfigModel clientConfig = (FjnxClientConfigModel) this.apiClientConfig;
        //
        FjnxClientConfigModel config = newFjnxClientConfigModel(clientConfig);
        if (merchantConfig != null) {
            // merchantId
            config.setAgentId(StringUtils.isEmpty(merchantConfig.getAgentId()) ? clientConfig.getAgentId() : merchantConfig.getAgentId());
            // merchantId
            config.setAppId(StringUtils.isEmpty(merchantConfig.getAppId()) ? clientConfig.getAppId() : merchantConfig.getAppId());
            // 私钥
            config.setFubeiPrivateKey(StringUtils.isEmpty(merchantConfig.getFubeiPrivateKey())? clientConfig.getFubeiPrivateKey(): merchantConfig.getFubeiPrivateKey());
            // 公钥
            config.setPayCompanyPublicKey(StringUtils.isEmpty(merchantConfig.getPayCompanyPublicKey()) ? clientConfig.getPayCompanyPublicKey(): merchantConfig.getPayCompanyPublicKey());
            // md5 密钥
            config.setMd5SignKey(merchantConfig.getMd5SignKey());
            // sm4key
            config.setSm4Key(((FjnxClientConfigModel) merchantConfig).getSm4Key());
        }
        context.setApiClientConfig(config);

        context.setClientInfoModel(getClientInfo());
        return context;
    }

    /**
     * 对象复制
     * @param clientConfigModel
     * @return {@link DefaultClientConfigModel}
     */
    private FjnxClientConfigModel newFjnxClientConfigModel(FjnxClientConfigModel clientConfigModel) {
        FjnxClientConfigModel configModel = new FjnxClientConfigModel();
        configModel.setAppId(clientConfigModel.getAppId());
        configModel.setAgentId(clientConfigModel.getAgentId());
        configModel.setFubeiPrivateKey(clientConfigModel.getFubeiPrivateKey());
        configModel.setPayCompanyPublicKey(clientConfigModel.getPayCompanyPublicKey());
        configModel.setPassword(clientConfigModel.getPassword());
        configModel.setApiParentURL(clientConfigModel.getApiParentURL());
        configModel.setMd5SignKey(clientConfigModel.getMd5SignKey());
        configModel.setSignTypeEnum(clientConfigModel.getSignTypeEnum());
        configModel.setVrifySignResponse(clientConfigModel.isVrifySignResponse());
        configModel.setCheckParam(clientConfigModel.isCheckParam());
        configModel.setConnectionTimeout(clientConfigModel.getConnectionTimeout());
        configModel.setReadTimeout(clientConfigModel.getReadTimeout());
        configModel.setCharset(clientConfigModel.getCharset());
        configModel.setHump(clientConfigModel.isHump());
        configModel.setExtendParam(new HashMap<>(clientConfigModel.getExtendParam()));
        configModel.setSingIgnoreNull(clientConfigModel.isSingIgnoreNull());
        return configModel;
    }


    /**
      * 构建请求参数
      *
      * @param request
      * @param context
      * @return
      */
    @Override
    protected ApiRequestModel buildApiRequestModel(FjnxBizRequest request, DefaultRequestContext context) {
        // sdk客户端配置
        FjnxClientConfigModel configModel = (FjnxClientConfigModel) context.getApiClientConfig();

        ApiRequestModel apiRequestModel = new ApiRequestModel();
        // 设置网关地址
        apiRequestModel.setApiURL(context.getApiClientConfig().getApiParentURL() + context.getIApiDefinition().getApiURI());

        // 请求头信息
        Map<String, String> headerMap = new HashMap<String, String>();
        headerMap.put("Content-Type","application/json;charset=utf-8");
        headerMap.put("Accept", "application/json");
        headerMap.put("Accept-Encoding", "identity");
        apiRequestModel.setHeadMap(headerMap);

        // 数据补偿
        // 设置平台编号
        request.setApplication(configModel.getAgentId());
        Date now = new Date();
        // 请求实际爱你
        request.setTradeDate(DateUtil.format(now, DatePattern.PURE_DATE_PATTERN));
        request.setTradeTime(DateUtil.format(now, DatePattern.PURE_TIME_PATTERN));
        request.setVersion("1.0");
        if (StringUtils.isBlank(request.getTradeNo())) {
            request.setTradeNo(request.getTradeDate() + request.getTradeTime() + RandomUtil.randomNumbers(10));
        }


        // 生成随机密钥
        String randomSecretKey = UUID.randomUUID().toString().replace("-", "").toUpperCase();
        // 数字信封
        String digitalEnvelope = securityTools.digitalEnvelope(randomSecretKey);

        // 将业务参数转化为json
        String plaintext = JSONObject.toJSONString(request);
        // 提取业务请求参数的数据摘要
        String digest = SM3.create().digestHex(plaintext).toUpperCase();
        // 使用摘要数据生成签名
        String signature = securityTools.sign(digest);

        // 请求头信息
        FjnxHeadRequest headRequest = new FjnxHeadRequest();
        headRequest.setMerInstId(configModel.getAppId());
        headRequest.setSysInstId("S000000002");
        // 获取数字信封
        headRequest.setDigitalEnvelope(digitalEnvelope);
        // 加签
        headRequest.setSignature(signature);
        // 默认使用sm2
        headRequest.setSignType("03");
//        headRequest.setApplication(configModel.getAgentId());

        // 业务参数密文（SM4对称加密）
        String ciphertext = new SM4(HexUtil.decodeHex(randomSecretKey)).encryptHex(plaintext).toUpperCase();

        // 设置实际请求的http请求体
        JSONObject requestJson = new JSONObject();
        requestJson.put("head", headRequest);
        requestJson.put("body", ciphertext);
        apiRequestModel.setRequestBody(requestJson.toJSONString());
        apiRequestModel.setRequest(new FjnxBaseRequest(headRequest, request));
        return apiRequestModel;
    }

     /**
      * 处理客户端信息
      *
      * @return
      */
    @Override
    protected ClientInfoModel getClientInfo() {
        ClientInfoModel clientInfoModel = new ClientInfoModel();
        clientInfoModel.setClientName("福建农信");
        clientInfoModel.setClientCode("fjnxpay-sdk");
        return clientInfoModel;
    }

     @Override
     protected FjnxBaseResponse buildApiResponse(ApiResponseModel apiResponseModel, ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) {
         // 1.响应结果反序列化    2. 返回结果验签
         // 客户端配置
         FjnxClientConfigModel clientConfigModel = (FjnxClientConfigModel) requestContext.getApiClientConfig();
         // 响应结果
         JSONObject resJson = JSONObject.parseObject(apiResponseModel.getResponseBody());
         // 业务密文
         String encryptedBody =  (String) resJson.remove("body");
         // 响应结果
         FjnxBaseResponse response = resJson.toJavaObject(FjnxBaseResponse.class);
         if  (StringUtils.isBlank(encryptedBody)) {
             return response;
         }
         // 响应头信息
         FjnxHeadResponse headResponse = response.getHead();
         // 解密数字信封获取密钥
         String secretKey = securityTools.decryptDigitalEnvelope(headResponse.getDigitalEnvelope());
         System.out.println("解密秘钥：" + secretKey);
         // 解密返回报文
         String decryptedBody = new SM4(HexUtil.decodeHex(secretKey)).decryptStr(encryptedBody);
         System.out.println("解密明文: " + decryptedBody);
         // 业务响应结果
         FjnxBizResponse fjnxBizResponse = JSONObject.parseObject(decryptedBody, requestContext.getIApiDefinition().getResponseClass());
         response.setBody(fjnxBizResponse);
         apiResponseModel.setResponse(response);

         // 通过明文 获取摘要
         String digest = SM3.create().digestHex(decryptedBody).toUpperCase();
         // 是否验签
         Boolean verifySign = securityTools.verify(headResponse.getSignature(), digest);
         // 如果需要验签
         if (clientConfigModel.isVrifySignResponse() && !Boolean.TRUE.equals(verifySign)) {
            throw new FsApiException("验签失败");
         }
        return response;
     }

     @Override
    protected ApiResponseModel httpRequest(ApiRequestModel apiRequestModel, DefaultRequestContext requestContext) throws IOException {
        return httpRequestHandler.httpRequest(apiRequestModel, requestContext);
    }
}