package com.fshows.util.fnefpay.gj;


import com.fshows.util.fnefpay.sdk.FnefPayConfig;
import com.fshows.util.fnefpay.sdk.FnefPayConstants;
import com.fshows.util.fnefpay.sdk.FnefPayUtils;
import com.fshows.util.fnefpay.util.CerFilter;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;


/**
 * 国际证书工具类，主要用于对证书的加载和使用
 */
public class CertUtils_RSA {
    /**
     * 证书容器，存储对商户请求报文签名私钥证书
     */
    private static KeyStore keyStore = null;
    /**
     * 验证返回报文签名的公钥证书存储Map
     */
    private static Map<String, X509Certificate> certMap = new HashMap<String, X509Certificate>();
    /**
     * 外部传入私钥存储Map
     */
    private final static Map<String, KeyStore> keyStoreMap = new ConcurrentHashMap<String, KeyStore>();

    static {
        init();
    }

    /**
     * 初始化所有证书
     */
    private static void init() {
        try {
            addProvider(); //向系统添加BC provider
            initSignCert(); //初始化签名私钥证书
            initValidateCertFromDir(); //初始化所有的验签证书
        } catch (Exception e) {
//			LogUtils.getLogger().error("初始化所有证书失败", e);
        }
    }

    /**
     * 添加签名，验签，加密算法提供者
     */
    private static void addProvider() {
        if (Security.getProvider("BC") == null) {
//			LogUtils.getLogger().info("add BC provider");
            Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
        } else {
            Security.removeProvider("BC"); //解决eclipse调试时tomcat自动重新加载时，BC存在不明原因异常的问题。
            Security.addProvider(new org.bouncycastle.jce.provider.BouncyCastleProvider());
//			LogUtils.getLogger().info("re-add BC provider");
        }
        // printSysInfo();
    }

    /**
     * 用配置文件中配置的私钥路径和密码 加载签名证书
     */
    private static void initSignCert() {
        if (!FnefPayConstants.SIGNMETHOD_RSA.equals(FnefPayConfig.getConfig().getSignMethod())) {
//			LogUtils.getLogger().info("非rsa签名方式，不加载签名证书。");
            return;
        }

        if (FnefPayConfig.getConfig().getSignCertPath() == null
                || FnefPayConfig.getConfig().getSignCertPwd() == null) {
//			LogUtils.getLogger().error("WARN: " + FnefPayConfig.SDK_SIGNCERT_PATH + "或" + FnefPayConfig.SDK_SIGNCERT_PWD
//					+ "为空。 停止加载签名证书。");
            return;
        }

        if (null != keyStore) {
            keyStore = null;
        }

        try {
            keyStore = getKeyInfo(FnefPayConfig.getConfig().getSignCertPath(),
                    FnefPayConfig.getConfig().getSignCertPwd());
//			LogUtils.getLogger().info("InitSignCert Successful. CertId=[" + getSignCertId() + "]");
        } catch (IOException e) {
//			LogUtils.getLogger().error("InitSignCert Error", e);
        }
    }

    /**
     * 用配置文件配置路径 加载验证签名证书
     */
    private static void initValidateCertFromDir() {
        if (!FnefPayConstants.SIGNMETHOD_RSA.equals(FnefPayConfig.getConfig().getSignMethod())) {
//			LogUtils.getLogger().info("非rsa签名方式，不加载验签证书。");
            return;
        }

        certMap.clear();
        String dir = FnefPayConfig.getConfig().getValidateCertDir();
//		LogUtils.getLogger().info("加载验证签名证书目录==>" + dir);
        if (FnefPayUtils.isEmpty(dir)) {
//			LogUtils.getLogger().error("WARN: acpsdk.validateCert.dir is empty");
            return;
        }
        CertificateFactory cf = null;
        FileInputStream in = null;
        try {
            cf = CertificateFactory.getInstance("X.509", "BC");
        } catch (NoSuchProviderException e) {
//			LogUtils.getLogger().error("LoadVerifyCert Error: No BC Provider", e);
            return;
        } catch (CertificateException e) {
//			LogUtils.getLogger().error("LoadVerifyCert Error", e);
            return;
        }

        File fileDir = new File(dir);
        File[] files = fileDir.listFiles(new CerFilter());
        for (int i = 0; i < files.length; i++) {
            File file = files[i];
            try {
                in = new FileInputStream(file.getAbsolutePath());
                X509Certificate validateCert = (X509Certificate) cf.generateCertificate(in);
                if (validateCert == null) {
//					LogUtils.getLogger().error("Load verify cert error, " + file.getAbsolutePath() + " has error cert content.");
                    continue;
                }
                certMap.put(validateCert.getSerialNumber().toString(), validateCert);
                // 打印证书加载信息,供测试阶段调试
//				LogUtils.getLogger().info("[" + file.getAbsolutePath() + "][CertId="+ validateCert.getSerialNumber().toString() + "]");
            } catch (CertificateException e) {
//				LogUtils.getLogger().error("LoadVerifyCert Error", e);
            } catch (FileNotFoundException e) {
//				LogUtils.getLogger().error("LoadVerifyCert Error File Not Found", e);
            } finally {
                if (null != in) {
                    try {
                        in.close();
                    } catch (IOException e) {
//						LogUtils.getLogger().error(e.toString());
                    }
                }
            }
        }
//		LogUtils.getLogger().info("LoadVerifyCert Finish");
    }

    /**
     * 通过keyStore 获取私钥签名证书PrivateKey对象
     *
     * @return
     */
    public static PrivateKey getSignCertPrivateKey() {
        try {
            Enumeration<String> aliasenum = keyStore.aliases();
            String keyAlias = null;
            if (aliasenum.hasMoreElements()) {
                keyAlias = aliasenum.nextElement();
            }
            PrivateKey privateKey = (PrivateKey) keyStore.getKey(keyAlias,
                    FnefPayConfig.getConfig().getSignCertPwd().toCharArray());
            return privateKey;
        } catch (KeyStoreException e) {
//			LogUtils.getLogger().error("getSignCertPrivateKey Error", e);
            return null;
        } catch (UnrecoverableKeyException e) {
//			LogUtils.getLogger().error("getSignCertPrivateKey Error", e);
            return null;
        } catch (NoSuchAlgorithmException e) {
//			LogUtils.getLogger().error("getSignCertPrivateKey Error", e);
            return null;
        }
    }

    /**
     * 通过certId获取验签证书Map中对应证书PublicKey
     *
     * @param certId 证书物理序号
     * @return 通过证书编号获取到的公钥
     */
    public static PublicKey getValidatePublicKey(String certId) {
        X509Certificate cf = null;
        if (certMap.containsKey(certId)) {
            // 存在certId对应的证书对象
            cf = certMap.get(certId);
            return cf.getPublicKey();
        } else {
            // 不存在则重新Load证书文件目录
            initValidateCertFromDir();
            if (certMap.containsKey(certId)) {
                // 存在certId对应的证书对象
                cf = certMap.get(certId);
                return cf.getPublicKey();
            } else {
//				LogUtils.getLogger().error("缺少certId=[" + certId + "]对应的验签证书.");
                return null;
            }
        }
    }

    /**
     * 获取配置文件acp_sdk.properties中配置的签名私钥证书certId
     *
     * @return 证书的物理编号
     */
    public static String getSignCertId() {
        try {
            Enumeration<String> aliasenum = keyStore.aliases();
            String keyAlias = null;
            if (aliasenum.hasMoreElements()) {
                keyAlias = aliasenum.nextElement();
            }
            X509Certificate cert = (X509Certificate) keyStore.getCertificate(keyAlias);
            return cert.getSerialNumber().toString();
        } catch (Exception e) {
//			LogUtils.getLogger().error("getSignCertId Error", e);
            return null;
        }
    }

    /**
     * 将签名私钥证书文件读取为证书存储对象
     *
     * @param pfxkeyfile 证书文件名
     * @param keypwd     证书密码
     * @return 证书对象
     * @throws IOException
     */
    private static KeyStore getKeyInfo(String pfxkeyfile, String keypwd) throws IOException {
//		LogUtils.getLogger().info("加载签名证书==>" + pfxkeyfile);
        FileInputStream fis = null;
        try {
            KeyStore ks = KeyStore.getInstance("PKCS12", "BC");
//			LogUtils.getLogger().info("Load RSA CertPath=[" + pfxkeyfile + "],Pwd=["+ keypwd + "]");
            fis = new FileInputStream(pfxkeyfile);
            char[] nPassword = null;
            nPassword = null == keypwd || "".equals(keypwd.trim()) ? null : keypwd.toCharArray();
            if (null != ks) {
                ks.load(fis, nPassword);
            }
            return ks;
        } catch (Exception e) {
//			LogUtils.getLogger().error("getKeyInfo Error", e);
            return null;
        } finally {
            if (null != fis)
                fis.close();
        }
    }

    /**
     * 打jre中印算法提供者列表
     */
    private static void printProviders() {
//		LogUtils.getLogger().info("Providers List:");
        Provider[] providers = Security.getProviders();
        for (int i = 0; i < providers.length; i++) {
//			LogUtils.getLogger().info(i + 1 + "." + providers[i].getName());
        }
    }

    /**
     * 证书加载
     *
     * @param certFilePath 证书路径
     * @param certPwd      证书密码
     */
    private static void loadSignCert(String certFilePath, String certPwd) {
        KeyStore keyStore = null;
        try {
            keyStore = getKeyInfo(certFilePath, certPwd);
            keyStoreMap.put(certFilePath, keyStore);
        } catch (IOException e) {
        }
    }

    /**
     * 获取PrivateKey对象
     *
     * @param certPath 证书路径
     * @param certPwd  证书密码
     * @return
     */
    public static PrivateKey getSignCertPrivateKeyByStoreMap(String certPath, String certPwd) {
        if (!keyStoreMap.containsKey(certPath)) {
            loadSignCert(certPath, certPwd);
        }
        try {
            Enumeration<String> aliasenum = keyStoreMap.get(certPath).aliases();
            String keyAlias = null;
            if (aliasenum.hasMoreElements()) {
                keyAlias = aliasenum.nextElement();
            }
            PrivateKey privateKey = (PrivateKey) keyStoreMap.get(certPath).getKey(keyAlias, certPwd.toCharArray());
            return privateKey;
        } catch (KeyStoreException e) {
            return null;
        } catch (UnrecoverableKeyException e) {
            return null;
        } catch (NoSuchAlgorithmException e) {
            return null;
        }
    }
}
