package com.fshows.com.fbank.openapi.sdk.util;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Enumeration;

/**
 * 从证书文件中获取公私钥单元类
 *
 * @author miwm.sdc
 * @date 2017-12-13
 */
public class CertFileUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(CertFileUtils.class);

    private static final String STORE_TYPE_PKCS12 = "PKCS12";

    private static final String CERT_TYPE_X509 = "X.509";

    /**
     * RSA算法
     */
    public static final String ALGORITHM_RSA = "RSA";

    /**
     * 从cer文件中获取公钥字符串
     *
     * @param cerFileName 公钥证书文件
     */
    public static String getPublicKeyFormCerFile(String cerFileName) {
        Path path = Paths.get(cerFileName);
        try (InputStream inStream = Files.newInputStream(path)) {
            CertificateFactory instance = CertificateFactory.getInstance(CERT_TYPE_X509);
            Certificate certificate = instance.generateCertificate(inStream);
            PublicKey publicKey = certificate.getPublicKey();
            return Base64.encodeBase64String(publicKey.getEncoded());
        } catch (CertificateException | IOException e) {
            LOGGER.error("get public key from cer file error ", e);
        }

        return null;
    }

    /**
     * 从pfx文件中获取私钥字符串
     *
     * @param pfxFileName 证书文件名称
     * @param pfxPassword 证书密钥
     */
    public static String getPrivateKeyFromPfxFile(String pfxFileName, String pfxPassword) {
        try {
            KeyStore keyStore = getKeyStore(pfxFileName, pfxPassword);
            String keyAlias = getKeyAlias(keyStore);
            PrivateKey privateKey = (PrivateKey) keyStore.getKey(keyAlias, pfxPassword.toCharArray());
            return Base64.encodeBase64String(privateKey.getEncoded());
        } catch (Exception e) {
            LOGGER.error("get private key from pfx file error ", e);
        }

        return null;
    }

    public static String getPrivateKeyFromPfxFile(InputStream pfxFileName, String pfxPassword) {
        try {
            KeyStore keyStore = getKeyStore(pfxFileName, pfxPassword);
            String keyAlias = getKeyAlias(keyStore);
            PrivateKey privateKey = (PrivateKey) keyStore.getKey(keyAlias, pfxPassword.toCharArray());
            return Base64.encodeBase64String(privateKey.getEncoded());
        } catch (Exception e) {
            LOGGER.error("get private key from pfx file error ", e);
        }

        return null;
    }

    static KeyStore getKeyStore(InputStream pfxFileName, String pfxPassword) throws Exception {
        try (InputStream inStream = pfxFileName) {
            KeyStore keyStore = KeyStore.getInstance(STORE_TYPE_PKCS12);
            keyStore.load(inStream, pfxPassword.toCharArray());
            return keyStore;
        }
    }

    /**
     * 获取证书库
     *
     * @param pfxFileName 证书文件名
     * @param pfxPassword 证书密码
     */
    static KeyStore getKeyStore(String pfxFileName, String pfxPassword) throws Exception {
        Path path = Paths.get(pfxFileName);
        try (InputStream inStream = Files.newInputStream(path)) {
            KeyStore keyStore = KeyStore.getInstance(STORE_TYPE_PKCS12);
            keyStore.load(inStream, pfxPassword.toCharArray());
            return keyStore;
        }
    }

    /**
     * 获取证书别名
     *
     * @param keyStore 证书库
     */
    static String getKeyAlias(KeyStore keyStore) throws KeyStoreException {
        String keyAlias = null;
        Enumeration<String> aliases = keyStore.aliases();
        if (aliases.hasMoreElements()) {
            keyAlias = aliases.nextElement();
        }
        return keyAlias;
    }

    /**
     * 根据公钥字符串生成公钥对象
     *
     * @param publicKey 公钥字符串
     */
    public static PublicKey toPublicKey(String publicKey) throws Exception {
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM_RSA);
        byte[] keyBytes = Base64.decodeBase64(publicKey);
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
        // 生成公钥对象
        return keyFactory.generatePublic(x509KeySpec);
    }

    /**
     * 根据私钥字符串生成私钥对象
     *
     * @param privateKey 私钥字符串
     */
    public static PrivateKey toPrivateKey(String privateKey) throws Exception {
        byte[] keyBytes = Base64.decodeBase64(privateKey);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM_RSA);
        return keyFactory.generatePrivate(keySpec);
    }

}
