package com.fshows.com.fbank.openapi.sdk.service;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.fshows.com.fbank.openapi.sdk.client.FBankOpenApiClient;
import com.fshows.com.fbank.openapi.sdk.client.OpenParameters;
import com.fshows.com.fbank.openapi.sdk.config.Configuration;
import com.fshows.com.fbank.openapi.sdk.constant.AlgorithmTypeEnum;
import com.fshows.com.fbank.openapi.sdk.constant.ModuleCodeEnum;
import com.fshows.com.fbank.openapi.sdk.constant.RequestConstants;
import com.fshows.com.fbank.openapi.sdk.crypto.CryptoEncryptionService;
import com.fshows.com.fbank.openapi.sdk.crypto.CryptoSignatureService;
import com.fshows.com.fbank.openapi.sdk.util.AESUtils;
import com.fshows.com.fbank.openapi.sdk.util.CryptoServiceUtils;
import com.fshows.com.fbank.openapi.sdk.util.SerialNoUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.TreeMap;

/**
 * @author miwenming
 * @date 2020/5/28 20:53
 */
public abstract class AbstractOpenApiService implements OpenApiService {
    final Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public String execute(FBankOpenApiClient client, OpenParameters context) throws Exception {
        Configuration configuration = client.getConfiguration();
        // 组装请求报文体
        String body = buildBody(configuration, context);

        // 获取请求连接
        HttpURLConnection connection = getHttpURLConnection(configuration, context);

        byte[] response = getResponse(connection, body, configuration.charset());
        String result = verifyAndDecrypt(connection, configuration, context, response);
        connection.disconnect();
        // 验签解密
        return result;
    }

    /**
     * 组装请求报文体
     *
     * @param configuration 配置对象
     * @param context       请求上下文
     */
    private String buildBody(Configuration configuration, OpenParameters context) throws Exception {
        String randomKey = AESUtils.randomKey();
        String encryptType = configuration.encryptType();
        CryptoEncryptionService aesEncryptionService = CryptoServiceUtils.getCryptoAESEncryptionService(encryptType);
        String data = aesEncryptionService.encrypt(context.getParams(), randomKey);
        logger.info("randomKey is {}, encryptType is {}, encrypt result is {}", randomKey, encryptType, data);

        String signType = configuration.signType();
        CryptoEncryptionService rsaEncryptionService = CryptoServiceUtils.getCryptoRSAEncryptionService(signType);
        String randomKeyEncrypt = rsaEncryptionService.encrypt(randomKey, configuration.fbankEncPubKey());
        logger.info("signType is {}, randomKeyEncrypt is {}", signType, randomKeyEncrypt);

        TreeMap<String, Object> requestMap = new TreeMap<>();
        requestMap.put(RequestConstants.RequestBody.KEY_DATA, data);
        requestMap.put(RequestConstants.RequestBody.KEY_RANDOM_KEY_ENCRYPT, randomKeyEncrypt);
        requestMap.put(RequestConstants.RequestBody.KEY_MERCHANT_NO, configuration.merchantNo());
        requestMap.put(RequestConstants.RequestBody.KEY_CHANNEL_NO, configuration.channelNo());
        requestMap.put(RequestConstants.RequestBody.KEY_SDK_VERSION, RequestConstants.SDK.VERSION);
        requestMap.put(RequestConstants.RequestBody.KEY_TIMESTAMP, System.currentTimeMillis());
        requestMap.put(RequestConstants.RequestBody.KEY_SIGN_TYPE, signType);
        requestMap.put(RequestConstants.RequestBody.KEY_ENCRYPT_TYPE, encryptType);

        if (null != configuration.appId() && !configuration.appId().isEmpty()) {
            requestMap.put(RequestConstants.RequestBody.KEY_APP_ID, configuration.appId());
            requestMap.put(RequestConstants.RequestBody.KEY_SITE_ID, configuration.siteId());
        }

        String userId = context.getUserId();
        if (null != userId && !userId.isEmpty()) {
            requestMap.put(RequestConstants.RequestBody.KEY_USER_ID, userId);
        }

        JSONObject requestJO = new JSONObject(requestMap);
        String requestJSONString = requestJO.toJSONString();
        CryptoSignatureService cryptoSignatureService = CryptoServiceUtils.getCryptoSignatureService(signType);
        // 对加密后的输入参数签名
        String signData = cryptoSignatureService.signature(requestJSONString, configuration);
        logger.info("signature result is {}", signData);
        requestJO.put(RequestConstants.RequestBody.KEY_SIGN_DATA, signData);

        return requestJO.toJSONString();
    }

    /**
     * 获取HttpURLConnection连接
     *
     * @param configuration 配置对象
     * @param context 请求上下文
     */
    private HttpURLConnection getHttpURLConnection(Configuration configuration, OpenParameters context)
            throws IOException {
        URL url = new URL(pseudoModuleCode(context, getRemoteAddress(configuration)));
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        // 设置http 的通用请求头
        setHttpHeaders(configuration, context, connection);
        // 需要输出
        connection.setDoOutput(true);
        // 需要输入
        connection.setDoInput(true);
        // 不允许缓存
        connection.setUseCaches(false);
        // 暂时仅支持POST方法
        connection.setRequestMethod(getHttpMethod());
        // 设置连接超时时间
        connection.setConnectTimeout(configuration.connectionTimeout());
        // 设置读取超时时间
        connection.setReadTimeout(configuration.readTimeout());

        return connection;
    }

    /**
     * 获取请求地址
     *
     * @param configuration 配置对象
     */
    protected String getRemoteAddress(Configuration configuration) {
        return configuration.remoteAddress();
    }

    /**
     * 转换请求地址 对于module_code为openapi-function的请求，将api_code的值以?function=XXXXXX形式拼接到请求地址上
     *
     * @param context 请求上下文
     * @param address 请求地址
     */
    private String pseudoModuleCode(OpenParameters context, String address) {
        if (!address.endsWith("/")) {
            address += "/";
        }
        if (ModuleCodeEnum.OPEN_API_FUNCTION.rawValue().equalsIgnoreCase(context.getModuleCode())) {
            address += "?function=" + context.getApiCode();
        }
        return address;
    }

    /**
     * 设置HTTP请求头
     *
     * @param configuration 配置对象
     * @param context 请求上下文
     * @param connection HttpURLConnection对象
     */
    private void setHttpHeaders(Configuration configuration, OpenParameters context, HttpURLConnection connection) {
        connection.setRequestProperty(RequestConstants.RequestHeader.KEY_API_NAME, context.getApiName());
        connection.setRequestProperty(RequestConstants.RequestHeader.KEY_API_CODE, context.getApiCode());
        String moduleCode = context.getModuleCode();
        if (null != moduleCode) {
            connection.setRequestProperty(RequestConstants.RequestHeader.KEY_MODULE_CODE, moduleCode);
        }
        setAlgorithm(configuration, connection, moduleCode);
        connection.setRequestProperty(RequestConstants.RequestHeader.KEY_API_VERSION, context.getApiVersion());
        String clientSerialNo = SerialNoUtils.calculateSerialNo(context.getParams());
        connection.setRequestProperty(RequestConstants.RequestHeader.KEY_CLIENT_SERIAL_NO, clientSerialNo);
        connection.setRequestProperty("Content-Type", "application/json");
    }

    protected void setAlgorithm(Configuration configuration, HttpURLConnection connection, String moduleCode) {
        // 硬件服务器签名服务需要上送algorithm请求为rsaDetached
        if (AlgorithmTypeEnum.RSA_HARDWARE.getAlgorithmType().equals(configuration.signType())) {
            connection.setRequestProperty(RequestConstants.RequestHeader.KEY_ALGORITHM,
                    RequestConstants.AlgorithmService.ALGORITHM_TYPE_RSA_DETACHED);
        }

        if (null != moduleCode) {
            if (ModuleCodeEnum.OPEN_API_BASE_SERVICE.rawValue().equalsIgnoreCase(moduleCode)) {
                connection.setRequestProperty(RequestConstants.RequestHeader.KEY_ALGORITHM,
                        RequestConstants.AlgorithmService.ALGORITHM_TYPE_RSA);
            }
        }
    }

    /**
     * 获取HTTP请求方法，默认为POST
     */
    protected String getHttpMethod() {
        return "POST";
    }

    /**
     * 解析响应结果，关闭输入输出流
     *
     * @param connection HttpURLConnection实例
     * @param encryptedBody 加密报文
     * @param charset 所使用字符集
     */
    private byte[] getResponse(HttpURLConnection connection, String encryptedBody, Charset charset)
            throws IOException {
        try (OutputStream outputStream = connection.getOutputStream()) {
            if (null != encryptedBody) {
                // 发送请求数据
                outputStream.write(encryptedBody.getBytes(charset));
                outputStream.flush();
            }
        }
        try (InputStream in = connection.getInputStream()) {
            // 读取数据到字节数组
            return read(in);
        }
    }

    /**
     * 读取数据到字节数组
     *
     * @param inputStream 输入流
     */
    private byte[] read(InputStream inputStream) throws IOException {
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            final int defaultBufferSize = 1024 * 8;
            byte[] buffer = new byte[defaultBufferSize];
            int num;

            logger.info("read data begin");
            while ((num = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, num);
            }
            byte[] readResult = outputStream.toByteArray();
            logger.info("read data end");

            return readResult;
        }
    }

    /**
     * 验签解密
     *
     * @param connection HttpURLConnection连接实例
     * @param configuration 配置对象
     * @param context 上下文
     * @param response 响应报文
     */
    public String verifyAndDecrypt(HttpURLConnection connection, Configuration configuration,
                                   OpenParameters context, byte[] response)
            throws Exception {
        String responseStr = new String(response, configuration.charset());
        JSONObject responseJO = JSONObject.parseObject(responseStr);
        if (null == responseJO || !responseJO.containsKey(RequestConstants.RequestBody.KEY_SIGN_DATA)) {
            return responseStr;
        }
        // 使用双证书版本的公钥进行加密
        String pubKey = configuration.fbankSignPubKey();
        // 签名验证，失败抛出异常
        JSONObject json = verifySignature(configuration, responseStr, pubKey);
        // 解密返回结果
        return decryptResult(configuration, json);
    }

    /**
     * 对响应结果进行签名验证
     *
     * @param encryptedResult 加密签名后的数据
     * @param publicKey 平台公钥
     * @return 根据响应结果解析的json对象
     */
    private JSONObject verifySignature(Configuration configuration, String encryptedResult, String publicKey)
            throws Exception {
        Map<String, Object> encryptedResultMap
                = JSONObject.parseObject(encryptedResult, new TypeReference<TreeMap<String, Object>>() {
        });
        Object signData = encryptedResultMap.remove(RequestConstants.RequestBody.KEY_SIGN_DATA);
        JSONObject encryptedResultJson = new JSONObject(encryptedResultMap);
        CryptoSignatureService cryptoService = CryptoServiceUtils.getCryptoSignatureService(configuration.signType());

        boolean verifyResult = cryptoService.verifySignature(
                encryptedResultJson.toJSONString(), signData.toString(), publicKey);
        if (!verifyResult) {
            logger.error("The result signature verify failed, encryptedResult is : {} ", encryptedResult);
            throw new IllegalArgumentException("Verify signature failed.");
        }
        return encryptedResultJson;
    }

    /**
     * 解密AES密钥，然后用AES密钥对返回结果进行解密
     *
     * @param configuration 配置类
     * @param json 根据返回结果得到的json对象
     * @return 解密后的业务数据
     */
    private String decryptResult(Configuration configuration, JSONObject json) throws Exception {
        String encryptedRandomKey = json.getString(RequestConstants.RequestBody.KEY_RANDOM_KEY_ENCRYPT);
        CryptoEncryptionService rsaCryptoEncryptionService
                = CryptoServiceUtils.getCryptoRSAEncryptionService(configuration.signType());
        String randomKey = rsaCryptoEncryptionService.decrypt(encryptedRandomKey, configuration.privateKey());

        CryptoEncryptionService aesCryptoEncryptionService
                = CryptoServiceUtils.getCryptoAESEncryptionService(configuration.encryptType());
        return aesCryptoEncryptionService.decrypt(json.getString(RequestConstants.RequestBody.KEY_DATA), randomKey);
    }
}
