package com.fshows.com.fbank.openapi.sdk.util;

import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.Security;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;

/**
 * AES加解密单元类
 *
 * @author tangjunmao
 * @date 2018/10/22
 */
public class AESUtils {

    /**
     * 加密方式
     */
    private static final String ALGORITHM_AES = "AES";
    /**
     * 安全随机签名算法
     */
    private static final String SECURE_ALGORITHM_SHA1PRNG = "SHA1PRNG";

    /**
     * 加密算法
     */
    private static final String ENCRYPT_ALGORITHM_ECB = "AES/ECB/PKCS5Padding";

    /**
     * PBKDF2WithHmacSHA1安全随机签名算法
     */
    private static final String SECURE_ALGORITHM_PBKDF2 = "PBKDF2WithHmacSHA1";

    /**
     * 加密算法CBC
     */
    private static final String ENCRYPT_ALGORITHM_CBC = "AES/CBC/PKCS5Padding";

    private static final int KEY_LENGTH_128 = 128;

    private static final int KEY_LENGTH_16 = 16;

    /**
     * 该类频繁创建会导致内存泄漏
     */
    private static volatile BouncyCastleProvider bouncyCastleProvider = new BouncyCastleProvider();

    /**
     * 生成随机密钥，一次一密
     */
    public static String randomKey() {
        long randomKey = ThreadLocalRandom.current().nextLong();
        return String.valueOf(randomKey);
    }

    /**
     * 报文加密AES/ECB/PKCS5Padding
     *
     * @param data 待加密报文
     * @param key 加密密钥
     */
    public static String encryptWithECB(String data, String key) throws Exception {
        byte[] aesData = doCipher(data.getBytes(StandardCharsets.UTF_8), key, Cipher.ENCRYPT_MODE);
        return Base64.encodeBase64String(aesData);
    }

    /**
     * 报文解密AES/ECB/PKCS5Padding
     *
     * @param data 待解密报文
     * @param key 解密密钥
     */
    public static String decryptWithECB(String data, String key) throws Exception {
        byte[] plain = doCipher(Base64.decodeBase64(data), key, Cipher.DECRYPT_MODE);
        return new String(plain, StandardCharsets.UTF_8);
    }

    /**
     * 加解密公共方法AES/ECB/PKCS5Padding
     *
     * @param dataBytes 待加密/解密的报文字节
     * @param key 加密/解密密钥
     * @param mode 模式 1-加密 2-解密
     */
    private static byte[] doCipher(byte[] dataBytes, String key, int mode) throws Exception {
        SecureRandom secureRandom = SecureRandom.getInstance(SECURE_ALGORITHM_SHA1PRNG);
        secureRandom.setSeed(key.getBytes(StandardCharsets.UTF_8));
        KeyGenerator keyGenerator = KeyGenerator.getInstance(ALGORITHM_AES);
        keyGenerator.init(KEY_LENGTH_128, secureRandom);
        SecretKey secretKey = keyGenerator.generateKey();
        SecretKeySpec keySpec = new SecretKeySpec(secretKey.getEncoded(), ALGORITHM_AES);
        Security.addProvider(bouncyCastleProvider);
        Cipher cipher = Cipher.getInstance(ENCRYPT_ALGORITHM_ECB);
        cipher.init(mode, keySpec);
        return cipher.doFinal(dataBytes);
    }

    /**
     * 报文加密AES/CBC/PKCS5Padding
     * aes-cbc模式加密在加密和解密是需要一个初始化向量(Initialization Vector, IV)，
     * 在每次加密之前或者解密之后，使用初始化向量与明文或密文异或
     *
     * @param data 待加密报文
     * @param key 加密密钥
     */
    public static String encryptWithCBC(String data, String key) throws Exception {
        SecretKeySpec secretKeySpec = secretKeyInitWithPBKDF2(key);
        Security.addProvider(bouncyCastleProvider);
        Cipher cipher = Cipher.getInstance(ENCRYPT_ALGORITHM_CBC);
        cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, new IvParameterSpec(getIV(key)));
        byte[] encryptedDataBytes = cipher.doFinal(data.getBytes(StandardCharsets.UTF_8));
        return Base64.encodeBase64String(encryptedDataBytes);
    }

    /**
     * 报文解密AES/CBC/PKCS5Padding
     *
     * @param data 待解密报文
     * @param key 解密密钥
     */
    public static String decryptWithCBC(String data, String key) throws Exception {
        byte[] dataBytes = Base64.decodeBase64(data);
        Security.addProvider(bouncyCastleProvider);
        Cipher aesCipher = Cipher.getInstance(ENCRYPT_ALGORITHM_CBC);
        ;
        SecretKeySpec keySpec = secretKeyInitWithPBKDF2(key);
        aesCipher.init(Cipher.DECRYPT_MODE, keySpec, new IvParameterSpec(getIV(key)));
        byte[] decryptedDataBytes = aesCipher.doFinal(dataBytes);
        return new String(decryptedDataBytes, StandardCharsets.UTF_8);
    }

    /**
     * 构造密钥
     *
     * @param key 加解密密钥
     * @return SecretKeySpec 密钥对象
     */
    private static SecretKeySpec secretKeyInitWithPBKDF2(String key)
            throws NoSuchAlgorithmException, InvalidKeySpecException {
        int iterations = 1000;
        char[] chars = key.toCharArray();
        byte[] salt = String.valueOf(key.hashCode()).getBytes(StandardCharsets.UTF_8);
        PBEKeySpec spec = new PBEKeySpec(chars, salt, iterations, KEY_LENGTH_128);
        SecretKeyFactory skf = SecretKeyFactory.getInstance(SECURE_ALGORITHM_PBKDF2);
        return new SecretKeySpec(skf.generateSecret(spec).getEncoded(), ALGORITHM_AES);
    }

    /**
     * 密钥长度不足16位时补足
     *
     * @param key 密钥
     */
    private static byte[] getIV(String key) {
        int keyLength = key.length();
        StringBuilder keyBuilder = new StringBuilder(key);
        while (keyLength < KEY_LENGTH_16) {
            //右补0
            keyBuilder.append("0");
            keyLength = keyBuilder.length();
        }
        key = keyBuilder.toString();
        return Arrays.copyOf(key.getBytes(), KEY_LENGTH_16);
    }

    public static void main(String[] args) throws Exception {
        String data = "testInfo";
        String aesKey = "1234567891234567";
        System.out.println(encryptWithECB(data, aesKey));
    }

}
